<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use MailerPressPro\Core\EmailManager\BounceDetectionInterface;
use MailerPressPro\Core\EmailManager\BounceDetectionTrait;
use WP_Error;

class BrevoService extends AbstractEmailService implements BounceDetectionInterface
{
    use BounceDetectionTrait;
    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): bool|WP_Error
    {
        $response = wp_remote_post('https://api.brevo.com/v3/smtp/email', [
            'headers' => [
                'Content-Type' => 'application/json',
                'api-key' => $emailData['apiKey'],
            ],
            'body' => wp_json_encode([
                'sender' => [
                    'name' => $emailData['sender_name'],
                    'email' => $emailData['sender_to'],
                ],
                'to' => [['email' => $emailData['to']]],
                'subject' => $emailData['subject'],
                'htmlContent' => $emailData['body'],
            ]),
        ]);

        // Log the email attempt with detailed error messages
        if (is_wp_error($response)) {
            $errorMessage = sprintf(
                __('Brevo API request failed: %s', 'mailerpress-pro'),
                $response->get_error_message()
            );
            $this->logEmail($emailData, 'brevo', $response, $errorMessage);

            // For test emails, return WP_Error to get explicit error message
            if (!empty($emailData['isTest'])) {
                return new WP_Error('brevo_request_failed', $errorMessage);
            }
            return false;
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $responseBody = wp_remote_retrieve_body($response);

        if (201 !== $statusCode) {
            $errorMessage = $this->getErrorMessageFromResponse($statusCode, $responseBody);
            $this->logEmail($emailData, 'brevo', false, $errorMessage);

            // For test emails, return WP_Error to get explicit error message
            if (!empty($emailData['isTest'])) {
                return new WP_Error('brevo_send_error', $errorMessage);
            }
            return false;
        }

        // Log success
        $this->logEmail($emailData, 'brevo', true);
        return true;
    }

    /**
     * Extracts and formats an explicit error message from the Brevo API response
     * 
     * @param int $code HTTP response code
     * @param string $resBody Response body
     * @return string Formatted and explicit error message
     */
    private function getErrorMessageFromResponse(int $code, string $resBody): string
    {
        // Explicit error messages for common HTTP codes
        $httpErrorMessages = [
            400 => __('Brevo request error (400): The email parameters are invalid. Please check the email address format, subject, and content.', 'mailerpress-pro'),
            401 => __('Brevo authentication error (401): The API key is invalid or missing. Please check your API key in the Brevo settings.', 'mailerpress-pro'),
            403 => __('Brevo access denied (403): Your API key does not have the necessary permissions or your account has been suspended.', 'mailerpress-pro'),
            404 => __('Brevo resource not found (404): The domain or requested resource is not found. Please check your domain configuration.', 'mailerpress-pro'),
            422 => __('Brevo validation error (422): The email data is not valid. Please check the email addresses, content format, and attachments.', 'mailerpress-pro'),
            429 => __('Brevo rate limit exceeded (429): Too many requests have been sent. Please wait before trying again.', 'mailerpress-pro'),
            500 => __('Brevo server error (500): An internal error occurred on Brevo\'s side. Please try again later.', 'mailerpress-pro'),
        ];

        // Base message with HTTP code
        $baseMessage = $httpErrorMessages[$code] ?? sprintf(
            __('Brevo API error (HTTP %d)', 'mailerpress-pro'),
            $code
        );

        // Try to extract additional details from the response body
        if (!empty($resBody)) {
            $errorData = json_decode($resBody, true);

            if (json_last_error() === JSON_ERROR_NONE && is_array($errorData)) {
                $details = [];

                // Brevo returns errors in 'message' or 'error'
                if (isset($errorData['message'])) {
                    $details[] = $errorData['message'];
                }

                if (isset($errorData['error'])) {
                    $errorText = is_string($errorData['error'])
                        ? $errorData['error']
                        : json_encode($errorData['error']);
                    $details[] = $errorText;
                }

                // Build the final message with details
                if (!empty($details)) {
                    $detailsText = implode(' ', array_unique($details));
                    return sprintf(
                        __('%s Details: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $detailsText
                    );
                }
            } else {
                // If the body is not valid JSON, add it as is
                $resBodyTrimmed = trim($resBody);
                if (!empty($resBodyTrimmed) && strlen($resBodyTrimmed) < 500) {
                    return sprintf(
                        __('%s API response: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $resBodyTrimmed
                    );
                }
            }
        }

        return $baseMessage;
    }

    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'brevo',
            'name' => 'Brevo',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="44" height="44" fill="none" viewBox="4 3 92 92"><g filter="url(#filter0_d_4669_137863)"><rect width="92" height="92" x="4" y="3" fill="#0B996E" rx="46"></rect><path fill="#fff" d="M37.694 49.02V28.3H50.14c4.203 0 6.98 2.462 6.98 6.198 0 4.244-3.615 7.47-11.015 9.933-5.046 1.611-7.315 2.97-8.157 4.586l-.254.003Zm0 20.802v-8.66c0-3.82 3.196-7.556 7.653-9 3.954-1.36 7.231-2.718 10.008-4.16 3.7 2.21 5.97 6.027 5.97 10.019 0 6.791-6.393 11.8-15.054 11.8h-8.577Zm-7.569 7.3h16.819c12.784 0 22.368-8.065 22.368-18.763 0-5.86-2.942-11.121-8.157-14.519 2.692-2.718 3.954-5.86 3.954-9.68 0-7.895-5.635-13.16-14.127-13.16H30.125v56.122Z"></path></g><defs><filter id="filter0_d_4669_137863" width="100" height="100" x="0" y="0" color-interpolation-filters="sRGB" filterUnits="userSpaceOnUse"><feFlood flood-opacity="0" result="BackgroundImageFix"></feFlood><feColorMatrix in="SourceAlpha" result="hardAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"></feColorMatrix><feOffset dy="1"></feOffset><feGaussianBlur stdDeviation="2"></feGaussianBlur><feColorMatrix values="0 0 0 0 0.0687866 0 0 0 0 0.097585 0 0 0 0 0.37981 0 0 0 0.0779552 0"></feColorMatrix><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_4669_137863"></feBlend><feBlend in="SourceGraphic" in2="effect1_dropShadow_4669_137863" result="shape"></feBlend></filter></defs></svg>',
            'link' => 'https://www.brevo.com/pricing/',
            'createAccountLink' => 'https://onboarding.brevo.com/account/register',
            'linkApiKey' => 'https://app.brevo.com/settings/keys/api',
            'description' => __(
                'Confidently send transactional emails with Brevo, formerly Sendinblue. With an impressive free offer, Brevo allows you to send up to 300 transactional emails per day! And for those who need to send more, just pay for what you send. For more information on getting started with Brevo, check out our documentation.',
                'mailerpress-pro'
            ),
            'recommended' => true,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ],
            'bounce_doc_url' => 'https://mailerpress.com/docs/bounce-tracking/brevo',
        ];
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_offset', $this->config()['key']);
        $limit = 100;
        $offset = 0;

        $url = sprintf(
            "https://api.brevo.com/v3/contacts?limit=%d&offset=%d",
            $limit,
            $offset
        );

        $response = wp_remote_get($url, [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['contacts'])) {
            foreach ($data['contacts'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'],
                    'email' => $contact['email'],
                    'remote_lists' => $contact['listIds'],
                    'attributes' => [
                        'first_name' => $contact['attributes']['PRENOM'],
                        'last_name' => $contact['attributes']['NOM'],
                    ],
                ]);
            }

            // Update the offset
            $newOffset = $offset + count($data['contacts']);
            update_option($optionName, $newOffset);

            if (count($data['contacts']) === $limit) {
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            }
        }
    }

    public function syncLists(string $apiKey): void
    {
        $optionName = sprintf('%s_lists_last_offset', $this->config()['key']);
        $limit = 20;
        $offset = (int)get_option($optionName, 0);
        $total = null;

        do {
            $url = sprintf(
                "https://api.brevo.com/v3/contacts/lists?limit=%d&offset=%d",
                $limit,
                $offset
            );

            $response = wp_remote_get($url, [
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                ]
            ]);

            if (is_wp_error($response)) {
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if ($total === null && isset($data['count'])) {
                $total = (int)$data['count'];
            }

            $lists = $data['lists'] ?? [];
            if (!empty($lists)) {
                foreach ($lists as $list) {
                    // Do something with the list
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }

                // Update the stored offset after each batch
                $offset += $limit;
                update_option($optionName, $offset);
            }
        } while ($offset < $total);

        delete_option($optionName);
    }

    public function syncTags(string $apiKey): void {}

    public function pushContacts(string $apiKey, array $data): mixed
    {

        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];

        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            // Insert the contact
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                 FROM $list_sync_table
                 WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)

                    )
                );
            }

            $url = 'https://api.brevo.com/v3/contacts';

            $body = json_encode([
                'email' => $contact->email,
                'attributes' => [
                    'FIRSTNAME' => $contact->first_name,
                    'LASTNAME' => $contact->last_name,
                ],
                'listIds' => array_map('intval', $remoteLists), // Convert list IDs to integers
                'updateEnabled' => true, // To update the contact if already exists
            ]);

            $response = wp_remote_post($url, [
                'method' => 'POST',
                'headers' => [
                    'api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60, // Set a timeout for the request
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR', // You can set this based on your error
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            // Successfully added/updated the contact in Brevo
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['code']) && $data['code'] != 200) {
                // Handle API error response if any
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $data['code'], // You can set this based on your error
                    'error_message' => $data['message'],
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data['id'])) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $data['id'],
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }


            return $data;
        }

        return null;
    }

    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS); // Adjust to your actual constant
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $folderId = $this->getFolder($apiKey, 'MailerPress');

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            $response = wp_remote_post('https://api.brevo.com/v3/contacts/lists', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'api-key' => $apiKey,
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                    'folderId' => (int)$folderId,
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error('brevo_request_failed', 'Request to Brevo failed.', $response->get_error_message());
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('brevo_api_error', 'Brevo API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");


        $remoteListId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_list_id FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d",
                $listId,
                $active_provider_id
            )
        );

        if ($remoteListId) {
            wp_remote_request("https://api.brevo.com/v3/contacts/lists/{$remoteListId}", [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);
        }
    }

    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    private function getFolder(string $apiKey, string $folder_name)
    {
        //        $option_key = 'mailerpress_brevo_folder_id';
        //        $folder_id = get_option($option_key);
        //
        //        if ($folder_id) {
        //            return $folder_id;
        //        }

        // Try to find the folder via Brevo API
        $response = wp_remote_get('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Failed to fetch folders.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200 || empty($body['folders'])) {
            return new WP_Error('brevo_api_error', 'Error fetching folders list.', $body);
        }


        // Check if folder already exists
        foreach ($body['folders'] as $folder) {
            if (strcasecmp($folder['name'], $folder_name) === 0) {
                //                update_option($option_key, $folder['id']);
                return $folder['id'];
            }
        }

        // Folder doesn't exist, so create it
        $response = wp_remote_post('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'body' => wp_json_encode([
                'name' => $folder_name,
            ]),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Folder creation failed.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 201) {
            return new WP_Error('brevo_api_error', 'Brevo API error.', $body);
        }

        //        update_option($option_key, $body['id']);

        return $body['id'];
    }


    public function deleteAllLists(string $apiKey): void
    {
        $folder_id = get_option('mailerpress_brevo_folder_id');

        if (!$folder_id) {
            return;
        }

        $response = wp_remote_request("https://api.brevo.com/v3/contacts/folders/{$folder_id}", [
            'method' => 'DELETE',
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204) {
            // Folder deleted successfully
            delete_option('mailerpress_brevo_folder_id');
        } else {
            $body = wp_remote_retrieve_body($response);
        }
    }

    public function deleteContact(string $apiKey, array $data): void
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return;
        }

        $contact = $data['contact'];

        if (empty($contact->email)) {
            return;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);

        $active_provider_id = $wpdb->get_var(
            "SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1"
        );

        $remoteContactId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_contact_id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d",
                $contact->contact_id,
                $active_provider_id
            )
        );

        if ($remoteContactId) {
            $response = wp_remote_request("https://api.brevo.com/v3/contacts/" . urlencode($contact->email), [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code === 204) {
                // Delete local sync entry after successful API deletion
                $wpdb->delete(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => $active_provider_id,
                    ],
                    ['%d', '%d']
                );
            } else {
                $response_body = wp_remote_retrieve_body($response);
                $error = json_decode($response_body, true);
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $error['code'] ?? 'UNKNOWN',
                    'error_message' => $error['message'] ?? 'Unknown error',
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
            }
        }
    }

    // ============================================
    // Bounce Detection Implementation
    // ============================================

    /**
     * Brevo supporte les webhooks pour les événements d'email
     */
    public function supportsWebhookBounceDetection(): bool
    {
        return true;
    }

    /**
     * Brevo ne supporte plus l'API pour récupérer les événements
     * Seuls les webhooks sont utilisés pour la détection de bounces
     */
    public function supportsApiBounceDetection(): bool
    {
        return false;
    }

    /**
     * Configure un webhook Brevo pour recevoir les notifications de bounce
     * 
     * Documentation Brevo: https://developers.brevo.com/docs/how-to-use-webhooks
     */
    public function configureBounceWebhook(string $apiKey, string $webhookUrl): array|false
    {
        // Vérifier si un webhook existe déjà
        $existingWebhooks = $this->getExistingWebhooks($apiKey);

        // Chercher un webhook existant avec la même URL
        foreach ($existingWebhooks as $webhook) {
            if (isset($webhook['url']) && $webhook['url'] === $webhookUrl) {
                update_option('mailerpress_brevo_bounce_webhook_id', $webhook['id']);
                return [
                    'id' => $webhook['id'],
                    'url' => $webhook['url'],
                    'description' => $webhook['description'] ?? 'MailerPress Bounce Detection',
                ];
            }
        }

        // Créer un nouveau webhook
        $webhookData = [
            'url' => $webhookUrl,
            'description' => 'MailerPress Bounce Detection',
            'events' => [
                'hardBounce',
                'softBounce',
                'bounce',
                'invalid',
                'blocked',
            ],
        ];

        $response = wp_remote_post('https://api.brevo.com/v3/webhooks', [
            'headers' => [
                'api-key' => $apiKey,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => wp_json_encode($webhookData),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code !== 201) {
            $errorData = json_decode($body, true);
            $errorMessage = $errorData['message'] ?? $errorData['error'] ?? 'Unknown error';
            return false;
        }

        $result = json_decode($body, true);

        if ($result && isset($result['id'])) {
            update_option('mailerpress_brevo_bounce_webhook_id', $result['id']);
            return [
                'id' => $result['id'],
                'url' => $result['url'],
                'description' => $result['description'] ?? 'MailerPress Bounce Detection',
            ];
        }

        return false;
    }

    /**
     * Supprime le webhook de bounce configuré
     */
    public function removeBounceWebhook(string $apiKey, ?string $webhookId = null): bool
    {
        $idToDelete = $webhookId ?? get_option('mailerpress_brevo_bounce_webhook_id');

        if (empty($idToDelete)) {
            return false;
        }

        $response = wp_remote_request("https://api.brevo.com/v3/webhooks/{$idToDelete}", [
            'method' => 'DELETE',
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204 || $code === 200) {
            delete_option('mailerpress_brevo_bounce_webhook_id');
            return true;
        }

        return false;
    }

    /**
     * Récupère les bounces depuis l'API Brevo
     * 
     * Documentation: https://developers.brevo.com/docs/get-email-event-reports
     */
    public function fetchBouncesFromApi(string $apiKey, array $options = []): array
    {
        $startDate = $options['start_date'] ?? date('Y-m-d', strtotime('-7 days'));
        $endDate = $options['end_date'] ?? date('Y-m-d');
        $limit = $options['limit'] ?? 50;
        $offset = $options['offset'] ?? 0;

        $url = sprintf(
            'https://api.brevo.com/v3/smtp/statistics/events?startDate=%s&endDate=%s&limit=%d&offset=%d&event=bounce,hardBounce,softBounce,invalid,blocked',
            urlencode($startDate),
            urlencode($endDate),
            $limit,
            $offset
        );

        $response = wp_remote_get($url, [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return [];
        }

        $result = json_decode(wp_remote_retrieve_body($response), true);

        if (!$result || !isset($result['events'])) {
            return [];
        }

        $bounces = [];
        foreach ($result['events'] as $event) {
            if (in_array($event['event'], ['bounce', 'hardBounce', 'softBounce', 'invalid', 'blocked'], true)) {
                $bounces[] = [
                    'email' => $event['email'] ?? null,
                    'event' => $event['event'],
                    'reason' => $event['reason'] ?? $event['event'],
                    'date' => $event['date'] ?? null,
                    'message_id' => $event['messageId'] ?? null,
                ];
            }
        }

        return $bounces;
    }

    /**
     * Traite un webhook de bounce reçu depuis Brevo
     * 
     * Format du webhook Brevo:
     * {
     *   "event": "bounce",
     *   "email": "example@example.com",
     *   "id": 123,
     *   "date": "2023-01-01T00:00:00Z",
     *   "message-id": "<message-id>",
     *   "reason": "550 5.1.1 User unknown",
     *   "tag": "tag",
     *   "ts_event": 1234567890
     * }
     */
    public function processBounceWebhook(array $payload): bool
    {
        if (!isset($payload['event']) || !isset($payload['email'])) {
            return false;
        }

        $event = $payload['event'];
        $email = $payload['email'];
        $reason = $payload['reason'] ?? $event;

        // Traiter uniquement les événements de bounce et blocked
        // Brevo peut envoyer les événements avec underscore (hard_bounce, soft_bounce) ou camelCase (hardBounce, softBounce)
        // "blocked" est traité comme un bounce car l'email est bloqué et ne peut pas être délivré
        $bounceEvents = ['bounce', 'hardBounce', 'softBounce', 'hard_bounce', 'soft_bounce', 'invalid', 'blocked'];

        if (!in_array($event, $bounceEvents, true)) {
            return false;
        }

        $metadata = [
            'event' => $event,
            'message_id' => $payload['message-id'] ?? null,
            'date' => $payload['date'] ?? null,
            'tag' => $payload['tag'] ?? null,
        ];

        $result = $this->markContactAsBounced($email, $reason, $metadata);

        return $result;
    }

    /**
     * Obtient l'URL du webhook pour Brevo
     */
    public function getBounceWebhookUrl(): ?string
    {
        $webhookId = get_option('mailerpress_brevo_bounce_webhook_id');

        if (empty($webhookId)) {
            return null;
        }

        // Construire l'URL du webhook MailerPress
        $webhookUrl = rest_url('mailerpress/v1/esp/bounce/brevo');

        return $webhookUrl;
    }

    /**
     * Valide la signature d'un webhook Brevo
     * 
     * Brevo utilise une signature HMAC-SHA256 dans l'en-tête X-Brevo-Signature
     */
    public function verifyWebhookSignature(string $payload, string|array $signature, ?string $secret = null): bool
    {
        // Brevo envoie une signature de type string (pas array)
        if (!is_string($signature) || empty($signature)) {
            return false;
        }

        // Récupérer la clé secrète du webhook depuis les options
        $webhookSecret = $secret ?? get_option('mailerpress_brevo_webhook_secret');

        if (empty($webhookSecret)) {
            // Si aucun secret n'est configuré, on accepte la requête
            // (pour la compatibilité avec les configurations existantes)
            return true;
        }

        // Calculer la signature attendue
        $expectedSignature = hash_hmac('sha256', $payload, $webhookSecret);

        // Comparer les signatures de manière sécurisée
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Récupère les webhooks existants depuis Brevo
     * 
     * @param string $apiKey
     * @return array
     */
    private function getExistingWebhooks(string $apiKey): array
    {
        $response = wp_remote_get('https://api.brevo.com/v3/webhooks', [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return [];
        }

        $result = json_decode(wp_remote_retrieve_body($response), true);

        if ($result && isset($result['webhooks'])) {
            return $result['webhooks'];
        }

        return [];
    }
}
