<?php

declare(strict_types=1);

namespace MailerPressPro\Core\Attributes;

\defined('ABSPATH') || exit;

use MailerPressPro\Core\Kernel;

#[\Attribute]
class Endpoint
{
    private string $methods;
    private array|string $route;
    private $permissionCallback;
    private array $args;

    public function __construct(
        array|string $route,
        string $methods = 'GET',
        $permissionCallback = null,
        array $args = []
    ) {
        $this->route = $route;
        $this->methods = $methods;
        $this->permissionCallback = $permissionCallback;
        $this->args = $args;
    }

    public function execute($callable): void
    {
        if (true === Kernel::getContainer()->get('enable_rest')) {
            add_action('rest_api_init', function () use ($callable): void {
                register_rest_route(
                    Kernel::getContainer()->get('rest_namespace'),
                    '/' . $this->route,
                    [
                        'methods' => $this->methods,
                        'callback' => $callable,
                        'permission_callback' => $this->resolvePermissionCallback($callable),
                        'args' => $this->args,
                    ]
                );
            });
        }
    }

    private function resolvePermissionCallback($callable): callable
    {
        if (\is_string($this->permissionCallback)) {
            // Special handling for WordPress global functions
            if ($this->permissionCallback === '__return_true') {
                return function () {
                    return true;
                };
            }

            // Check if it's a WordPress global function
            if (\function_exists($this->permissionCallback)) {
                // Return a closure that calls the global function
                $funcName = $this->permissionCallback;
                return function (...$args) use ($funcName) {
                    return \call_user_func($funcName, ...$args);
                };
            }

            // If permissionCallback is a string, treat it as a method on the same class
            return function (...$args) use ($callable) {
                return \call_user_func([$callable[0], $this->permissionCallback], ...$args);
            };
        } elseif (\is_callable($this->permissionCallback)) {
            // If permissionCallback is already callable, use it directly
            return $this->permissionCallback;
        }

        // Default to allowing all requests
        return function () {
            return true;
        };
    }
}
