<?php

namespace MailerPressPro\Api;

use MailerPress\Api\Permissions;
use MailerPressPro\Core\Attributes\Endpoint;
use MailerPress\Core\Enums\Tables;

class Cpt
{
    #[Endpoint(
        'custom-fields/(?P<field_key>[a-zA-Z0-9_-]+)',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function delete_field(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;

        $field_key = sanitize_key($request['field_key']);
        $definition_table = Tables::get(Tables::MAILERPRESS_CUSTOM_FIELD_DEFINITIONS);
        $entries_table = $wpdb->prefix . 'mailerpress_contact_custom_fields';

        // Delete the custom field definition
        $deleted = $wpdb->delete($definition_table, ['field_key' => $field_key], ['%s']);

        if ($deleted === false) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => __('Failed to delete the custom field.', 'mailerpress-pro')
            ], 500);
        }

        // Delete all entries in contact custom fields using this field_key
        $wpdb->delete($entries_table, ['field_key' => $field_key], ['%s']);

        return new \WP_REST_Response([
            'success' => true,
            'message' => __('Custom field and its contact entries deleted successfully.', 'mailerpress-pro')
        ], 200);
    }

    #[Endpoint(
        'custom-field/all',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function getAllCustomFields(): \WP_REST_Response
    {
        global $wpdb;
        $table = Tables::get(Tables::MAILERPRESS_CUSTOM_FIELD_DEFINITIONS);

        $fields = $wpdb->get_results("SELECT * FROM {$table} ORDER BY field_key ASC", ARRAY_A);

        // Unserialize options for select fields
        foreach ($fields as &$field) {
            if (!empty($field['options'])) {
                $unserialized = maybe_unserialize($field['options']);
                // ensure it's an array of objects with label/value
                if (is_array($unserialized)) {
                    $field['options'] = array_map(fn($opt) => is_array($opt) ? $opt : ['label' => $opt, 'value' => $opt], $unserialized);
                } else {
                    $field['options'] = [];
                }
            } else {
                $field['options'] = [];
            }
        }

        return new \WP_REST_Response($fields, 200);
    }

    #[Endpoint(
        'custom-fields',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function create_field(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;

        $field_key = sanitize_key($request['field_key']);
        $label = sanitize_text_field($request['label']);
        $type = sanitize_text_field($request['type']);
        $options = maybe_serialize($request['options']);
        $required = !empty($request['required']);

        $table = Tables::get(Tables::MAILERPRESS_CUSTOM_FIELD_DEFINITIONS);
        $wpdb->insert($table, [
            'field_key' => $field_key,
            'label' => $label,
            'type' => $type,
            'options' => $options,
            'required' => $required,
        ]);

        $id = $wpdb->insert_id;

        // Return full field object for frontend
        return new \WP_REST_Response([
            'success' => true,
            'field' => [
                'id' => $id,
                'label' => $label,
                'value' => $field_key,   // this is what the select uses
                'field_key' => $field_key,
                'type' => $type,
                'required' => $required,
                'options' => is_array($request['options']) ? $request['options'] : [],
            ],
        ], 200);
    }

    #[Endpoint(
        'custom-fields/(?P<field_key>[a-zA-Z0-9_-]+)',
        methods: 'PUT',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function update_field(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;

        $definition_table = Tables::get(Tables::MAILERPRESS_CUSTOM_FIELD_DEFINITIONS);
        $entries_table = $wpdb->prefix . 'mailerpress_contact_custom_fields';

        $body = $request->get_json_params();
        $field_id = !empty($body['id']) ? intval($body['id']) : null;
        $field_key_param = $request->get_param('field_key');

        // Try to find field by ID first (most reliable), then by field_key
        $existing = null;

        if ($field_id) {
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$definition_table} WHERE id = %d",
                $field_id
            ));
        }

        // If not found by ID, try by field_key parameter
        if (!$existing && $field_key_param) {
            $old_field_key = sanitize_key($field_key_param);
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$definition_table} WHERE field_key = %s",
                $old_field_key
            ));
        }

        if (!$existing) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => __('Custom field not found.', 'mailerpress-pro')
            ], 404);
        }

        $old_field_key = $existing->field_key;
        $new_field_key = sanitize_key($body['field_key'] ?? $old_field_key);
        $label = sanitize_text_field($body['label'] ?? '');
        $type = sanitize_text_field($body['type'] ?? 'text');
        $options = maybe_serialize($body['options'] ?? []);
        $required = !empty($body['required'] ?? false);

        // If field_key changed, update all entries in contact_custom_fields
        if ($old_field_key !== $new_field_key) {
            // Check if new field_key already exists
            $key_exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$definition_table} WHERE field_key = %s",
                $new_field_key
            ));

            if ($key_exists > 0) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => __('A custom field with this key already exists.', 'mailerpress-pro')
                ], 400);
            }

            // Update all contact entries with the new field_key
            $wpdb->update(
                $entries_table,
                ['field_key' => $new_field_key],
                ['field_key' => $old_field_key],
                ['%s'],
                ['%s']
            );
        }

        // Update the field definition
        $updated = $wpdb->update(
            $definition_table,
            [
                'field_key' => $new_field_key,
                'label' => $label,
                'type' => $type,
                'options' => $options,
                'required' => $required,
            ],
            ['field_key' => $old_field_key],
            ['%s', '%s', '%s', '%s', '%d'],
            ['%s']
        );

        if ($updated === false) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => __('Failed to update the custom field.', 'mailerpress-pro')
            ], 500);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => __('Custom field updated successfully.', 'mailerpress-pro'),
            'field' => [
                'id' => $existing->id,
                'label' => $label,
                'value' => $new_field_key,
                'field_key' => $new_field_key,
                'type' => $type,
                'required' => $required,
                'options' => is_array($body['options']) ? $body['options'] : [],
            ],
        ], 200);
    }

    #[Endpoint(
        'custom-fields/list',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageSettings'],
    )]
    public function all_custom_fields(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $table = Tables::get(Tables::MAILERPRESS_CUSTOM_FIELD_DEFINITIONS);

        $search = $request->get_param('search');
        $per_page = isset($_GET['perPages']) ? (int)(sanitize_key(wp_unslash($_GET['perPages']))) : 20;
        $page = isset($_GET['paged']) ? (int)(sanitize_key(wp_unslash($_GET['paged']))) : 1;
        $offset = ($page - 1) * $per_page;

        // Filters
        $where = '1=1';
        $params = [];
        if (!empty($search)) {
            $where .= ' AND t.label LIKE %s';
            $params[] = '%' . $wpdb->esc_like($search) . '%';
        }

        // Ordering
        $allowed_orderby = ['label', 'type', 'field_key', 'id', 'created_at'];
        $allowed_order = ['ASC', 'DESC'];
        $orderby_param = $request->get_param('orderby');
        $order_param = strtoupper($request->get_param('order') ?? 'DESC');

        $orderby = in_array($orderby_param, $allowed_orderby, true) ? $orderby_param : 'label';
        $order = in_array($order_param, $allowed_order, true) ? $order_param : 'ASC';
        $orderBy = sprintf('t.%s %s', $orderby, $order);

        // Total count
        $total_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} t WHERE {$where}",
            $params
        ));

        $total_pages = ceil($total_count / $per_page);

        // Get custom field definitions
        $query = $wpdb->prepare(
            "SELECT
            t.*,
            t.id as id
        FROM {$table} t
        WHERE {$where}
        ORDER BY {$orderBy}
        LIMIT %d OFFSET %d",
            [...$params, $per_page, $offset]
        );

        $posts = $wpdb->get_results($query);

        // ✅ Unserialize options
        foreach ($posts as $post) {
            $post->options = maybe_unserialize($post->options);
        }

        // Build response
        $response = [
            'posts' => $posts,
            'pages' => $total_pages,
            'count' => $total_count,
        ];

        return new \WP_REST_Response($response, 200);
    }

}
