<?php

namespace MailerPressPro\Services\AI\Images;

use MailerPressPro\Core\ImageGeneratorInterface;
use WP_Error;

class GeminiImageGenerator implements ImageGeneratorInterface
{
    private string $apiKey;
    private string $modelId;

    public function __construct(string $apiKey, string $modelId = 'gemini-2.5-flash-image')
    {
        $this->apiKey = $apiKey;
        $this->modelId = $modelId;
    }

    public function generate(string $prompt, int $number = 1, string $size = '1:1'): array|WP_Error
    {
        // For testing (no real API call)
        if (defined('MP_FAKE_AI_IMAGES') && MP_FAKE_AI_IMAGES) {
            return ["Bien sûr, voici un chat dans les nuages pour vous :"];
        }

        $endpoint = sprintf(
            'https://generativelanguage.googleapis.com/v1beta/models/%s:generateContent?key=%s',
            $this->modelId,
            $this->apiKey
        );

        $payload = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $prompt],
                    ],
                ],
            ],
            'generationConfig' => [
                'imageConfig' => [
                    'aspectRatio' => $size,
                ],
            ],
        ];

        $response = wp_remote_post($endpoint, [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($payload),
            'timeout' => 90,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('http_error', $response->get_error_message(), ['status' => 500]);
        }

        $body = wp_remote_retrieve_body($response);
        $respData = json_decode($body, true);
        if ($respData === null) {
            return new WP_Error('invalid_json', 'Could not parse JSON: ' . $body);
        }

        if (isset($respData['error'])) {
            return new WP_Error('api_error', $respData['error']['message'] ?? 'Unknown error');
        }

        $images = [];
        // The response has "candidates"
        foreach ($respData['candidates'] ?? [] as $candidate) {
            foreach ($candidate['content']['parts'] ?? [] as $part) {
                // If there is inlineData, use it
                if (!empty($part['inlineData']['data'])) {
                    $mime = $part['inlineData']['mimeType'] ?? 'image/png';
                    $images[] = 'data:' . $mime . ';base64,' . $part['inlineData']['data'];
                }
            }
        }

        if (empty($images)) {
            return new WP_Error('no_image', 'No image content returned.');
        }

        return $images;
    }

    public function variation(string $prompt, string $imageUrl, int $number = 1, string $size = '1:1'): array|WP_Error
    {
        // For testing (fake mode)
        if (defined('MP_FAKE_AI_IMAGES') && MP_FAKE_AI_IMAGES) {
            return ["Variation of $imageUrl (simulated)"];
        }

        // Detect if $imageUrl is already base64
        if (str_starts_with($imageUrl, 'data:')) {
            $base64 = preg_replace('#^data:image/\w+;base64,#i', '', $imageUrl);
        } else {
            $imageData = @file_get_contents($imageUrl);
            if ($imageData === false) {
                return new WP_Error('invalid_image_url', __('Could not fetch the provided image URL.', 'mailerpress'));
            }
            $base64 = base64_encode($imageData);
        }

        $endpoint = sprintf(
            'https://generativelanguage.googleapis.com/v1beta/models/%s:generateContent?key=%s',
            $this->modelId,
            $this->apiKey
        );

        $payload = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $prompt],
                        ['inlineData' => [
                            'mimeType' => 'image/png',
                            'data' => $base64,
                        ]],
                    ],
                ],
            ],
            'generationConfig' => [
                'imageConfig' => [
                    'aspectRatio' => $size, // one of: "1:1", "2:3", "3:2", etc.
                ],
            ],
        ];

        $response = wp_remote_post($endpoint, [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => wp_json_encode($payload),
            'timeout' => 120,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('http_error', $response->get_error_message(), ['status' => 500]);
        }

        $body = wp_remote_retrieve_body($response);
        $respData = json_decode($body, true);

        if ($respData === null) {
            return new WP_Error('invalid_json', 'Could not parse JSON: ' . $body);
        }

        if (isset($respData['error'])) {
            return new WP_Error('api_error', $respData['error']['message'] ?? 'Unknown API error');
        }

        $images = [];
        foreach ($respData['candidates'] ?? [] as $candidate) {
            foreach ($candidate['content']['parts'] ?? [] as $part) {
                if (!empty($part['inlineData']['data'])) {
                    $mime = $part['inlineData']['mimeType'] ?? 'image/png';
                    $images[] = 'data:' . $mime . ';base64,' . $part['inlineData']['data'];
                }
            }
        }

        if (empty($images)) {
            return new WP_Error('no_variation', __('No variation image returned by Gemini.', 'mailerpress'));
        }

        return $images;
    }

}
