<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\SegmentCondition;
use MailerPressPro\Core\Segmentation\Operators;

/**
 * Segment condition: filter contacts based on whether (and when) they have made a WooCommerce purchase.
 *
 * `value` semantics by operator:
 * ---------------------------------
 * - IS       : value ignored. Means **has purchased at least once**.
 * - IS_NOT   : value ignored. Means **has never purchased**.
 * - AFTER    : value **must** be `YYYY-MM-DD`. Contact must have a purchase *after* that date.
 * - BEFORE   : value **must** be `YYYY-MM-DD`. Contact must have a purchase *before* that date.
 */
class WcPaidCustomer implements SegmentCondition
{
    private string $operator;
    private ?string $date;

    private const ALLOWED_OPERATORS = [
        Operators::IS,
        Operators::IS_NOT,
        Operators::AFTER,
        Operators::BEFORE,
    ];

    public function __construct(?string $value, string $operator)
    {
        $operator = strtolower($operator);

        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} is not supported for WcPaidCustomer.");
        }

        if ($operator === Operators::AFTER || $operator === Operators::BEFORE) {
            if (!$value || !$this->validateDate($value)) {
                throw new \InvalidArgumentException("Invalid date format: {$value}. Expected YYYY-MM-DD.");
            }
            $this->date = $value;
        } else {
            $this->date = null;
        }

        $this->operator = $operator;
    }

    public function toSqlCondition(): string
    {
        global $wpdb;

        $ordersTable = $wpdb->prefix . 'posts';
        $metaTable = $wpdb->prefix . 'postmeta';
        $wcOrdersTable = $wpdb->prefix . 'wc_orders';

        // Email match condition for wp_posts + postmeta
        $emailMatchCondition = "pm.meta_key = '_billing_email' AND LOWER(pm.meta_value) = LOWER(c.email)";

        // Check if custom wc_orders table exists
        $useCustomOrdersTable = $wpdb->get_var(
                $wpdb->prepare(
                    "SHOW TABLES LIKE %s",
                    $wcOrdersTable
                )
            ) === $wcOrdersTable;

        if ($useCustomOrdersTable) {
            // Use custom wc_orders table queries
            switch ($this->operator) {
                case Operators::IS:
                    return sprintf(
                        "EXISTS (
                        SELECT 1 FROM %s o
                        WHERE o.billing_email = c.email
                          AND o.status IN ('wc-completed', 'wc-processing', 'wc-on-hold')
                    )",
                        $wcOrdersTable
                    );

                case Operators::IS_NOT:
                    return sprintf(
                        "NOT EXISTS (
                        SELECT 1 FROM %s o
                        WHERE o.billing_email = c.email
                          AND o.status IN ('wc-completed', 'wc-processing', 'wc-on-hold')
                    )",
                        $wcOrdersTable
                    );

                case Operators::AFTER:
                    $escapedDate = esc_sql($this->date);
                    return sprintf(
                        "EXISTS (
                        SELECT 1 FROM %s o
                        WHERE o.billing_email = c.email
                          AND o.status IN ('wc-completed', 'wc-processing', 'wc-on-hold')
                          AND o.date_created_gmt > '%s'
                    )",
                        $wcOrdersTable,
                        $escapedDate
                    );

                case Operators::BEFORE:
                    $endOfDay = esc_sql($this->date . ' 23:59:59');
                    return sprintf(
                        "EXISTS (
                        SELECT 1 FROM %s o
                        WHERE o.billing_email = c.email
                          AND o.status IN ('wc-completed', 'wc-processing', 'wc-on-hold')
                          AND o.date_created_gmt < '%s'
                    )",
                        $wcOrdersTable,
                        $endOfDay
                    );
            }
        }

        return '';
    }

    private function validateDate(string $date): bool
    {
        $d = \DateTime::createFromFormat('Y-m-d', $date);
        return $d && $d->format('Y-m-d') === $date;
    }

    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'wc_paid_customer',
            'label' => __('WooCommerce Purchase History', 'mailerpress-pro'),
            'operators' => [
                Operators::IS => 'has purchased',
                Operators::IS_NOT => 'has not purchased',
                Operators::AFTER => 'purchased after date',
                Operators::BEFORE => 'purchased before date',
            ],
            'input_type_map' => [
                Operators::IS => 'boolean',
                Operators::IS_NOT => 'boolean',
                Operators::AFTER => 'date',
                Operators::BEFORE => 'date',
            ],
            'input_type' => 'boolean_or_date',
        ];
    }

    public static function canRun(): bool
    {
        return is_plugin_active('woocommerce/woocommerce.php');
    }
}
