<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\SegmentCondition;
use MailerPressPro\Core\Segmentation\Operators;

class UpdatedAtCondition implements SegmentCondition
{
    private string $operator;
    private string $date;

    private const FIELD = 'c.updated_at'; // Fixed field
    private const ALLOWED_OPERATORS = [
        Operators::AFTER,
        Operators::BEFORE,
        Operators::ON,
    ];

    public function __construct(string $operator, string $date)
    {
        $operator = strtolower($operator);

        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} is not supported for UpdatedAtCondition.");
        }

        if (!$this->validateDate($date)) {
            throw new \InvalidArgumentException("Invalid date format: {$date}. Expected YYYY-MM-DD.");
        }

        $this->operator = $operator;
        $this->date = $date;
    }

    public function toSqlCondition(): string
    {
        $op = match ($this->operator) {
            Operators::BEFORE => '<',
            Operators::AFTER  => '>',
            Operators::ON     => '=',
            default           => throw new \LogicException("Unexpected operator: {$this->operator}"),
        };

        $safeDate = esc_sql($this->date);
        $dateTimeSuffix = ($this->operator === Operators::BEFORE || $this->operator === Operators::ON) ? ' 23:59:59' : '';

        return sprintf("%s %s '%s%s'", self::FIELD, $op, $safeDate, $dateTimeSuffix);
    }

    private function validateDate(string $date): bool
    {
        $d = \DateTime::createFromFormat('Y-m-d', $date);
        return $d && $d->format('Y-m-d') === $date;
    }

    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'updated_at',
            'label' => __('Updated Date', 'mailerpress-pro'),
            'operators' => [
                Operators::AFTER => 'after',
                Operators::BEFORE => 'before',
                Operators::ON => 'on',
            ],
            'input_type' => 'date',
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}
