<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\Operators;
use MailerPressPro\Core\Segmentation\SegmentCondition;

class SubscriptionStatusCondition implements SegmentCondition
{
    private string $status;
    private string $operator;

    private const ALLOWED_OPERATORS = [
        Operators::EQUALS,
        Operators::NOT_EQUALS,
    ];

    public function __construct(string $status, string $operator = Operators::EQUALS)
    {
        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} not supported for SubscriptionStatusCondition");
        }

        $this->status = sanitize_key(strtolower($status));
        $this->operator = $operator;
    }

    public function toSqlCondition(): string
    {
        $opSql = $this->operator === Operators::NOT_EQUALS ? '<>' : '=';

        return sprintf(
            "c.subscription_status %s '%s'",
            $opSql,
            esc_sql($this->status)
        );
    }

    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'subscription_status',
            'label' => __('Subscription status', 'mailerpress-pro'),
            'operators' => [
                Operators::EQUALS => 'equals',
                Operators::NOT_EQUALS => 'not equals',
            ],
            'input_type' => 'select', // dropdown select input
            // optionally you could add a list of allowed values here too
            'allowed_values' => [
                'subscribed' => 'Subscribed',
                'unsubscribed' => 'Unsubscribed',
                'pending' => 'Pending',
            ],
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}
