<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\SegmentCondition;
use MailerPressPro\Core\Segmentation\Operators;

/**
 * Segment condition: filter contacts based on whether (and when) they opened an email.
 *
 * `value` semantics by operator
 * ---------------------------------
 * - IS       : value ignored. Means **has opened at least once**.
 * - IS_NOT   : value ignored. Means **has never opened**.
 * - AFTER    : value **must** be `YYYY-MM-DD`. Contact must have an open *after* that date.
 * - BEFORE   : value **must** be `YYYY-MM-DD`. Contact must have an open *on/before* that date
 *              (comparison is `< date 23:59:59`).
 */
class OpenEmailCondition implements SegmentCondition
{
    private string $operator;   // normalized lower-case operator
    private ?string $date;      // date string when operator requires one, otherwise null

    private const ALLOWED_OPERATORS = [
        Operators::IS,
        Operators::IS_NOT,
        Operators::AFTER,
        Operators::BEFORE,
    ];

    /**
     * @param string|null $value    `YYYY-MM-DD` for AFTER/BEFORE (required there), ignored for IS/IS_NOT.
     * @param string      $operator One of Operators::IS, ::IS_NOT, ::AFTER, ::BEFORE.
     */
    public function __construct(?string $value, string $operator)
    {
        $operator = strtolower($operator);

        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} is not supported for OpenEmailCondition.");
        }

        // Only validate/retain a date when the operator requires one.
        if ($operator === Operators::AFTER || $operator === Operators::BEFORE) {
            if (!$value || !$this->validateDate($value)) {
                throw new \InvalidArgumentException("Invalid date format: {$value}. Expected YYYY-MM-DD.");
            }
            $this->date = $value;
        } else {
            // IS / IS_NOT -> ignore provided value entirely.
            $this->date = null;
        }

        $this->operator = $operator;
    }

    /**
     * Produce the SQL snippet for this condition.
     *
     * EXPECTATION: The contacts query builder will alias the contacts table as `c`
     * and the email-open tracking table as `met`.
     *
     * @return string SQL fragment (no leading WHERE).
     */
    public function toSqlCondition(): string
    {
        global $wpdb;
        $trackingTable = $wpdb->prefix . 'mailerpress_email_tracking';
        $contactField  = 'c.contact_id'; // Assuming main table alias is 'c'

        switch ($this->operator) {
            case Operators::IS: // Has opened
                return sprintf(
                    "EXISTS (SELECT 1 FROM %s met WHERE met.contact_id = %s AND met.opened_at IS NOT NULL)",
                    $trackingTable,
                    $contactField
                );

            case Operators::IS_NOT: // Has NOT opened
                return sprintf(
                    "NOT EXISTS (SELECT 1 FROM %s met WHERE met.contact_id = %s AND met.opened_at IS NOT NULL)",
                    $trackingTable,
                    $contactField
                );

            case Operators::AFTER:
                // Use > midnight of the date (i.e., strictly after the date)
                return $wpdb->prepare(
                    "EXISTS (SELECT 1 FROM {$trackingTable} met WHERE met.contact_id = {$contactField} AND met.opened_at > %s)",
                    $this->date
                );

            case Operators::BEFORE:
                // On/before date => < date 23:59:59
                $endOfDay = $this->date . ' 23:59:59';
                return $wpdb->prepare(
                    "EXISTS (SELECT 1 FROM {$trackingTable} met WHERE met.contact_id = {$contactField} AND met.opened_at < %s)",
                    $endOfDay
                );
        }

        return '';
    }

    /**
     * Validate an input date string.
     */
    private function validateDate(string $date): bool
    {
        $d = \DateTime::createFromFormat('Y-m-d', $date);
        return $d && $d->format('Y-m-d') === $date;
    }

    /**
     * Metadata consumed by the admin UI.
     *
     * We expose operator labels AND the expected input type for each operator so the UI can render either
     * a boolean toggle (for IS / IS_NOT) or a date picker (for AFTER / BEFORE).
     */
    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'openers',
            'label' => __('Email Open Activity', 'mailerpress-pro'),
            'operators' => [
                Operators::IS     => 'has opened',
                Operators::IS_NOT => 'has not opened',
                Operators::AFTER  => 'opened after date',
                Operators::BEFORE => 'opened before date',
            ],
            'input_type_map' => [
                Operators::IS     => 'boolean',
                Operators::IS_NOT => 'boolean',
                Operators::AFTER  => 'date',
                Operators::BEFORE => 'date',
            ],
            'input_type' => 'boolean_or_date', // backward compatibility
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}
