<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\SegmentCondition;
use MailerPressPro\Core\Segmentation\Operators;

class CreatedAtCondition implements SegmentCondition
{
    private string $operator;
    private string $date;

    private const FIELD = 'c.created_at'; // Fixed field
    private const ALLOWED_OPERATORS = [
        Operators::AFTER,
        Operators::BEFORE,
        Operators::ON,
    ];

    public function __construct(string $operator, string $date)
    {
        $operator = strtolower($operator);

        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} is not supported for CreatedAtCondition.");
        }

        if (!$this->validateDate($date)) {
            throw new \InvalidArgumentException("Invalid date format: {$date}. Expected YYYY-MM-DD.");
        }

        $this->operator = $operator;
        $this->date = $date;
    }

    public function toSqlCondition(): string
    {
        $safeDate = esc_sql($this->date);

        switch ($this->operator) {
            case Operators::BEFORE:
                $op = '<';
                // Before end of day
                return sprintf("%s %s '%s 23:59:59'", self::FIELD, $op, $safeDate);

            case Operators::AFTER:
                $op = '>';
                // After start of day
                return sprintf("%s %s '%s 00:00:00'", self::FIELD, $op, $safeDate);

            case Operators::ON:
                // Range for entire day
                return sprintf(
                    "(%s >= '%s 00:00:00' AND %s <= '%s 23:59:59')",
                    self::FIELD,
                    $safeDate,
                    self::FIELD,
                    $safeDate
                );

            default:
                throw new \LogicException("Unexpected operator: {$this->operator}");
        }
    }


    private function validateDate(string $date): bool
    {
        $d = \DateTime::createFromFormat('Y-m-d', $date);
        return $d && $d->format('Y-m-d') === $date;
    }

    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'created_at',
            'label' => __('Created Date', 'mailerpress-pro'),
            'operators' => [
                Operators::AFTER => 'after',
                Operators::BEFORE => 'before',
                Operators::ON => 'on',
            ],
            'input_type' => 'date',
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}
