<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use WP_Error;

class SendGridService extends AbstractEmailService
{
    private Contacts $contactModel;
    private Lists $listModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        $apiKey = $emailData['apiKey'] ?? '';

        if (empty($apiKey)) {
            return false;
        }

        $url = 'https://api.sendgrid.com/v3/mail/send';

        // Build the request body according to SendGrid API
        $body = [
            'personalizations' => [
                [
                    'to' => array_map(fn($to) => ['email' => $to], (array)($emailData['to'] ?? [])),
                    'subject' => $emailData['subject'] ?? '',
                ],
            ],
            'from' => [
                'email' => $emailData['sender_to'] ?? get_bloginfo('admin_email'),
                'name' => $emailData['sender_name'] ?? get_bloginfo('name'),
            ],
            'content' => [
                [
                    'type' => $emailData['content_type'] ?? 'text/html',
                    'value' => $emailData['body'] ?? '',
                ],
            ],
        ];

        // Handle CC/BCC if present
        if (!empty($emailData['cc'])) {
            $body['personalizations'][0]['cc'] = array_map(fn($cc) => ['email' => $cc], (array)$emailData['cc']);
        }
        if (!empty($emailData['bcc'])) {
            $body['personalizations'][0]['bcc'] = array_map(fn($bcc) => ['email' => $bcc], (array)$emailData['bcc']);
        }

        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($body),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error(
                sprintf(
                    __('SendGrid sendEmail error: %s', 'mailerpress'),
                    $response->get_error_message()
                )
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);

        // SendGrid returns 202 Accepted on success
        if ($status_code !== 202) {
            $body = wp_remote_retrieve_body($response);

            // Try to decode JSON response from SendGrid
            $decoded = json_decode($body, true);

            if (json_last_error() === JSON_ERROR_NONE && isset($decoded['errors'][0]['message'])) {
                $errorMessage = $decoded['errors'][0]['message'];
            } else {
                $errorMessage = __('An unexpected error occurred while sending the email.', 'mailerpress');
            }

            // Log full error for admins/debug
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SendGrid sendEmail error: ' . $body);
            }

            return new WP_Error(
                'sendgrid_send_error',
                sprintf(
                /* translators: %s: Error message from SendGrid or fallback */
                    __('SendGrid error: %s', 'mailerpress'),
                    esc_html($errorMessage)
                )
            );
        }


        return true;
    }

    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'sendgrid',
            'name' => 'SendGrid',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="44" height="44" viewBox="4 3.043 92 92" fill="none"><g filter="url(#filter0_d_2944_85289)"><rect width="92" height="92" x="4" y="3.043" fill="#294661" rx="46"></rect><path fill="#A3E1F2" d="M25 40.04v17.003h16.998V74.04h17.003v-34H25Z"></path><path fill="#1A82E2" d="M41.998 57.044H25v16.998h16.998V57.044Z"></path><path fill="#00B3E3" d="M58.998 40.04V23.042H41.996V57.043h34V40.04H58.998Z"></path><path fill="#1A82E2" d="M76 23.043H59.002v16.998H76V23.043ZM42.131 40.173h17.131v17.131H42.131z"></path></g><defs><filter id="filter0_d_2944_85289" width="100" height="100" x="0" y=".043" color-interpolation-filters="sRGB" filterUnits="userSpaceOnUse"><feFlood flood-opacity="0" result="BackgroundImageFix"></feFlood><feColorMatrix in="SourceAlpha" result="hardAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"></feColorMatrix><feOffset dy="1"></feOffset><feGaussianBlur stdDeviation="2"></feGaussianBlur><feColorMatrix values="0 0 0 0 0.0687866 0 0 0 0 0.097585 0 0 0 0 0.37981 0 0 0 0.0779552 0"></feColorMatrix><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_2944_85289"></feBlend><feBlend in="SourceGraphic" in2="effect1_dropShadow_2944_85289" result="shape"></feBlend></filter></defs></svg>',
            'link' => 'https://www.brevo.com/fr/pricing/',
            'createAccountLink' => 'https://onboarding.brevo.com/account/register',
            'linkApiKey' => 'https://app.brevo.com/settings/keys/api',
            'description' => __('Send at scale with Twilio SendGrid, which boasts an industry-leading 99% deliverability rate. SendGrid offers both a free 100 emails per day plan and, if you need to exceed that limit, a selection of preset pricing plans, starting at $19.95 per month for up to 50,000 emails. For more information on getting started with SendGrid, check out our documentation.',
                'mailerpress'),
            'recommended' => false,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ]
        ];
    }

    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_token', $this->config()['key']);
        $pageToken = get_option($optionName, null);
        $limit = 100;

        $url = add_query_arg([
            'page_size' => $limit,
            'page_token' => $pageToken ?: null,
        ], 'https://api.sendgrid.com/v3/marketing/contacts');

        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            error_log('SendGrid API request failed: ' . $response->get_error_message());
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            error_log("SendGrid API returned status code {$status_code}");
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['result'])) {
            foreach ($data['result'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'] ?? null,
                    'email' => $contact['email'],
                    'remote_lists' => $contact['list_ids'] ?? [],
                    'attributes' => [
                        'first_name' => $contact['first_name'] ?? null,
                        'last_name' => $contact['last_name'] ?? null,
                    ],
                ]);
            }

            // Store the next page token if it exists
            if (!empty($data['next_page_token'])) {
                update_option($optionName, $data['next_page_token']);
                // Re-enqueue next batch
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            } else {
                // No more pages, clear stored token
                delete_option($optionName);
            }
        }
    }


    public function syncLists(string $apiKey): void
    {
        $limit = 20;
        $offset = 0;

        do {
            $url = add_query_arg([
                'page_size' => $limit,
                'page_token' => $offset > 0 ? $offset : null, // SendGrid uses tokens for pagination
            ], 'https://api.sendgrid.com/v3/marketing/lists');

            $response = wp_remote_get($url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Accept' => 'application/json',
                ],
            ]);

            if (is_wp_error($response)) {
                error_log('SendGrid API request failed: ' . $response->get_error_message());
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                error_log("SendGrid API returned status code {$status_code}");
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            $lists = $data['result'] ?? [];

            if (!empty($lists)) {
                foreach ($lists as $list) {
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }
            }

            $offset = $data['next_page_token'] ?? null;
        } while ($offset);
    }

    public function syncTags(string $apiKey): void
    {
    }

    public function pushContacts(string $apiKey, array $data): mixed
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];

        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                     FROM $list_sync_table
                     WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)
                    )
                );
            }

            $url = 'https://api.sendgrid.com/v3/marketing/contacts';

            $body = json_encode([
                'list_ids' => $remoteLists,
                'contacts' => [
                    [
                        'email' => $contact->email,
                        'firstName' => $contact->first_name,
                        'lastName' => $contact->last_name,
                    ]
                ]
            ]);

            $response = wp_remote_post($url, [
                'method' => 'PUT',
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['errors'])) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'SENDGRID_ERROR',
                    'error_message' => json_encode($data['errors']),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data)) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $contact->email, // SendGrid does not return ID, use email
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }

            return $data;
        }

        return null;
    }


    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            // Create list in SendGrid
            $response = wp_remote_post('https://api.sendgrid.com/v3/marketing/lists', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error('sendgrid_request_failed', 'Request to SendGrid failed.',
                    $response->get_error_message());
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('sendgrid_api_error', 'SendGrid API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }


    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        // TODO: Implement deleteLists() method.
    }

    public function deleteAllLists(string $apiKey): void
    {
        // TODO: Implement deleteAllLists() method.
    }

    public function deleteContact(string $apiKey, array $data)
    {
        // TODO: Implement deleteContact() method.
    }
}
