<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use WP_Error;

class Resend extends AbstractEmailService
{
    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        // Load Resend API key from options or $emailData
        $services   = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['resend']['conf'] ?? [];
        $apiKey     = $emailData['apiKey'] ?? $clientConf['api_key'] ?? '';

        if (empty($apiKey)) {
            return new WP_Error('resend_missing_key', __('Resend API key is missing', 'mailerpress-pro'));
        }

        $url = 'https://api.resend.com/emails';

        // Build recipients
        $to = [];
        foreach ((array)($emailData['to'] ?? []) as $recipient) {
            $to[] = ['email' => $recipient];
        }

        $cc = [];
        foreach ((array)($emailData['cc'] ?? []) as $recipient) {
            $cc[] = ['email' => $recipient];
        }

        $bcc = [];
        foreach ((array)($emailData['bcc'] ?? []) as $recipient) {
            $bcc[] = ['email' => $recipient];
        }

        $body = [
            'from'    => sprintf(
                '%s <%s>',
                $emailData['sender_name'] ?? get_bloginfo('name'),
                $emailData['sender_to'] ?? get_bloginfo('admin_email')
            ),
            'to'      => (array)($emailData['to'] ?? []),  // array of strings
            'cc'      => !empty($emailData['cc']) ? (array)$emailData['cc'] : null,
            'bcc'     => !empty($emailData['bcc']) ? (array)$emailData['bcc'] : null,
            'subject' => $emailData['subject'] ?? '',
            'html'    => $emailData['body'] ?? '',
        ];

        // Remove null values
        $body = array_filter($body, fn($v) => !is_null($v));

        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode($body),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error(
                'resend_request_failed',
                sprintf(__('Resend request failed: %s', 'mailerpress-pro'), $response->get_error_message())
            );
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $resBody    = wp_remote_retrieve_body($response);
        $decoded    = json_decode($resBody, true);

        if ($statusCode !== 202 && $statusCode !== 200) {
            $errorMsg = $decoded['error'] ?? __('Unknown Resend error', 'mailerpress-pro');

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[MailerPress Resend] Send error: ' . $resBody);
            }

            return new WP_Error('resend_send_error', sprintf(__('Resend API error: %s', 'mailerpress-pro'), esc_html($errorMsg)));
        }

        return true;
    }


    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'resend',
            'name' => __('Resend', 'mailerpress-pro'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="44" height="44" viewBox="0 0 92 92" fill="none"><rect width="92" height="92" rx="46" fill="black"></rect><path d="M50.4381 24.75C58.5726 24.75 63.4101 29.5879 63.4101 36.0098C63.4101 42.4318 58.5726 47.2696 50.4381 47.2696H46.3278L66.75 66.75H52.3216L36.7804 51.9791C35.6674 50.9519 35.1537 49.7531 35.1536 48.7257C35.1536 47.27 36.1814 45.9856 38.1508 45.429L46.1566 43.2881C49.1963 42.4746 51.2945 40.1198 51.2945 37.0372C51.294 33.2698 48.2114 31.0864 44.4011 31.0864H24.75V24.75H50.4381Z" fill="white"></path></svg>',
            'link' => 'https://mailersend.com/',
            'createAccountLink' => 'https://resend.com/signup',
            'linkApiKey' => 'https://resend.com/api-keys',
            'description' => __('Email for developers that makes it easy to write, format, and send emails. Deliver transactional and marketing emails at scale.', 'mailerpress'),
            'recommended' => true,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ]
        ];
    }


}
