<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use WP_Error;

class PostMark extends AbstractEmailService
{
    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        // Load Postmark API key from options or $emailData
        $services   = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['postmark']['conf'] ?? [];
        $apiKey     = $emailData['apiKey'] ?? $clientConf['api_key'] ?? '';

        if (empty($apiKey)) {
            return new WP_Error('postmark_missing_key', __('Postmark API key is missing', 'mailerpress-pro'));
        }

        $url = 'https://api.postmarkapp.com/email';

        // Prepare recipients
        $to  = implode(',', (array)($emailData['to'] ?? []));
        $cc  = !empty($emailData['cc']) ? implode(',', (array)$emailData['cc']) : null;
        $bcc = !empty($emailData['bcc']) ? implode(',', (array)$emailData['bcc']) : null;

        // Prepare request body
        $body = [
            'From'     => sprintf(
                '%s <%s>',
                $emailData['sender_name'] ?? get_bloginfo('name'),
                $emailData['sender_to'] ?? get_bloginfo('admin_email')
            ),
            'To'       => $to,
            'Cc'       => $cc,
            'Bcc'      => $bcc,
            'Subject'  => $emailData['subject'] ?? '',
            'HtmlBody' => $emailData['body'] ?? '',
        ];

        // Remove null values
        $body = array_filter($body, fn($v) => !is_null($v));

        // Send request
        $response = wp_remote_post($url, [
            'headers' => [
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
                'X-Postmark-Server-Token' => $apiKey,
            ],
            'body'    => wp_json_encode($body),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error(
                'postmark_request_failed',
                sprintf(__('Postmark request failed: %s', 'mailerpress-pro'), $response->get_error_message())
            );
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $resBody    = wp_remote_retrieve_body($response);
        $decoded    = json_decode($resBody, true);

        if ($statusCode !== 200) {
            $errorMsg = $decoded['Message'] ?? __('Unknown Postmark error', 'mailerpress-pro');

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[MailerPress Postmark] Send error: ' . $resBody);
            }

            return new WP_Error('postmark_send_error', sprintf(__('Postmark API error: %s', 'mailerpress-pro'), esc_html($errorMsg)));
        }

        return true;
    }


    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'postmark',
            'name' => __('PostMark', 'mailerpress-pro'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="44" height="44" fill="none" viewBox="4 3 92 92"><g filter="url(#filter0_d_3041_90643)"><rect width="92" height="92" x="4" y="3" fill="#FFDE00" rx="46"></rect><path fill="#000" d="M32.135 66.58h2.608c1.34 0 2.186-.86 2.186-2.223V33.961c0-1.362-.846-2.222-2.186-2.222h-2.608V25H51.03c10.152 0 18.19 5.233 18.19 15.198 0 10.036-8.038 15.27-18.19 15.27h-6.487v8.89c0 1.361.846 2.221 2.256 2.221h5.288v6.811H32.135v-6.81Zm18.33-18.138c6.698 0 10.576-2.867 10.576-8.1 0-5.377-3.877-8.101-10.575-8.101h-5.923v16.273h5.922v-.072Z"></path></g><defs><filter id="filter0_d_3041_90643" width="100" height="100" x="0" y="0" color-interpolation-filters="sRGB" filterUnits="userSpaceOnUse"><feFlood flood-opacity="0" result="BackgroundImageFix"></feFlood><feColorMatrix in="SourceAlpha" result="hardAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"></feColorMatrix><feOffset dy="1"></feOffset><feGaussianBlur stdDeviation="2"></feGaussianBlur><feColorMatrix values="0 0 0 0 0.0687866 0 0 0 0 0.097585 0 0 0 0 0.37981 0 0 0 0.0779552 0"></feColorMatrix><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_3041_90643"></feBlend><feBlend in="SourceGraphic" in2="effect1_dropShadow_3041_90643" result="shape"></feBlend></filter></defs></svg>',
            'link' => 'https://postmarkapp.com',
            'createAccountLink' => 'https://account.postmarkapp.com/sign_up',
            'linkApiKey' => 'https://account.postmarkapp.com/servers',
            'description' => __('Owned by ActiveCampaign, Postmark is a popular email-sending service with an impressive reputation for reliability and deliverability. Postmark offers a free plan that allows you to send up to 100 emails a month. Over 100, prices vary depending on the number of emails sent. For more information on how to get started with Postmark, read our documentation.', 'mailerpress'),
            'recommended' => false,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ]
        ];
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_offset', $this->config()['key']);
        $limit = 100;
        $offset = 0;

        $url = sprintf(
            "https://api.brevo.com/v3/contacts?limit=%d&offset=%d",
            $limit,
            $offset
        );

        $response = wp_remote_get($url, [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            error_log('Brevo API request failed: ' . $response->get_error_message());
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            error_log("Brevo API returned status code {$status_code}");
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['contacts'])) {
            foreach ($data['contacts'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'],
                    'email' => $contact['email'],
                    'remote_lists' => $contact['listIds'],
                    'attributes' => [
                        'first_name' => $contact['attributes']['PRENOM'],
                        'last_name' => $contact['attributes']['NOM'],
                    ],
                ]);
            }

            // Update the offset
            $newOffset = $offset + count($data['contacts']);
            update_option($optionName, $newOffset);

            if (count($data['contacts']) === $limit) {
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            }
        }
    }

    public function syncLists(string $apiKey): void
    {
        $optionName = sprintf('%s_lists_last_offset', $this->config()['key']);
        $limit = 20;
        $offset = (int)get_option($optionName, 0);
        $total = null;

        do {
            $url = sprintf(
                "https://api.brevo.com/v3/contacts/lists?limit=%d&offset=%d",
                $limit,
                $offset
            );

            $response = wp_remote_get($url, [
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                ]
            ]);

            if (is_wp_error($response)) {
                error_log('Brevo API request failed: ' . $response->get_error_message());
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                error_log("Brevo API returned status code {$status_code}");
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if ($total === null && isset($data['count'])) {
                $total = (int)$data['count'];
            }

            $lists = $data['lists'] ?? [];
            if (!empty($lists)) {
                foreach ($lists as $list) {
                    // Do something with the list
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }

                // Update the stored offset after each batch
                $offset += $limit;
                update_option($optionName, $offset);
            }


        } while ($offset < $total);

        delete_option($optionName);

    }

    public function syncTags(string $apiKey): void
    {
    }

    public function pushContacts(string $apiKey, array $data): mixed
    {

        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];

        error_log('cotact' . json_encode($contact));


        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            // Insert the contact
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                 FROM $list_sync_table
                 WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)

                    )
                );
            }

            $url = 'https://api.brevo.com/v3/contacts';

            $body = json_encode([
                'email' => $contact->email,
                'attributes' => [
                    'FIRSTNAME' => $contact->first_name,
                    'LASTNAME' => $contact->last_name,
                ],
                'listIds' => array_map('intval', $remoteLists), // Convert list IDs to integers
                'updateEnabled' => true, // To update the contact if already exists
            ]);

            $response = wp_remote_post($url, [
                'method' => 'POST',
                'headers' => [
                    'api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60, // Set a timeout for the request
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR', // You can set this based on your error
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            // Successfully added/updated the contact in Brevo
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['code']) && $data['code'] != 200) {
                // Handle API error response if any
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $data['code'], // You can set this based on your error
                    'error_message' => $data['message'],
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data['id'])) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $data['id'],
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }


            return $data;
        }

        return null;
    }

    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS); // Adjust to your actual constant
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $folderId = $this->getFolder($apiKey, 'MailerPress');

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            $response = wp_remote_post('https://api.brevo.com/v3/contacts/lists', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'api-key' => $apiKey,
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                    'folderId' => (int)$folderId,
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error('brevo_request_failed', 'Request to Brevo failed.', $response->get_error_message());
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('brevo_api_error', 'Brevo API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");


        $remoteListId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_list_id FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d",
                $listId,
                $active_provider_id
            )
        );

        if ($remoteListId) {
            wp_remote_request("https://api.brevo.com/v3/contacts/lists/{$remoteListId}", [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);
        }

    }

    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    private function getFolder(string $apiKey, string $folder_name)
    {
//        $option_key = 'mailerpress_brevo_folder_id';
//        $folder_id = get_option($option_key);
//
//        if ($folder_id) {
//            return $folder_id;
//        }

        // Try to find the folder via Brevo API
        $response = wp_remote_get('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Failed to fetch folders.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200 || empty($body['folders'])) {
            return new WP_Error('brevo_api_error', 'Error fetching folders list.', $body);
        }


        // Check if folder already exists
        foreach ($body['folders'] as $folder) {
            if (strcasecmp($folder['name'], $folder_name) === 0) {
//                update_option($option_key, $folder['id']);
                return $folder['id'];
            }
        }

        // Folder doesn't exist, so create it
        $response = wp_remote_post('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'body' => wp_json_encode([
                'name' => $folder_name,
            ]),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Folder creation failed.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 201) {
            return new WP_Error('brevo_api_error', 'Brevo API error.', $body);
        }

//        update_option($option_key, $body['id']);

        return $body['id'];
    }


    public function deleteAllLists(string $apiKey): void
    {
        $folder_id = get_option('mailerpress_brevo_folder_id');

        if (!$folder_id) {
            return;
        }

        $response = wp_remote_request("https://api.brevo.com/v3/contacts/folders/{$folder_id}", [
            'method' => 'DELETE',
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            error_log("MailerPress: Failed to delete Brevo folder #{$folder_id} - " . $response->get_error_message());
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204) {
            // Folder deleted successfully
            delete_option('mailerpress_brevo_folder_id');
        } else {
            $body = wp_remote_retrieve_body($response);
            error_log("MailerPress: Failed to delete Brevo folder #{$folder_id} - {$body}");
        }
    }

    public function deleteContact(string $apiKey, array $data): void
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return;
        }

        $contact = $data['contact'];

        if (empty($contact->email)) {
            return;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);

        $active_provider_id = $wpdb->get_var(
            "SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1"
        );

        $remoteContactId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_contact_id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d",
                $contact->contact_id,
                $active_provider_id
            )
        );

        if ($remoteContactId) {
            $response = wp_remote_request("https://api.brevo.com/v3/contacts/" . urlencode($contact->email), [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code === 204) {
                // Delete local sync entry after successful API deletion
                $wpdb->delete(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => $active_provider_id,
                    ],
                    ['%d', '%d']
                );
            } else {
                $response_body = wp_remote_retrieve_body($response);
                $error = json_decode($response_body, true);
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $error['code'] ?? 'UNKNOWN',
                    'error_message' => $error['message'] ?? 'Unknown error',
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
            }
        }
    }

}
