<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use WP_Error;

class Mailjet extends AbstractEmailService
{
    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        // Load Mailjet config from options
        $services   = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailjet']['conf'] ?? [];

        $apiKey    = $clientConf['api_key'] ?? '';
        $apiSecret = $clientConf['api_secret'] ?? '';

        if (empty($apiKey) || empty($apiSecret)) {
            return new WP_Error('mailjet_config_missing', __('Mailjet API key/secret is missing.', 'mailerpress'));
        }

        if (empty($emailData['to'])) {
            return new WP_Error('mailjet_no_recipient', __('No recipient specified.', 'mailerpress'));
        }

        $url = 'https://api.mailjet.com/v3.1/send';

        $message = [
            'From' => [
                'Email' => $emailData['sender_to'] ?? get_bloginfo('admin_email'),
                'Name'  => $emailData['sender_name'] ?? get_bloginfo('name'),
            ],
            'To'       => array_map(fn($to) => ['Email' => $to], (array)$emailData['to']),
            'Subject'  => $emailData['subject'] ?? '',
            'HTMLPart' => $emailData['body'] ?? '',
        ];

        if (!empty($emailData['cc'])) {
            $message['Cc'] = array_map(fn($cc) => ['Email' => $cc], (array)$emailData['cc']);
        }

        if (!empty($emailData['bcc'])) {
            $message['Bcc'] = array_map(fn($bcc) => ['Email' => $bcc], (array)$emailData['bcc']);
        }

        $body = ['Messages' => [$message]];

        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode($body),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('mailjet_request_failed', $response->get_error_message());
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $resBody    = wp_remote_retrieve_body($response);
        error_log('[MailerPress Mailjet] Response: ' . $resBody);

        if ($statusCode !== 200) {
            $decoded = json_decode($resBody, true);
            $errorMessage = 'An unexpected error occurred.';

            if (json_last_error() === JSON_ERROR_NONE && isset($decoded['Messages'][0]['Errors'][0]['ErrorMessage'])) {
                $errorMessage = $decoded['Messages'][0]['Errors'][0]['ErrorMessage'];
            }

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[MailerPress Mailjet] API error (' . $statusCode . '): ' . $resBody);
            }

            return new WP_Error('mailjet_send_error', sprintf(__('Mailjet error: %s', 'mailerpress'), esc_html($errorMessage)));
        }

        return true;
    }


    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'mailjet',
            'name' => __('Mailjet', 'mailerpress-pro'),
            'icon' => '<svg width="32" height="32" fill="rgb(111, 103, 217)" role="img" viewBox="0 0 104 104" color="mj50" class="StyledIcon__StyledSVG-eKBrmB kCMxxC StyledIcon-epMSEY cdLgWb MailjetLogoMono-crPHje jrTbNw c-n StyledIcon-epMSEY cdLgWb MailjetLogoMono-crPHje jrTbNw"><g><path d="m93.5,10.5c-2.45-2.46-6.12-3.21-9.35-1.92L12.17,37.37c-3.21,1.29-5.3,4.23-5.44,7.69-.14,3.46,1.69,6.56,4.78,8.11l21.31,10.65,7.75-7.75-20.96-10.48,64.65-25.86-25.86,64.65-10.48-20.96-7.75,7.75,10.65,21.31c1.49,2.98,4.43,4.79,7.73,4.79.12,0,.25,0,.38,0,3.46-.14,6.4-2.23,7.69-5.44l28.79-71.99c1.29-3.22.54-6.89-1.92-9.35Z"></path></g></svg>',
            'link' => 'https://mailjet.com/',
            'createAccountLink' => 'https://app.mailjet.com/signup',
            'linkApiKey' => 'https://app.mailjet.com/account/apikeys',
            'description' => __('Mailjet is an all-in-one email delivery service designed for both marketing and transactional emails. It offers powerful APIs, real-time analytics, and advanced personalization features. With a free plan that allows sending up to 6,000 emails per month (200 emails per day), Mailjet makes it easy to scale as your needs grow. For more information on how to get started with Mailjet, read our documentation.', 'mailerpress'),
            'recommended' => false,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ]
        ];
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_offset', $this->config()['key']);
        $limit = 100;
        $offset = 0;

        $url = sprintf(
            "https://api.brevo.com/v3/contacts?limit=%d&offset=%d",
            $limit,
            $offset
        );

        $response = wp_remote_get($url, [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            error_log('Brevo API request failed: ' . $response->get_error_message());
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            error_log("Brevo API returned status code {$status_code}");
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['contacts'])) {
            foreach ($data['contacts'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'],
                    'email' => $contact['email'],
                    'remote_lists' => $contact['listIds'],
                    'attributes' => [
                        'first_name' => $contact['attributes']['PRENOM'],
                        'last_name' => $contact['attributes']['NOM'],
                    ],
                ]);
            }

            // Update the offset
            $newOffset = $offset + count($data['contacts']);
            update_option($optionName, $newOffset);

            if (count($data['contacts']) === $limit) {
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            }
        }
    }

    public function syncLists(string $apiKey): void
    {
        $optionName = sprintf('%s_lists_last_offset', $this->config()['key']);
        $limit = 20;
        $offset = (int)get_option($optionName, 0);
        $total = null;

        do {
            $url = sprintf(
                "https://api.brevo.com/v3/contacts/lists?limit=%d&offset=%d",
                $limit,
                $offset
            );

            $response = wp_remote_get($url, [
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                ]
            ]);

            if (is_wp_error($response)) {
                error_log('Brevo API request failed: ' . $response->get_error_message());
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                error_log("Brevo API returned status code {$status_code}");
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if ($total === null && isset($data['count'])) {
                $total = (int)$data['count'];
            }

            $lists = $data['lists'] ?? [];
            if (!empty($lists)) {
                foreach ($lists as $list) {
                    // Do something with the list
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }

                // Update the stored offset after each batch
                $offset += $limit;
                update_option($optionName, $offset);
            }


        } while ($offset < $total);

        delete_option($optionName);

    }

    public function syncTags(string $apiKey): void
    {
    }

    public function pushContacts(string $apiKey, array $data): mixed
    {

        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];

        error_log('cotact' . json_encode($contact));


        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            // Insert the contact
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                 FROM $list_sync_table
                 WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)

                    )
                );
            }

            $url = 'https://api.brevo.com/v3/contacts';

            $body = json_encode([
                'email' => $contact->email,
                'attributes' => [
                    'FIRSTNAME' => $contact->first_name,
                    'LASTNAME' => $contact->last_name,
                ],
                'listIds' => array_map('intval', $remoteLists), // Convert list IDs to integers
                'updateEnabled' => true, // To update the contact if already exists
            ]);

            $response = wp_remote_post($url, [
                'method' => 'POST',
                'headers' => [
                    'api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60, // Set a timeout for the request
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR', // You can set this based on your error
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            // Successfully added/updated the contact in Brevo
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['code']) && $data['code'] != 200) {
                // Handle API error response if any
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $data['code'], // You can set this based on your error
                    'error_message' => $data['message'],
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data['id'])) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $data['id'],
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }


            return $data;
        }

        return null;
    }

    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS); // Adjust to your actual constant
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $folderId = $this->getFolder($apiKey, 'MailerPress');

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            $response = wp_remote_post('https://api.brevo.com/v3/contacts/lists', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'api-key' => $apiKey,
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                    'folderId' => (int)$folderId,
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error('brevo_request_failed', 'Request to Brevo failed.', $response->get_error_message());
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('brevo_api_error', 'Brevo API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");


        $remoteListId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_list_id FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d",
                $listId,
                $active_provider_id
            )
        );

        if ($remoteListId) {
            wp_remote_request("https://api.brevo.com/v3/contacts/lists/{$remoteListId}", [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);
        }

    }

    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    private function getFolder(string $apiKey, string $folder_name)
    {
//        $option_key = 'mailerpress_brevo_folder_id';
//        $folder_id = get_option($option_key);
//
//        if ($folder_id) {
//            return $folder_id;
//        }

        // Try to find the folder via Brevo API
        $response = wp_remote_get('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Failed to fetch folders.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200 || empty($body['folders'])) {
            return new WP_Error('brevo_api_error', 'Error fetching folders list.', $body);
        }


        // Check if folder already exists
        foreach ($body['folders'] as $folder) {
            if (strcasecmp($folder['name'], $folder_name) === 0) {
//                update_option($option_key, $folder['id']);
                return $folder['id'];
            }
        }

        // Folder doesn't exist, so create it
        $response = wp_remote_post('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'body' => wp_json_encode([
                'name' => $folder_name,
            ]),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Folder creation failed.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 201) {
            return new WP_Error('brevo_api_error', 'Brevo API error.', $body);
        }

//        update_option($option_key, $body['id']);

        return $body['id'];
    }


    public function deleteAllLists(string $apiKey): void
    {
        $folder_id = get_option('mailerpress_brevo_folder_id');

        if (!$folder_id) {
            return;
        }

        $response = wp_remote_request("https://api.brevo.com/v3/contacts/folders/{$folder_id}", [
            'method' => 'DELETE',
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            error_log("MailerPress: Failed to delete Brevo folder #{$folder_id} - " . $response->get_error_message());
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204) {
            // Folder deleted successfully
            delete_option('mailerpress_brevo_folder_id');
        } else {
            $body = wp_remote_retrieve_body($response);
            error_log("MailerPress: Failed to delete Brevo folder #{$folder_id} - {$body}");
        }
    }

    public function deleteContact(string $apiKey, array $data): void
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return;
        }

        $contact = $data['contact'];

        if (empty($contact->email)) {
            return;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);

        $active_provider_id = $wpdb->get_var(
            "SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1"
        );

        $remoteContactId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_contact_id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d",
                $contact->contact_id,
                $active_provider_id
            )
        );

        if ($remoteContactId) {
            $response = wp_remote_request("https://api.brevo.com/v3/contacts/" . urlencode($contact->email), [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code === 204) {
                // Delete local sync entry after successful API deletion
                $wpdb->delete(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => $active_provider_id,
                    ],
                    ['%d', '%d']
                );
            } else {
                $response_body = wp_remote_retrieve_body($response);
                $error = json_decode($response_body, true);
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $error['code'] ?? 'UNKNOWN',
                    'error_message' => $error['message'] ?? 'Unknown error',
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
            }
        }
    }

}
