<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use WP_Error;

class AmazonSES extends AbstractEmailService
{
    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): bool
    {
        $response = wp_remote_post('https://api.brevo.com/v3/smtp/email', [
            'headers' => [
                'Content-Type' => 'application/json',
                'api-key' => $emailData['apiKey'],
            ],
            'body' => wp_json_encode([
                'sender' => [
                    'name' => $emailData['sender_name'],
                    'email' => $emailData['sender_to'],
                ],
                'to' => [['email' => $emailData['to']]],
                'subject' => $emailData['subject'],
                'htmlContent' => $emailData['body'],
            ]),
        ]);

        return !is_wp_error($response) && 201 === wp_remote_retrieve_response_code($response);
    }

    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'amazonses',
            'name' => __('Amazon SES', 'mailerpress-pro'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="4 3 92 92" fill="none"><g filter="url(#filter0_d_3582_119696)"><rect width="92" height="92" x="4" y="3" fill="#F1F1F1" rx="46"></rect><g clip-path="url(#clip0_3582_119696)"><path fill="#252F3E" d="M34.324 45.688c0 .874.094 1.583.258 2.103.188.52.423 1.087.751 1.7.117.19.164.379.164.544 0 .236-.14.473-.445.709l-1.479.992c-.211.142-.422.213-.61.213-.235 0-.47-.118-.704-.33a7.31 7.31 0 0 1-.845-1.111c-.235-.402-.47-.85-.727-1.394-1.83 2.173-4.13 3.26-6.9 3.26-1.971 0-3.543-.567-4.693-1.7-1.15-1.135-1.737-2.647-1.737-4.537 0-2.008.704-3.639 2.136-4.867 1.431-1.229 3.332-1.843 5.75-1.843.797 0 1.618.07 2.487.189.868.118 1.76.307 2.698.52V38.41c0-1.796-.375-3.048-1.102-3.78-.752-.733-2.019-1.087-3.826-1.087-.821 0-1.666.095-2.534.307a18.6 18.6 0 0 0-2.535.803 6.69 6.69 0 0 1-.821.307c-.164.048-.282.071-.375.071-.33 0-.493-.236-.493-.732v-1.158c0-.378.047-.661.164-.827.117-.165.329-.33.657-.496.821-.425 1.807-.78 2.957-1.063 1.15-.307 2.37-.449 3.66-.449 2.794 0 4.835.638 6.15 1.914 1.29 1.276 1.947 3.213 1.947 5.812v7.655h.047Zm-9.528 3.591c.775 0 1.573-.142 2.417-.425.845-.284 1.596-.803 2.23-1.512.375-.45.657-.945.798-1.512.14-.567.234-1.252.234-2.056v-.992a19.479 19.479 0 0 0-2.159-.402 17.568 17.568 0 0 0-2.206-.142c-1.572 0-2.722.308-3.496.946-.775.637-1.15 1.535-1.15 2.717 0 1.11.282 1.937.868 2.504.563.59 1.385.874 2.464.874Zm18.845 2.552c-.423 0-.704-.071-.892-.237-.188-.141-.352-.472-.493-.92L36.741 32.41c-.14-.473-.211-.78-.211-.945 0-.378.188-.59.563-.59h2.3c.446 0 .751.07.915.235.188.142.329.473.47.922l3.942 15.64 3.661-15.64c.117-.473.258-.78.446-.922.188-.141.516-.236.939-.236h1.877c.446 0 .751.071.939.236.188.142.352.473.446.922l3.708 15.83 4.06-15.83c.14-.473.304-.78.469-.922.188-.141.493-.236.915-.236h2.182c.376 0 .587.19.587.59 0 .119-.023.237-.047.379-.023.142-.07.33-.164.59l-5.656 18.264c-.14.472-.305.78-.492.921-.188.142-.493.236-.892.236h-2.019c-.445 0-.75-.07-.938-.236-.188-.165-.352-.472-.446-.945l-3.638-15.239-3.614 15.215c-.117.473-.258.78-.445.945-.188.166-.517.237-.94.237h-2.017Zm30.155.638c-1.22 0-2.44-.142-3.614-.426-1.173-.283-2.088-.59-2.698-.945-.376-.212-.634-.449-.728-.661a1.678 1.678 0 0 1-.14-.662V48.57c0-.496.187-.732.54-.732.14 0 .28.023.421.07.141.048.353.142.587.237.798.354 1.666.638 2.582.827.938.189 1.854.283 2.792.283 1.479 0 2.629-.26 3.427-.78.797-.519 1.22-1.275 1.22-2.244 0-.661-.211-1.205-.634-1.654-.422-.448-1.22-.85-2.37-1.228l-3.403-1.063c-1.713-.544-2.98-1.347-3.755-2.41-.774-1.04-1.173-2.197-1.173-3.426 0-.992.211-1.867.634-2.623a6.064 6.064 0 0 1 1.69-1.937c.703-.543 1.501-.945 2.44-1.229.939-.283 1.924-.401 2.957-.401.516 0 1.056.023 1.572.094.54.071 1.033.166 1.525.26.47.118.916.236 1.338.378.423.142.751.284.986.426.328.189.563.378.704.59.14.19.211.45.211.78v1.11c0 .496-.188.756-.54.756-.187 0-.493-.094-.892-.283-1.337-.614-2.839-.922-4.505-.922-1.338 0-2.394.213-3.121.662-.728.449-1.103 1.134-1.103 2.103 0 .661.234 1.228.704 1.677.469.45 1.337.898 2.581 1.3l3.332 1.063c1.69.543 2.91 1.3 3.638 2.268.727.969 1.08 2.08 1.08 3.308 0 1.016-.212 1.937-.61 2.74a6.344 6.344 0 0 1-1.714 2.08c-.727.59-1.596 1.015-2.605 1.322-1.056.331-2.159.497-3.356.497Z"></path><path fill="#F90" fill-rule="evenodd" d="M78.232 63.951c-7.721 5.741-18.938 8.79-28.584 8.79-13.517 0-25.697-5.033-34.896-13.397-.727-.662-.07-1.56.798-1.04 9.95 5.813 22.224 9.333 34.92 9.333 8.565 0 17.976-1.796 26.635-5.481 1.291-.591 2.394.85 1.127 1.795Z" clip-rule="evenodd"></path><path fill="#F90" fill-rule="evenodd" d="M81.447 60.265c-.986-1.275-6.524-.614-9.035-.307-.751.095-.869-.567-.188-1.063 4.412-3.119 11.663-2.22 12.508-1.181.845 1.063-.234 8.364-4.365 11.86-.633.544-1.243.26-.962-.449.939-2.339 3.027-7.607 2.042-8.86Z" clip-rule="evenodd"></path></g></g><defs><clipPath id="clip0_3582_119696"><path fill="#fff" d="M14.048 30H85.39v43H14.05z"></path></clipPath><filter id="filter0_d_3582_119696" width="100" height="100" x="0" y="0" color-interpolation-filters="sRGB" filterUnits="userSpaceOnUse"><feFlood flood-opacity="0" result="BackgroundImageFix"></feFlood><feColorMatrix in="SourceAlpha" result="hardAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"></feColorMatrix><feOffset dy="1"></feOffset><feGaussianBlur stdDeviation="2"></feGaussianBlur><feColorMatrix values="0 0 0 0 0.0687866 0 0 0 0 0.097585 0 0 0 0 0.37981 0 0 0 0.0779552 0"></feColorMatrix><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_3582_119696"></feBlend><feBlend in="SourceGraphic" in2="effect1_dropShadow_3582_119696" result="shape"></feBlend></filter></defs></svg>',
            'link' => 'https://aws.amazon.com/ses/',
            'createAccountLink' => 'https://onboarding.brevo.com/account/register',
            'linkApiKey' => 'https://app.brevo.com/settings/keys/api',
            'description' => __('Amazon SES offers a reliable and cost-effective service for sending and receiving emails using your own domain. It leverages Amazon’s robust infrastructure, making it a powerful option for managing your email communication.', 'mailerpress'),
            'recommended' => false,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ]
        ];
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_offset', $this->config()['key']);
        $limit = 100;
        $offset = 0;

        $url = sprintf(
            "https://api.brevo.com/v3/contacts?limit=%d&offset=%d",
            $limit,
            $offset
        );

        $response = wp_remote_get($url, [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            error_log('Brevo API request failed: ' . $response->get_error_message());
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            error_log("Brevo API returned status code {$status_code}");
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['contacts'])) {
            foreach ($data['contacts'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'],
                    'email' => $contact['email'],
                    'remote_lists' => $contact['listIds'],
                    'attributes' => [
                        'first_name' => $contact['attributes']['PRENOM'],
                        'last_name' => $contact['attributes']['NOM'],
                    ],
                ]);
            }

            // Update the offset
            $newOffset = $offset + count($data['contacts']);
            update_option($optionName, $newOffset);

            if (count($data['contacts']) === $limit) {
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            }
        }
    }

    public function syncLists(string $apiKey): void
    {
        $optionName = sprintf('%s_lists_last_offset', $this->config()['key']);
        $limit = 20;
        $offset = (int)get_option($optionName, 0);
        $total = null;

        do {
            $url = sprintf(
                "https://api.brevo.com/v3/contacts/lists?limit=%d&offset=%d",
                $limit,
                $offset
            );

            $response = wp_remote_get($url, [
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                ]
            ]);

            if (is_wp_error($response)) {
                error_log('Brevo API request failed: ' . $response->get_error_message());
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                error_log("Brevo API returned status code {$status_code}");
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if ($total === null && isset($data['count'])) {
                $total = (int)$data['count'];
            }

            $lists = $data['lists'] ?? [];
            if (!empty($lists)) {
                foreach ($lists as $list) {
                    // Do something with the list
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }

                // Update the stored offset after each batch
                $offset += $limit;
                update_option($optionName, $offset);
            }


        } while ($offset < $total);

        delete_option($optionName);

    }

    public function syncTags(string $apiKey): void
    {
    }

    public function pushContacts(string $apiKey, array $data): mixed
    {

        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];

        error_log('cotact' . json_encode($contact));


        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            // Insert the contact
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                 FROM $list_sync_table
                 WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)

                    )
                );
            }

            $url = 'https://api.brevo.com/v3/contacts';

            $body = json_encode([
                'email' => $contact->email,
                'attributes' => [
                    'FIRSTNAME' => $contact->first_name,
                    'LASTNAME' => $contact->last_name,
                ],
                'listIds' => array_map('intval', $remoteLists), // Convert list IDs to integers
                'updateEnabled' => true, // To update the contact if already exists
            ]);

            $response = wp_remote_post($url, [
                'method' => 'POST',
                'headers' => [
                    'api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60, // Set a timeout for the request
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR', // You can set this based on your error
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            // Successfully added/updated the contact in Brevo
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['code']) && $data['code'] != 200) {
                // Handle API error response if any
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $data['code'], // You can set this based on your error
                    'error_message' => $data['message'],
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data['id'])) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $data['id'],
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }


            return $data;
        }

        return null;
    }

    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS); // Adjust to your actual constant
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $folderId = $this->getFolder($apiKey, 'MailerPress');

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            $response = wp_remote_post('https://api.brevo.com/v3/contacts/lists', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'api-key' => $apiKey,
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                    'folderId' => (int)$folderId,
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error('brevo_request_failed', 'Request to Brevo failed.', $response->get_error_message());
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('brevo_api_error', 'Brevo API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");


        $remoteListId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_list_id FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d",
                $listId,
                $active_provider_id
            )
        );

        if ($remoteListId) {
            wp_remote_request("https://api.brevo.com/v3/contacts/lists/{$remoteListId}", [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);
        }

    }

    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    private function getFolder(string $apiKey, string $folder_name)
    {
//        $option_key = 'mailerpress_brevo_folder_id';
//        $folder_id = get_option($option_key);
//
//        if ($folder_id) {
//            return $folder_id;
//        }

        // Try to find the folder via Brevo API
        $response = wp_remote_get('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Failed to fetch folders.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200 || empty($body['folders'])) {
            return new WP_Error('brevo_api_error', 'Error fetching folders list.', $body);
        }


        // Check if folder already exists
        foreach ($body['folders'] as $folder) {
            if (strcasecmp($folder['name'], $folder_name) === 0) {
//                update_option($option_key, $folder['id']);
                return $folder['id'];
            }
        }

        // Folder doesn't exist, so create it
        $response = wp_remote_post('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'body' => wp_json_encode([
                'name' => $folder_name,
            ]),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Folder creation failed.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 201) {
            return new WP_Error('brevo_api_error', 'Brevo API error.', $body);
        }

//        update_option($option_key, $body['id']);

        return $body['id'];
    }


    public function deleteAllLists(string $apiKey): void
    {
        $folder_id = get_option('mailerpress_brevo_folder_id');

        if (!$folder_id) {
            return;
        }

        $response = wp_remote_request("https://api.brevo.com/v3/contacts/folders/{$folder_id}", [
            'method' => 'DELETE',
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            error_log("MailerPress: Failed to delete Brevo folder #{$folder_id} - " . $response->get_error_message());
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204) {
            // Folder deleted successfully
            delete_option('mailerpress_brevo_folder_id');
        } else {
            $body = wp_remote_retrieve_body($response);
            error_log("MailerPress: Failed to delete Brevo folder #{$folder_id} - {$body}");
        }
    }

    public function deleteContact(string $apiKey, array $data): void
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return;
        }

        $contact = $data['contact'];

        if (empty($contact->email)) {
            return;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);

        $active_provider_id = $wpdb->get_var(
            "SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1"
        );

        $remoteContactId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_contact_id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d",
                $contact->contact_id,
                $active_provider_id
            )
        );

        if ($remoteContactId) {
            $response = wp_remote_request("https://api.brevo.com/v3/contacts/" . urlencode($contact->email), [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code === 204) {
                // Delete local sync entry after successful API deletion
                $wpdb->delete(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => $active_provider_id,
                    ],
                    ['%d', '%d']
                );
            } else {
                $response_body = wp_remote_retrieve_body($response);
                $error = json_decode($response_body, true);
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $error['code'] ?? 'UNKNOWN',
                    'error_message' => $error['message'] ?? 'Unknown error',
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
            }
        }
    }

}
