<?php

namespace MailerPressPro\Automation\Steps;

use MailerPressPro\Automation\AutomationActionRegistry;
use MailerPressPro\Automation\Utils\AutomationRepository;

class AutomationStepExecutor
{
    protected $repository;

    public function __construct()
    {
        $this->repository = new AutomationRepository();
    }

    /**
     * @throws \Exception
     */
    public function handle($step, $contact_id, $automation_id)
    {
        $settings = json_decode($step->settings, true);
        switch ($step->type) {
            case 'action':
                return $this->runAction($settings, $contact_id);
            case 'delay':
                return $this->scheduleDelay($step, $contact_id, $automation_id, $settings);
            case 'condition':
                return $this->evaluateCondition($step, $contact_id);
        }
    }

    protected function runAction($settings, $contact_id)
    {
        if (isset($settings['action'])) {

            $action_type = $settings['action']; // e.g., "send_email"


            $handler = AutomationActionRegistry::getHandler($action_type);

            if ($handler && is_callable($handler)) {
                $handler($settings, $contact_id);
            } else {
                throw new \Exception("No handler for action: $action_type");
            }

            return true;
        }

        return false;
    }

    protected function scheduleDelay($step, $contact_id, $automation_id, $settings)
    {
        $delay = $settings['delay']; // e.g. "2 days"
        $run_at = strtotime("+$delay");

        $this->repository->queueDelayedStep($automation_id, $contact_id, $step->step_id, $run_at);

        // Schedule using Action Scheduler
        as_schedule_single_action($run_at, 'mailerpress_run_delayed_step', [
            'automation_id' => $automation_id,
            'contact_id' => $contact_id,
            'step_id' => $step->step_id
        ]);

        return 'wait';
    }

    protected function evaluateCondition($step, $contact_id)
    {
        // Example: check if contact has tag
        $condition = json_decode($step->settings, true);
        $result = apply_filters('mailerpress/automation/condition_result', true, $condition, $contact_id);

        return $result ? $this->repository->getNextStepForBranch($step->step_id, 'yes')
            : $this->repository->getNextStepForBranch($step->step_id, 'no');
    }
}