<?php

namespace MailerPressPro\Api;

use Exception;
use MailerPressPro\Core\Attributes\Endpoint;
use MailerPressPro\Core\ImageGeneratorFactory;

class ImageGeneration
{
    #[Endpoint('generate-image', methods: 'POST')]
    public function run(\WP_REST_Request $request): \WP_Error|\WP_REST_Response|\WP_HTTP_Response
    {
        $prompt = sanitize_text_field($request->get_param('prompt'));
        $number = intval($request->get_param('number'));
        $size = sanitize_text_field($request->get_param('size'));
        $config = [];
        // Get provider settings
        $provider = get_option('mailerpress_ai_model_settings');
        if (is_string($provider)) {
            $provider = json_decode($provider, true);
        }

        if (
            false === $provider ||
            !isset($provider['image_ai'])
        ) {
            return new \WP_Error('generator_error', __('Missing image AI provider configuration', 'mailerpress'), ['status' => 500]);
        }

        $providerName = $provider['image_ai']['provider'] ?? '';

        if ($providerName === 'gemini') {
            $config['apiKey'] = $provider['api_keys']['gemini'];
            $config['model'] = '';
        } else {
            $config['apiKey'] = $provider['api_keys']['openai'];
            $config['model'] = $provider['image_ai']['model'];
        }

        try {
            $generator = ImageGeneratorFactory::make($providerName, $config);

            $images = $generator->generate($prompt, $number, $size);

            if (is_wp_error($images)) {
                return $images;
            }

            return rest_ensure_response($images);

        } catch (\Exception $e) {
            return new \WP_Error('generator_error', $e->getMessage(), ['status' => 500]);
        }
    }

    #[Endpoint('generate-variation', methods: 'POST')]
    public function variation(\WP_REST_Request $request): \WP_Error|\WP_REST_Response|\WP_HTTP_Response
    {
        $prompt = sanitize_text_field($request->get_param('prompt'));
        $image_url = $request->get_param('image_url');
        $number = intval($request->get_param('number', 1));
        $size = sanitize_text_field($request->get_param('size', '512x512'));


        // Get provider configuration
        $provider = get_option('mailerpress_ai_model_settings');
        if (is_string($provider)) {
            $provider = json_decode($provider, true);
        }

        if (
            false === $provider ||
            !isset($provider['image_ai'])
        ) {
            return new \WP_Error('generator_error', __('Missing image_ai provider configuration', 'mailerpress'), ['status' => 500]);
        }

        $providerName = $provider['image_ai']['provider'] ?? '';
        $config = [];

        if ($providerName === 'gemini') {
            $config['apiKey'] = $provider['api_keys']['gemini'];
            $config['model'] = '';
        } else {
            $config['apiKey'] = $provider['api_keys']['openai'];
            $config['model'] = $provider['image_ai']['model'];
        }

        try {
            $generator = ImageGeneratorFactory::make($providerName, $config);

            if (!method_exists($generator, 'variation')) {
                return new \WP_Error('not_supported', __('This provider does not support image variations.', 'mailerpress'), ['status' => 400]);
            }

            $images = $generator->variation($prompt, $image_url, $number, $size);

            if (is_wp_error($images)) {
                return $images;
            }

            return rest_ensure_response($images);

        } catch (\Exception $e) {
            return new \WP_Error('variation_error', $e->getMessage(), ['status' => 500]);
        }
    }


}