<?php

namespace MailerPressPro\Actions;

use MailerPressPro\Core\Attributes\Action;

class WhiteLabel
{
    #[Action('init')]
    public function init(): void
    {
        $options = $this->getOptions();

        if (empty($options) || empty($options['white_label_active'])) {
            return;
        }

        $this->replaceAdminMenu();
        $this->replacePluginRowMeta();
        $this->renamePluginsDisplay();
        $this->replaceUpdateCore();
    }

    private function getOptions(): array
    {
        // Use filter to allow external override
        $options = apply_filters('mailerpress_white_label_options', []);

        // Ensure array type
        if (is_string($options)) {
            $options = json_decode($options, true) ?: [];
        }

        return $options;
    }

    private function replaceAdminMenu(): void
    {
        add_action('admin_menu', function () {
            $options = $this->getOptions();
            if (empty($options)) {
                return;
            }

            global $menu;
            $menuTitle = $options['admin_menu_title'] ?? '';
            if ($menuTitle === '') {
                return;
            }

            foreach ($menu as &$item) {
                if (in_array($item[2], ['mailerpress', 'mailerpress-pro'])) {
                    $item[0] = $menuTitle;
                }
            }
        });
    }

    private function replacePluginRowMeta(): void
    {
        add_filter('plugin_row_meta', function ($links, $file) {
            $options = $this->getOptions();
            if (empty($options)) {
                return $links;
            }

            if (strpos($file, 'mailerpress') !== false) {
                return []; // Remove all meta links
            }

            return $links;
        }, 10, 2);
    }

    private function renamePluginsDisplay(): void
    {
        add_filter('all_plugins', function ($all_plugins) {
            $options = $this->getOptions();
            if (empty($options)) {
                return $all_plugins;
            }

            foreach ($all_plugins as $path => $plugin) {
                if (strpos($path, 'mailerpress-pro') !== false) {
                    $all_plugins[$path]['Name'] = $options['pro_plugin_name'] ?? $plugin['Name'];
                    $all_plugins[$path]['Author'] = $options['author_name'] ?? $plugin['Author'];
                    $all_plugins[$path]['Description'] = $options['pro_plugin_description'] ?? $plugin['Description'];
                    $all_plugins[$path]['Icon'] = '';
                    $all_plugins[$path]['Icons'] = [];
                } elseif (strpos($path, 'mailerpress') !== false) {
                    $all_plugins[$path]['Name'] = $options['free_plugin_name'] ?? $plugin['Name'];
                    $all_plugins[$path]['Author'] = $options['author_name'] ?? $plugin['Author'];
                    $all_plugins[$path]['Description'] = $options['free_plugin_description'] ?? $plugin['Description'];
                    $all_plugins[$path]['Icon'] = '';
                    $all_plugins[$path]['Icons'] = [];
                }
            }

            return $all_plugins;
        });
    }

    private function replaceUpdateCore(): void
    {
        add_filter('gettext', function ($translated, $text, $domain) {
            $options = $this->getOptions();
            if (empty($options)) {
                return $translated;
            }

            if (strpos($text, 'MailerPress Pro') !== false && !empty($options['pro_plugin_name'])) {
                $translated = str_replace('MailerPress Pro', $options['pro_plugin_name'], $translated);
            }

            if (strpos($text, 'MailerPress') !== false && !empty($options['free_plugin_name'])) {
                $translated = str_replace('MailerPress', $options['free_plugin_name'], $translated);
            }

            return $translated;
        }, 10, 3);

        add_filter('pre_set_site_transient_update_plugins', function ($transient) {
            $options = $this->getOptions();
            if (empty($options)) {
                return $transient;
            }

            foreach ($transient->response ?? [] as $file => $plugin) {
                if (strpos($file, 'mailerpress-pro') !== false) {
                    $plugin->slug = sanitize_title($options['pro_plugin_name'] ?? $plugin->slug);
                    $plugin->Description = $options['pro_plugin_description'] ?? $plugin->Description;
                    $plugin->icons = [];
                }

                if (strpos($file, 'mailerpress') !== false) {
                    $plugin->slug = sanitize_title($options['free_plugin_name'] ?? $plugin->slug);
                    $plugin->Description = $options['free_plugin_description'] ?? $plugin->Description;
                    $plugin->icons = [];
                }
            }

            return $transient;
        });
    }
}
