<?php

use MailerPress\Models\CustomFields;

if (!defined('ABSPATH')) {
    exit;
}

GFForms::include_feed_addon_framework();

class MailerPressGfAddon extends GFFeedAddOn
{
    private static $_instance = null;
    protected $_version = '0.0.1';
    protected $_min_gravityforms_version = '1.9.12';
    protected $_slug = 'mailerpress';
    protected $_path = 'mailerpress/mailerpress.php';
    protected $_full_path = __FILE__;
    protected $_url = 'http://www.gravityforms.com';
    protected $_title = 'Gravity Forms MailerPress Add-On';
    protected $_short_title = 'MailerPress';
    protected $_enable_rg_autoupgrade = true;
    protected $_capabilities = array('gravityforms_mailerpress', 'gravityforms_mailerpress_uninstall');
    protected $_capabilities_settings_page = 'gravityforms_mailerpress';
    protected $_capabilities_form_settings = 'gravityforms_mailerpress';
    protected $_capabilities_uninstall = 'gravityforms_mailerpress_uninstall';

    public static function get_instance()
    {

        if (null === self::$_instance) {
            self::$_instance = new self;
        }

        return self::$_instance;
    }

    public function feed_settings_fields()
    {
        $settings = array(
            array(
                'title' => esc_html__('MailerPress Feed Settings', 'mailerpress-pro'),
                'fields' => array(
                    array(
                        'name' => 'feed_name',
                        'label' => esc_html__('Name', 'mailerpress-pro'),
                        'type' => 'text',
                        'required' => true,
                        'class' => 'medium',
                        'tooltip' => sprintf(
                            '<h6>%s</h6>%s',
                            esc_html__('Name', 'mailerpress-pro'),
                            esc_html__('Enter a feed name to uniquely identify this setup.', 'mailerpress-pro')
                        ),
                    ),
                    array(
                        'name' => 'mailerpress_list',
                        'label' => esc_html__('MailerPress List', 'mailerpress-pro'),
                        'type' => 'select',
                        'choices' => $this->getMailerpressList(),
                        'required' => true,
                        'default_value' => '',
                        'tooltip' => sprintf(
                            '<h6>%s</h6>%s',
                            esc_html__('MailerPress list', 'mailerpress-pro'),
                            esc_html__(
                                'Select the MailerPress list you would like to add your contacts to.',
                                'mailerpress-pro'
                            )
                        ),
                    ),
                    array(
                        'name' => 'mailerpress_tag',
                        'label' => esc_html__('MailerPress tag', 'mailerpress-pro'),
                        'type' => 'select',
                        'choices' => $this->getMailerpressTags(),
                        'required' => false,
                        'tooltip' => sprintf(
                            '<h6>%s</h6>%s',
                            esc_html__('MailerPress List', 'mailerpress-pro'),
                            esc_html__(
                                'Select the MailerPress tag you would like to add your contacts to.',
                                'mailerpress-pro'
                            )
                        ),
                    ),
                    array(
                        'name' => 'options',
                        'label' => esc_html__('Options', 'mailerpress-pro'),
                        'type' => 'checkbox',
                        'choices' => array(
                            array(
                                'name' => 'double_optin',
                                'label' => esc_html__('Double Opt-In', 'mailerpress-pro'),
                                'default_value' => '1',
                                'value' => '1',
                                'onclick' => 'if(this.checked){jQuery("#mailerpress_doubleoptin_warning").hide();} else{jQuery("#mailerpress_doubleoptin_warning").show();}',
                                'tooltip' => sprintf(
                                    '<h6>%s</h6>%s',
                                    esc_html__('Double Opt-In', 'mailerpress-pro'),
                                    esc_html__(
                                        'When the double opt-in option is enabled, MailerPress will send a confirmation email to the user and will only add them to your MailerPress list upon confirmation.',
                                        'mailerpress-pro'
                                    )
                                ),
                            ),
                        ),
                    ),

                ),
            ),
            array(
                'dependency' => 'mailerpress_list',
                'fields' => array(
                    array(
                        'name' => 'mappedFields',
                        'label' => esc_html__('Map contact fields', 'mailerpress-pro'),
                        'type' => 'field_map',
                        'field_map' => $this->merge_vars_field_map(),
                        'tooltip' => sprintf(
                            '<h6>%s</h6>%s',
                            esc_html__('Map Fields', 'mailerpress-pro'),
                            esc_html__(
                                'Associate your MailerPress contact tags to the appropriate Gravity Form fields by selecting the appropriate form field from the list.',
                                'mailerpress-pro'
                            )
                        ),
                    ),
                    array(
                        'name' => 'optinCondition',
                        'label' => esc_html__('Conditional Logic', 'gravityformsmailerpress'),
                        'type' => 'feed_condition',
                        'tooltip' => sprintf(
                            '<h6>%s</h6>%s',
                            esc_html__('Conditional Logic', 'gravityformsmailerpress'),
                            esc_html__(
                                'When conditional logic is enabled, form submissions will only be exported to MailerPress when the conditions are met. When disabled all form submissions will be exported.',
                                'gravityformsmailerpress'
                            )
                        ),
                    ),
                ),
            )
        );

        return $settings;
    }

    public function get_column_value_mailerpress_list_name($feed)
    {
        $savedListId = rgar($feed['meta'], 'mailerpress_list');

        // Si aucune liste n'est sauvegardée, retourner un message
        if (empty($savedListId)) {
            return __('(No list selected)', 'mailerpress-pro');
        }

        $lists = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Lists::class)->getLists();
        foreach ($lists as $list) {
            // Comparaison flexible pour gérer les types string/int
            if ((string)$list['list_id'] === (string)$savedListId) {
                return $list['name'];
            }
        }
        return __('(List not found)', 'mailerpress-pro');
    }

    public function get_column_value_mailerpress_list_tag($feed)
    {
        $savedTagId = rgar($feed['meta'], 'mailerpress_tag');

        // Si aucun tag n'est sauvegardé, retourner un message
        if (empty($savedTagId)) {
            return __('(No tag selected)', 'mailerpress-pro');
        }

        $tags = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Tags::class)->getAll();
        foreach ($tags as $tag) {
            // Comparaison flexible pour gérer les types string/int
            if ((string)$tag->tag_id === (string)$savedTagId) {
                return $tag->name;
            }
        }
        return __('(Tag not found)', 'mailerpress-pro');
    }


    public function get_column_value_feed_name($feed)
    {
        return rgar($feed['meta'], 'feed_name') ?: __('(No Name)', 'mailerpress-pro');
    }

    public function feed_list_columns(): array
    {
        return array(
            'feed_name' => esc_html__('Name', 'mailerpress-pro'),
            'mailerpress_list_name' => esc_html__('MailerPress List', 'mailerpress-pro'),
            'mailerpress_list_tag' => esc_html__('MailerPress Tag', 'mailerpress-pro'),
        );
    }

    /**
     * Valide les paramètres du feed avant la sauvegarde
     */
    public function is_valid_feed($feed)
    {
        $feed_meta = $feed['meta'] ?? [];

        // Vérifier que la liste est sélectionnée
        $list = rgar($feed_meta, 'mailerpress_list', '');
        if (empty($list)) {
            $this->set_field_error('mailerpress_list', esc_html__('Please select a MailerPress list.', 'mailerpress-pro'));
            return false;
        }

        // Vérifier que le nom du feed est défini
        if (empty($feed_meta['feed_name'])) {
            $this->set_field_error('feed_name', esc_html__('Please enter a feed name.', 'mailerpress-pro'));
            return false;
        }

        return true;
    }


    public function get_mailerpress_custom_fields(): array
    {
        $customFieldsModel = \MailerPress\Core\Kernel::getContainer()->get(CustomFields::class);
        $fields = $customFieldsModel->all(); // returns array of objects with field_key, label, type

        $mappedFields = [];
        foreach ($fields as $field) {
            $mappedFields[$field->field_key] = [
                'name' => $field->field_key,
                'label' => $field->label,
                'required' => $field->required,
                'field_type' => ['text', 'email', 'name', 'hidden', 'textarea', 'select', 'checkbox', 'radio', 'phone', 'number', 'address', 'date'],
            ];
        }

        return $mappedFields;
    }


    public function merge_vars_field_map(): array
    {

        // Basic fields
        $field_map = array(
            'email' => array(
                'name' => 'email',
                'label' => esc_html__('Email Address', 'mailerpress-pro'),
                'required' => true,
                'field_type' => array('email', 'hidden'),
            ),
            'contactFirstName' => array(
                'name' => 'contactFirstName',
                'label' => esc_html__('First name', 'mailerpress-pro'),
                'required' => false,
                'field_type' => array('name', 'hidden'),
            ),
            'contactLastName' => array(
                'name' => 'contactLastName',
                'label' => esc_html__('Last name', 'mailerpress-pro'),
                'required' => false,
                'field_type' => array('name', 'hidden'),
            ),
        );

        // Add custom fields dynamically
        $customFields = $this->get_mailerpress_custom_fields();
        foreach ($customFields as $key => $field) {
            $field_map[$key] = $field;
        }

        return $field_map;
    }

    public function getMailerpressTags(): array
    {
        $tags = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Tags::class)->getAll();

        $formattedTags = array_map(function ($tag) {
            return [
                'label' => $tag->name,
                'value' => $tag->tag_id
            ];
        }, $tags);

        // Add default empty value
        array_unshift($formattedTags, [
            'label' => __('-- Select a tag --', 'mailerpress-pro'),
            'value' => ''
        ]);

        return $formattedTags;
    }

    public function getMailerpressList(): array
    {
        $lists = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Lists::class)->getLists();

        $formattedLists = array_map(function ($list) {
            // S'assurer que list_id est converti en string pour la compatibilité avec Gravity Forms
            return [
                'label' => $list['name'],
                'value' => (string)$list['list_id']
            ];
        }, $lists);

        // Add default empty value
        array_unshift($formattedLists, [
            'label' => __('-- Select a list --', 'mailerpress-pro'),
            'value' => ''
        ]);

        return $formattedLists;
    }

    /**
     * @throws \DI\DependencyException
     * @throws \DI\NotFoundException
     */
    public function process_feed(
        $feed,
        $entry,
        $form
    ) {
        $subscriptionOption = get_option('mailerpress_signup_confirmation', null);
        if (null !== $subscriptionOption) {
            $subscriptionOption = json_decode($subscriptionOption, true);
            // Vérifier que json_decode a retourné un tableau valide
            if (!is_array($subscriptionOption)) {
                $subscriptionOption = null;
            }
        }
        $contact_data = [];
        $field_map = $this->get_field_map_fields($feed, 'mappedFields');
        $email = $this->get_field_value($form, $entry, $field_map['email']);

        // Récupérer la liste avec rgar et s'assurer qu'elle n'est pas vide
        $list = rgar($feed['meta'], 'mailerpress_list', '');
        // Convertir en string si c'est un nombre pour la compatibilité
        $list = $list !== '' ? (string)$list : '';

        $tag = rgar($feed['meta'], 'mailerpress_tag', '');
        $tag = $tag !== '' ? (string)$tag : '';

        $confirmOption = rgar($feed['meta'], 'double_optin', '');

        // If email address is invalid, log error and return.
        if (GFCommon::is_invalid_or_empty_email($email)) {
            $this->add_feed_error(
                esc_html__('A valid Email address must be provided.', 'mailerpress-pro'),
                $feed,
                $entry,
                $form
            );
            return $entry;
        }

        // Vérifier que la liste est définie
        if (empty($list)) {
            $this->add_feed_error(
                esc_html__('MailerPress list is not configured. Please select a list in the feed settings.', 'mailerpress-pro'),
                $feed,
                $entry,
                $form
            );
            return $entry;
        }

        $contact_data['contactEmail'] = $email;
        $contact_data['lists'] = [['id' => $list]];

        // Ajouter le tag seulement s'il est défini
        if (!empty($tag)) {
            $contact_data['tags'] = [['id' => $tag]];
        }

        // Vérifier que $subscriptionOption est un tableau avant d'accéder à ses clés
        $enableSignupConfirmation = false;
        if (is_array($subscriptionOption) && isset($subscriptionOption['enableSignupConfirmation'])) {
            $enableSignupConfirmation = (bool)$subscriptionOption['enableSignupConfirmation'];
        }

        if (false === $enableSignupConfirmation) {
            $contact_data['contactStatus'] = 'subscribed';
        } else {
            $contact_data['contactStatus'] = $confirmOption === "1" ? 'pending' : 'subscribed';
        }
        $contact_data['opt_in_source'] = 'gravity_forms';
        $contact_data['optin_details'] = wp_json_encode([
            'form_id' => $form['id'],
            'form_title' => $form['title'],
            'entry_id' => $entry['id'],
        ]);

        $override_empty_fields = gf_apply_filters(
            'gform_mailerpress_override_empty_fields',
            array($form['id']),
            true,
            $form,
            $entry,
            $feed
        );

        // Log that empty fields will not be overridden.
        if (!$override_empty_fields) {
            $this->log_debug(__METHOD__ . '(): Empty fields will not be overridden.');
        }

        foreach ($field_map as $name => $field_id) {
            // If no field is mapped, skip it.
            if (rgblank($field_id)) {
                continue;
            }

            // Get field object.
            //$field = GFFormsModel::get_field($form, $field_id);
            $field_value = $this->get_field_value($form, $entry, $field_id);

            // If field value is empty and we are not overriding empty fields, skip it.
            if (empty($field_value) && (!$override_empty_fields)) {
                continue;
            }

            $contact_data[$name] = $field_value;
        }


        // --- Map custom fields dynamically ---
        $customFieldsModel = new CustomFields();
        $custom_fields_defs = $customFieldsModel->all();

        foreach ($custom_fields_defs as $field_def) {
            $mapped_field_id = $field_map[$field_def->field_key] ?? null;

            // If mapped field exists, get value from GF entry
            if ($mapped_field_id) {
                $field_value = $this->get_field_value($form, $entry, $mapped_field_id);

                if (!empty($field_value) || $override_empty_fields) {
                    // Store under contact_data['custom_fields'] keyed by field_key
                    $contact_data['custom_fields'][$field_def->field_key] = $field_value;
                }
            }
        }

        $result = add_mailerpress_contact($contact_data);

        if ($result['success']) {
            $this->add_note(
                $entry['id'],
                sprintf(
                    __('%s has been %s to MailerPress audience.', 'mailerpress-pro'),
                    $email,
                    !empty($result['update']) ? 'updated' : 'added'
                ),
                'success',
            );
        } else {
            $this->add_feed_error(
                sprintf(
                    __('Unable to %s contact inside MailerPress. (%s)', 'mailerpress-pro'),
                    $result['error'],
                    !empty($result['update']) ? 'update' : 'add'

                ),
                $feed,
                $entry,
                $form
            );
        }
    }

    public
    function feed_settings_page()
    {
        return true;
    }

    public function get_menu_icon()
    {

        return file_get_contents(\MailerPress\Core\Kernel::$config['root'] . '/build/public/images/mailerpress-logo.svg');
    }
}
