<?php

namespace MailerPressPro\Services\ThirdParty;

use FluentForm\App\Http\Controllers\IntegrationManagerController;
use FluentForm\Framework\Foundation\Application;
use FluentForm\Framework\Helpers\ArrayHelper;
use MailerPress\Models\CustomFields;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class MailerPressFluentFormIntegration extends IntegrationManagerController
{
    public $disableGlobalSettings = 'yes';

    public function __construct(Application $app)
    {
        parent::__construct(
            $app,
            'MailerPress',
            'mailerpress',
            '_fluentform_mailerpress_settings',
            'mailerpress_feed',
            37
        );

        // Use MailerPress logo from the main plugin (96x96 PNG for proper size in FluentForm)
        $this->logo = plugins_url('mailerpress/build/public/images/mailerpress-96x96.png');

        $this->description = __('Connect Fluent Forms with MailerPress to automatically add form submissions to your MailerPress audience.', 'mailerpress-pro');
        add_filter('fluentform/save_integration_value_' . $this->integrationKey, [$this, 'validate'], 10, 3);

        // Force synchronous execution (not async) - contact should be created immediately
        add_filter('fluentform/notifying_async_' . $this->integrationKey, '__return_false');

        $this->registerAdminHooks();
    }

    public function validate($settings, $integrationId, $formId)
    {
        $error = false;
        $errors = array();

        // Validate name field (required)
        if (empty($settings['name'])) {
            $error = true;
            $errors['name'] = [__('Feed Name is required', 'mailerpress-pro')];
        }

        // Validate list_id field (required)
        if (empty($settings['list_id'])) {
            $error = true;
            $errors['list_id'] = [__('MailerPress List is required', 'mailerpress-pro')];
        }

        // Validate other fields returned by getFields()
        foreach ($this->getFields() as $field) {
            if ($field['required'] && empty($settings[$field['key']])) {
                $error = true;
                $errors[$field['key']] = [__($field['label'] . ' is required', 'mailerpress-pro')];
            }
        }

        if ($error) {
            wp_send_json_error([
                'message' => __('Validation Failed', 'mailerpress-pro'),
                'errors' => $errors
            ], 423);
        }
        return $settings;
    }

    public function addGlobalMenu($setting)
    {
        return $setting;
    }

    public function pushIntegration($integrations, $formId)
    {
        $integrations[$this->integrationKey] = [
            'category' => '',
            'disable_global_settings' => 'yes',
            'logo' => $this->logo,
            'title' => $this->title . ' Integration',
            'is_active' => $this->isConfigured()
        ];

        return $integrations;
    }

    public function getIntegrationDefaults($settings, $formId)
    {
        $defaults = [
            'name' => '',
            'list_id' => '',
            'tags' => [],
            'subscription_status' => 'subscribed',
            'double_optin' => '1',
            'optin_source' => 'Fluent Form',
            'conditionals' => [
                'conditions' => [],
                'status' => false,
                'type' => 'all'
            ],
            'enabled' => true
        ];

        // Add custom fields defaults
        $customFields = $this->getCustomFields();
        foreach ($customFields as $fieldKey => $fieldLabel) {
            $defaults['custom_' . $fieldKey] = '';
        }

        return $defaults;
    }

    public function getSettingsFields($settings, $formId = null)
    {
        if (!function_exists('mailerpress_get_lists')) {
            return [];
        }

        $allFieldSettings = [
            'fields' => [
                [
                    'key' => 'name',
                    'label' => __('Feed Name', 'mailerpress-pro'),
                    'required' => true,
                    'placeholder' => __('Your Feed Name', 'mailerpress-pro'),
                    'component' => 'text'
                ],
                [
                    'key' => 'list_id',
                    'label' => __('MailerPress Lists', 'mailerpress-pro'),
                    'placeholder' => __('Select MailerPress List', 'mailerpress-pro'),
                    'required' => true,
                    'component' => 'select',
                    'tips' => __('Select MailerPress List', 'mailerpress-pro'),
                    'options' => $this->getLists()
                ],
            ],
            'integration_title' => $this->title
        ];

        $getFields = $this->getFields();

        $allFieldSettings['fields'] = array_merge($allFieldSettings['fields'], $getFields, [
            [
                'require_list' => false,
                'key' => 'conditionals',
                'label' => __('Conditional Logics', 'mailerpress-pro'),
                'tips' => __('Allow this integration conditionally based on your submission values', 'mailerpress-pro'),
                'component' => 'conditional_block'
            ],
            [
                'require_list' => false,
                'key' => 'enabled',
                'label' => __('Status', 'mailerpress-pro'),
                'component' => 'checkbox-single',
                'checkbox_label' => __('Enable this feed', 'mailerpress-pro')
            ]
        ]);

        return $allFieldSettings;
    }

    public function getMergeFields($list, $listId, $formId)
    {
        return false;
    }

    protected function getLists()
    {
        $modifiedList = [];

        if (!function_exists('mailerpress_get_lists')) {
            return $modifiedList;
        }

        $lists = mailerpress_get_lists();

        if (is_array($lists)) {
            foreach ($lists as $list) {
                if (is_object($list) && isset($list->list_id)) {
                    $modifiedList[$list->list_id] = $list->name ?? __('List', 'mailerpress-pro') . ' ' . $list->list_id;
                } elseif (is_array($list) && isset($list['list_id'])) {
                    $modifiedList[$list['list_id']] = $list['name'] ?? __('List', 'mailerpress-pro') . ' ' . $list['list_id'];
                }
            }
        }

        return $modifiedList;
    }

    protected function getTags()
    {
        $modifiedTags = [];

        if (!function_exists('mailerpress_get_tags')) {
            return $modifiedTags;
        }

        $tags = mailerpress_get_tags();

        if (is_array($tags)) {
            foreach ($tags as $tag) {
                if (is_object($tag) && isset($tag->tag_id)) {
                    $modifiedTags[$tag->tag_id] = $tag->name ?? __('Tag', 'mailerpress-pro') . ' ' . $tag->tag_id;
                } elseif (is_array($tag) && isset($tag['tag_id'])) {
                    $modifiedTags[$tag['tag_id']] = $tag['name'] ?? __('Tag', 'mailerpress-pro') . ' ' . $tag['tag_id'];
                }
            }
        }

        return $modifiedTags;
    }

    protected function getCustomFields()
    {
        $modifiedFields = [];

        try {
            // Get CustomFields instance from MailerPress container
            $customFieldsModel = \MailerPress\Core\Kernel::getContainer()->get(CustomFields::class);
            $fields = $customFieldsModel->all();

            if (is_array($fields)) {
                foreach ($fields as $field) {
                    if (is_object($field) && isset($field->field_key)) {
                        $modifiedFields[$field->field_key] = $field->label ?? $field->field_key;
                    }
                }
            }
        } catch (\Exception) {
        }

        return $modifiedFields;
    }

    protected function getFields()
    {
        $fields = [
            [
                'key' => 'email',
                'label' => __('Enter Email', 'mailerpress-pro'),
                'required' => true,
                'tips' => __('Enter Email', 'mailerpress-pro'),
                'component' => 'value_text'
            ],
            [
                'key' => 'firstname',
                'label' => __('Enter First Name', 'mailerpress-pro'),
                'required' => false,
                'tips' => __('Enter First Name', 'mailerpress-pro'),
                'component' => 'value_text'
            ],
            [
                'key' => 'lastname',
                'label' => __('Enter Last Name', 'mailerpress-pro'),
                'required' => false,
                'tips' => __('Enter Last Name', 'mailerpress-pro'),
                'component' => 'value_text'
            ],
            [
                'key' => 'tags',
                'label' => __('Select Tags', 'mailerpress-pro'),
                'required' => false,
                'tips' => __('Select Tags', 'mailerpress-pro'),
                'component' => 'select',
                'options' => $this->getTags(),
                'is_multiple' => true
            ],
            [
                'key' => 'subscription_status',
                'label' => __('Select Subscription Status', 'mailerpress-pro'),
                'required' => false,
                'tips' => __('Select Subscription Status', 'mailerpress-pro'),
                'component' => 'select',
                'options' => [
                    'subscribed' => __('Subscribed', 'mailerpress-pro'),
                    'unsubscribed' => __('Unsubscribed', 'mailerpress-pro'),
                    'pending' => __('Pending', 'mailerpress-pro')
                ]
            ],
            [
                'key' => 'double_optin',
                'label' => __('Double Opt-In', 'mailerpress-pro'),
                'required' => false,
                'tips' => __('When the double opt-in option is enabled, MailerPress will send a confirmation email to the user and will only add them to your MailerPress list upon confirmation.', 'mailerpress-pro'),
                'component' => 'checkbox-single',
                'checkbox_label' => __('Enable Double Opt-In', 'mailerpress-pro')
            ],
            [
                'key' => 'optin_source',
                'label' => __('Opt-in Source', 'mailerpress-pro'),
                'required' => false,
                'placeholder' => __('Fluent Form', 'mailerpress-pro'),
                'tips' => __('Specify the source of this opt-in. This will be stored with the contact for tracking purposes.', 'mailerpress-pro'),
                'component' => 'value_text'
            ]
        ];

        // Add custom fields dynamically
        $customFields = $this->getCustomFields();
        if (!empty($customFields)) {

            foreach ($customFields as $fieldKey => $fieldLabel) {
                $fields[] = [
                    'key' => 'custom_' . $fieldKey,
                    'label' => $fieldLabel,
                    'required' => false,
                    'component' => 'value_text',
                    'tips' => sprintf(__('Map %s value from a Fluent Forms field', 'mailerpress-pro'), $fieldLabel)
                ];
            }
        }

        return $fields;
    }

    public function notify($feed, $formData, $entry, $form)
    {

        $feedData = $feed['processedValues'] ?? [];

        if (!function_exists('add_mailerpress_contact')) {
            do_action('fluentform/integration_action_result', $feed, 'failed', __('Please install or activate MailerPress plugin.', 'mailerpress-pro'));
            return;
        }

        // Check if feed is enabled
        if (empty($feedData['enabled']) || $feedData['enabled'] === false) {
            return;
        }

        // Prepare contact data
        $contactData = [];

        // Email is required - processedValues already contains parsed values
        $email = ArrayHelper::get($feedData, 'email', '');

        // If email is not valid, try to get from formData (fallback)
        if (!is_email($email)) {
            $email = ArrayHelper::get($formData, $email, '');
        }

        if (empty($email) || !is_email($email)) {
            do_action('fluentform/integration_action_result', $feed, 'failed', __('Valid email address is required', 'mailerpress-pro'));
            return;
        }

        $contactData['contactEmail'] = sanitize_email($email);

        // First name - use processed value directly
        $firstName = ArrayHelper::get($feedData, 'firstname', '');
        if (!empty($firstName) && is_string($firstName)) {
            $contactData['contactFirstName'] = sanitize_text_field($firstName);
        }

        // Last name - use processed value directly
        $lastName = ArrayHelper::get($feedData, 'lastname', '');
        if (!empty($lastName) && is_string($lastName)) {
            $contactData['contactLastName'] = sanitize_text_field($lastName);
        }

        // List ID
        $listId = ArrayHelper::get($feedData, 'list_id', '');
        if (empty($listId)) {
            do_action('fluentform/integration_action_result', $feed, 'failed', __('MailerPress list ID is required', 'mailerpress-pro'));
            return;
        }

        $contactData['lists'] = [
            ['id' => intval($listId)]
        ];

        // Tags
        $tags = ArrayHelper::get($feedData, 'tags', []);
        if (!empty($tags)) {
            if (is_string($tags)) {
                $tags = explode(',', $tags);
            }
            if (is_array($tags) && !empty($tags)) {
                $contactData['tags'] = array_map(function ($tagId) {
                    return ['id' => intval($tagId)];
                }, array_filter($tags));
            }
        }

        // Handle double opt-in logic (same as Gravity Forms)
        $subscriptionOption = get_option('mailerpress_signup_confirmation', null);
        if (null !== $subscriptionOption) {
            $subscriptionOption = json_decode($subscriptionOption, true);
            // Vérifier que json_decode a retourné un tableau valide
            if (!is_array($subscriptionOption)) {
                $subscriptionOption = null;
            }
        }

        $confirmOption = ArrayHelper::get($feedData, 'double_optin', '');
        $confirmOption = $confirmOption === '1' || $confirmOption === true || $confirmOption === 1;

        // Vérifier que $subscriptionOption est un tableau avant d'accéder à ses clés
        $enableSignupConfirmation = false;
        if (is_array($subscriptionOption) && isset($subscriptionOption['enableSignupConfirmation'])) {
            $enableSignupConfirmation = (bool)$subscriptionOption['enableSignupConfirmation'];
        }

        // Set contact status based on double opt-in settings
        if (false === $enableSignupConfirmation) {
            $contactData['contactStatus'] = 'subscribed';
        } else {
            // If confirmOption is true and signup confirmation is enabled globally,
            // set to pending so double opt-in email is sent
            $contactData['contactStatus'] = $confirmOption ? 'pending' : 'subscribed';
        }

        // Custom fields mapping - using individual value_text fields with custom_ prefix
        $customFields = $this->getCustomFields();
        if (!empty($customFields)) {
            $contactCustomFields = [];

            foreach ($customFields as $fieldKey => $fieldLabel) {
                $fieldDataKey = 'custom_' . $fieldKey;
                $fieldValue = ArrayHelper::get($feedData, $fieldDataKey, '');

                if (!empty($fieldValue)) {
                    $contactCustomFields[sanitize_key($fieldKey)] = sanitize_text_field($fieldValue);
                }
            }

            if (!empty($contactCustomFields)) {
                $contactData['custom_fields'] = $contactCustomFields;
            }
        }

        // Add opt-in source - use configured value or default to 'fluentform'
        $optinSource = ArrayHelper::get($feedData, 'optin_source', 'Fluent Form');
        if (empty($optinSource)) {
            $optinSource = 'Fluent Form';
        }
        $contactData['opt_in_source'] = sanitize_text_field($optinSource);
        $contactData['optin_details'] = wp_json_encode([
            'form_id' => $form->id ?? null,
            'form_title' => $form->title ?? '',
            'entry_id' => $entry->id ?? null,
            'source' => sanitize_text_field($optinSource)
        ]);

        // Add contact to MailerPress immediately
        // Using try-catch to handle any errors gracefully
        try {
            $result = add_mailerpress_contact($contactData);
        } catch (\Exception $e) {
            $result = [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }

        if (isset($result['success']) && $result['success']) {
            do_action(
                'fluentform/integration_action_result',
                $feed,
                'success',
                __('MailerPress feed has been successfully initialized and pushed data', 'mailerpress-pro')
            );
        } else {
            $errorMessage = isset($result['error']) ? $result['error'] : __('Failed to add contact to MailerPress', 'mailerpress-pro');
            do_action('fluentform/integration_action_result', $feed, 'failed', $errorMessage);
        }
    }

    public function isConfigured()
    {
        return function_exists('add_mailerpress_contact');
    }
}
