<?php

namespace MailerPressPro\Services\ThirdParty;

use ElementorPro\Modules\Forms\Classes\Action_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use MailerPress\Models\CustomFields;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class MailerPressElementorFormAction extends Action_Base
{
    public function get_name()
    {
        return 'mailerpress';
    }

    public function get_label()
    {
        return __('MailerPress', 'mailerpress-pro');
    }

    public function register_settings_section($widget)
    {
        // Section: MailerPress Settings
        $widget->start_controls_section(
            'section_mailerpress',
            [
                'label' => __('MailerPress Settings', 'mailerpress-pro'),
                'condition' => [
                    'submit_actions' => $this->get_name(),
                ],
            ]
        );

        $mailerpress_lists = $this->get_mailerpress_lists();
        
        $widget->add_control(
            'mailerpress_list',
            [
                'label' => __('MailerPress Lists', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => $mailerpress_lists,
                'default' => [],
                'description' => __('Leave empty to automatically use the default list. Select one or more lists to add the contact to.', 'mailerpress-pro'),
            ]
        );

        $widget->add_control(
            'mailerpress_tags',
            [
                'label' => __('MailerPress Tags', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => $this->get_mailerpress_tags(),
                'default' => [],
                'description' => __('Select tags to assign to the contact.', 'mailerpress-pro'),
            ]
        );

        $widget->add_control(
            'mailerpress_double_optin',
            [
                'label' => __('Double Opt-In', 'mailerpress-pro'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'mailerpress-pro'),
                'label_off' => __('No', 'mailerpress-pro'),
                'return_value' => 'yes',
                'default' => 'yes',
                'description' => __('When enabled, MailerPress will send a confirmation email to the user and will only add them to your MailerPress list upon confirmation.', 'mailerpress-pro'),
            ]
        );

        $widget->add_control(
            'mailerpress_optin_source',
            [
                'label' => __('Opt-in Source', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('e.g., elementor, website, landing-page', 'mailerpress-pro'),
                'description' => __('Optional: Set a custom opt-in source identifier for tracking where contacts subscribed from. Leave empty to use default value.', 'mailerpress-pro'),
            ]
        );

        $widget->end_controls_section();

        // Section: Field Mapping
        $widget->start_controls_section(
            'section_mailerpress_field_mapping',
            [
                'label' => __('Field Mapping', 'mailerpress-pro'),
                'condition' => [
                    'submit_actions' => $this->get_name(),
                ],
            ]
        );

        $mapping_repeater = new Repeater();

        $mapping_repeater->add_control(
            'form_field_id',
            [
                'label' => __('Form Field', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('e.g., email, name, message', 'mailerpress-pro'),
                'description' => __('Enter the field ID from your form (e.g., "email", "name", "message").', 'mailerpress-pro'),
            ]
        );

        $mapping_repeater->add_control(
            'mailerpress_field',
            [
                'label' => __('MailerPress Field', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => $this->get_mailerpress_fields(),
            ]
        );

        $widget->add_control(
            'mailerpress_field_mapping',
            [
                'label' => __('Field Mapping', 'mailerpress-pro'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $mapping_repeater->get_controls(),
                'default' => [],
                'title_field' => '{{{ form_field_id }}} → {{{ mailerpress_field }}}',
                'description' => __('Map form fields to MailerPress fields. Leave empty to use default mapping (email, contactFirstName, contactLastName).', 'mailerpress-pro'),
            ]
        );

        $widget->end_controls_section();

        // Section: Custom Fields Mapping
        $widget->start_controls_section(
            'section_mailerpress_custom_fields',
            [
                'label' => __('Custom Fields Mapping', 'mailerpress-pro'),
                'condition' => [
                    'submit_actions' => $this->get_name(),
                ],
            ]
        );

        $custom_repeater = new Repeater();

        $custom_repeater->add_control(
            'form_field_id_custom',
            [
                'label' => __('Form Field', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('e.g., phone, company, message', 'mailerpress-pro'),
                'description' => __('Enter the field ID from your form.', 'mailerpress-pro'),
            ]
        );

        $custom_repeater->add_control(
            'custom_field_key',
            [
                'label' => __('Custom Field Key', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => $this->get_custom_fields(),
            ]
        );

        $widget->add_control(
            'mailerpress_custom_fields_mapping',
            [
                'label' => __('Custom Fields Mapping', 'mailerpress-pro'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $custom_repeater->get_controls(),
                'default' => [],
                'title_field' => '{{{ form_field_id_custom }}} → {{{ custom_field_key }}}',
                'description' => __('Map form fields to MailerPress custom fields.', 'mailerpress-pro'),
            ]
        );

        $widget->end_controls_section();
    }

    public function run($record, $ajax_handler)
    {
        if (!function_exists('add_mailerpress_contact')) {
            $ajax_handler->add_error_message(__('MailerPress plugin is not active.', 'mailerpress-pro'));
            return;
        }

        $settings = $record->get('form_settings');
        $raw_fields = $record->get('fields');

        // Get form fields as associative array
        // Store both by ID and by title/label for flexible mapping
        $fields = [];
        $fields_by_id = [];
        $fields_by_title = [];
        $fields_by_id_key = []; // Also store by 'id' key if present

        // Debug: Log all available fields for mapping reference
        $available_fields_info = [];

        foreach ($raw_fields as $id => $field) {
            // Handle different field structures
            $field_id = $field['id'] ?? $id;
            $field_value = $field['value'] ?? '';
            $field_title = $field['title'] ?? '';
            $field_label = $field['label'] ?? '';
            $field_name = $field['name'] ?? '';

            // Store by array key
            $fields[$id] = $field_value;
            $fields_by_id[$id] = $field_value;

            // Store by 'id' key if different from array key
            if ($field_id !== $id) {
                $fields_by_id_key[$field_id] = $field_value;
            }

            // Also index by title/label/name for easier mapping
            if (!empty($field_title)) {
                $fields_by_title[strtolower($field_title)] = $field_value;
            }
            if (!empty($field_label)) {
                $fields_by_title[strtolower($field_label)] = $field_value;
            }
            if (!empty($field_name)) {
                $fields_by_title[strtolower($field_name)] = $field_value;
            }

            // Collect field info for debugging
            $available_fields_info[] = [
                'array_key' => $id,
                'field_id' => $field_id,
                'title' => $field_title,
                'label' => $field_label,
                'name' => $field_name,
                'value' => $field_value,
            ];
        }
        // Get email (required)
        $email = '';

        // Try to find email from field mapping first
        $field_mapping = $settings['mailerpress_field_mapping'] ?? [];
        foreach ($field_mapping as $mapping) {
            $form_field_id = $mapping['form_field_id'] ?? '';
            $mailerpress_field = $mapping['mailerpress_field'] ?? '';

            if ($mailerpress_field === 'email' && !empty($form_field_id) && isset($fields[$form_field_id])) {
                $email = sanitize_email($fields[$form_field_id]);
                break;
            }
        }

        // Fallback: try common field IDs
        if (empty($email)) {
            $common_email_fields = ['email', 'e-mail', 'mail', 'your-email'];
            foreach ($common_email_fields as $field_id) {
                if (isset($fields[$field_id])) {
                    $email = sanitize_email($fields[$field_id]);
                    if (!empty($email) && is_email($email)) {
                        break;
                    }
                }
            }
        }

        // Last fallback: find any email field by type
        if (empty($email)) {
            foreach ($raw_fields as $id => $field) {
                if (isset($field['type']) && $field['type'] === 'email' && !empty($field['value'])) {
                    $email = sanitize_email($field['value']);
                    if (!empty($email) && is_email($email)) {
                        break;
                    }
                }
            }
        }

        if (empty($email) || !is_email($email)) {
            $ajax_handler->add_error_message(__('Valid email address is required.', 'mailerpress-pro'));
            return;
        }

        // Prepare contact data
        $contact_data = [
            'contactEmail' => $email,
        ];

        // Map standard fields
        $default_mapping = [
            'email' => 'contactEmail',
            'e-mail' => 'contactEmail',
            'mail' => 'contactEmail',
            'firstname' => 'contactFirstName',
            'first_name' => 'contactFirstName',
            'name' => 'contactFirstName', // Common fallback
            'lastname' => 'contactLastName',
            'last_name' => 'contactLastName',
            'surname' => 'contactLastName',
        ];

        // Process field mapping
        // Note: We process mapped fields even if empty to ensure updates work correctly
        foreach ($field_mapping as $mapping) {
            $form_field_id = $mapping['form_field_id'] ?? '';
            $mailerpress_field = $mapping['mailerpress_field'] ?? '';

            if (empty($form_field_id) || empty($mailerpress_field)) {
                continue;
            }

            // Try to find field value by ID first, then by title/label
            $field_value = null;

            // Method 1: Direct ID match
            if (isset($fields[$form_field_id])) {
                $field_value = $fields[$form_field_id];
            }
            // Method 2: Try to find by title/label (case-insensitive)
            elseif (isset($fields_by_title[strtolower($form_field_id)])) {
                $field_value = $fields_by_title[strtolower($form_field_id)];
            }
            // Method 3: Search in raw_fields by ID or title
            else {
                foreach ($raw_fields as $id => $field) {
                    // Check if ID matches
                    if ($id === $form_field_id) {
                        $field_value = $field['value'] ?? '';
                        break;
                    }
                    // Check if title or label matches (case-insensitive)
                    $field_title = strtolower($field['title'] ?? '');
                    $field_label = strtolower($field['label'] ?? '');
                    $search_id = strtolower($form_field_id);

                    if ($field_title === $search_id || $field_label === $search_id) {
                        $field_value = $field['value'] ?? '';
                        break;
                    }
                }
            }

            // If we found a field value, map it
            if ($field_value !== null && $field_value !== '') {
                if ($mailerpress_field === 'email') {
                    $contact_data['contactEmail'] = sanitize_email($field_value);
                } else {
                    $contact_data[$mailerpress_field] = sanitize_text_field($field_value);
                }
            }
        }

        // If no mapping found, try default mapping
        if (empty($field_mapping)) {
            foreach ($default_mapping as $form_field_id => $mailerpress_field) {
                if (isset($fields[$form_field_id]) && !empty($fields[$form_field_id])) {
                    if ($mailerpress_field === 'contactEmail') {
                        $contact_data['contactEmail'] = sanitize_email($fields[$form_field_id]);
                    } else {
                        $contact_data[$mailerpress_field] = sanitize_text_field($fields[$form_field_id]);
                    }
                }
            }
        }

        // Map custom fields
        $custom_fields_mapping = $settings['mailerpress_custom_fields_mapping'] ?? [];
        $custom_fields = [];

        foreach ($custom_fields_mapping as $mapping) {
            $form_field_id = trim($mapping['form_field_id_custom'] ?? '');
            $custom_field_key = trim($mapping['custom_field_key'] ?? '');

            if (empty($form_field_id) || empty($custom_field_key)) {
                continue;
            }

            // Try to find field value by multiple methods
            $field_value = null;
            $search_id_lower = strtolower($form_field_id);

            // Method 1: Direct ID match (array key)
            if (isset($fields[$form_field_id])) {
                $field_value = $fields[$form_field_id];
            }
            // Method 2: Match by 'id' key in field data
            elseif (isset($fields_by_id_key[$form_field_id])) {
                $field_value = $fields_by_id_key[$form_field_id];
            }
            // Method 3: Try to find by title/label/name (case-insensitive)
            elseif (isset($fields_by_title[$search_id_lower])) {
                $field_value = $fields_by_title[$search_id_lower];
            }
            // Method 4: Search in raw_fields by ID, title, label, or name
            else {
                foreach ($raw_fields as $id => $field) {
                    $field_id = $field['id'] ?? $id;
                    $field_title = strtolower($field['title'] ?? '');
                    $field_label = strtolower($field['label'] ?? '');
                    $field_name = strtolower($field['name'] ?? '');

                    // Check if ID matches (array key or 'id' key)
                    if ($id === $form_field_id || $field_id === $form_field_id) {
                        $field_value = $field['value'] ?? '';
                        break;
                    }
                    // Check if title, label, or name matches (case-insensitive)
                    if (
                        $field_title === $search_id_lower ||
                        $field_label === $search_id_lower ||
                        $field_name === $search_id_lower
                    ) {
                        $field_value = $field['value'] ?? '';
                        break;
                    }
                }
            }

            // If we found a field value, add it to custom fields
            if ($field_value !== null && $field_value !== '') {
                $custom_fields[$custom_field_key] = sanitize_text_field($field_value);
            }
        }

        // Always set custom_fields array, even if empty, to ensure proper structure
        $contact_data['custom_fields'] = $custom_fields;

        // Add lists (leave empty to use default list)
        $lists = $settings['mailerpress_list'] ?? [];

        if (is_string($lists)) {
            $lists = [$lists];
        }

        if (is_array($lists) && !empty($lists)) {
            $contact_data['lists'] = array_map(function ($list_id) {
                return ['id' => intval($list_id)];
            }, array_filter($lists));
        }
        // If no lists provided, the default list will be assigned automatically

        // Add tags
        $tags = $settings['mailerpress_tags'] ?? [];
        if (!empty($tags) && is_array($tags)) {
            $contact_data['tags'] = array_map(function ($tag_id) {
                return ['id' => intval($tag_id)];
            }, array_filter($tags));
        }

        // Subscription status
        $double_optin = $settings['mailerpress_double_optin'] ?? 'yes';
        $desiredStatus = ($double_optin === 'yes') ? 'pending' : 'subscribed';
        $contact_data['contactStatus'] = $desiredStatus;
        $contact_data['subscription_status'] = $desiredStatus;

        // Add opt-in source
        // First check if a custom optin_source is set in form action settings
        $custom_optin_source = !empty($settings['mailerpress_optin_source']) ? sanitize_text_field($settings['mailerpress_optin_source']) : '';
        
        if (!empty($custom_optin_source)) {
            // Use custom opt-in source from settings
            $contact_data['opt_in_source'] = $custom_optin_source;
        } else {
            // Use automatic logic based on double opt-in setting
            if ($desiredStatus === 'subscribed') {
                $contact_data['opt_in_source'] = 'manual';
            } else {
                $contact_data['opt_in_source'] = 'elementor';
            }
        }

        $contact_data['optin_details'] = wp_json_encode([
            'form_id' => $record->get('form_settings')['form_id'] ?? '',
            'source' => 'elementor_form',
        ]);

        // Add contact to MailerPress (will update if contact already exists)
        // The add_mailerpress_contact function should automatically update existing contacts
        // based on email address. We ensure all necessary data is provided for proper update.
        // 
        // Important: The function should handle updates automatically by email, but we ensure
        // that all fields are properly formatted and included to support both creation and update.
        // 
        // For existing contacts, the function should:
        // - Update contact information (name, custom fields, etc.)
        // - Add contact to specified lists (if not already in them)
        // - Add tags to contact
        // - Update subscription status if needed
        $result = add_mailerpress_contact($contact_data);

        if (!isset($result['success']) || !$result['success']) {
            $error_message = isset($result['error']) ? $result['error'] : __('Failed to add contact to MailerPress.', 'mailerpress-pro');
            $ajax_handler->add_error_message($error_message);
        }
    }

    protected function get_mailerpress_lists(): array
    {
        $lists = [];

        if (!function_exists('mailerpress_get_lists')) {
            return $lists;
        }

        $mailerpress_lists = mailerpress_get_lists();

        if (is_array($mailerpress_lists)) {
            foreach ($mailerpress_lists as $list) {
                if (is_object($list) && isset($list->list_id)) {
                    $lists[$list->list_id] = $list->name ?? __('List', 'mailerpress-pro') . ' ' . $list->list_id;
                } elseif (is_array($list) && isset($list['list_id'])) {
                    $lists[$list['list_id']] = $list['name'] ?? __('List', 'mailerpress-pro') . ' ' . $list['list_id'];
                }
            }
        }

        return $lists;
    }

    protected function get_mailerpress_tags(): array
    {
        $tags = [];

        if (!function_exists('mailerpress_get_tags')) {
            return $tags;
        }

        $mailerpress_tags = mailerpress_get_tags();

        if (is_array($mailerpress_tags)) {
            foreach ($mailerpress_tags as $tag) {
                if (is_object($tag) && isset($tag->tag_id)) {
                    $tags[$tag->tag_id] = $tag->name ?? __('Tag', 'mailerpress-pro') . ' ' . $tag->tag_id;
                } elseif (is_array($tag) && isset($tag['tag_id'])) {
                    $tags[$tag['tag_id']] = $tag['name'] ?? __('Tag', 'mailerpress-pro') . ' ' . $tag['tag_id'];
                }
            }
        }

        return $tags;
    }

    protected function get_mailerpress_fields(): array
    {
        return [
            '' => __('-- Select --', 'mailerpress-pro'),
            'email' => __('Email', 'mailerpress-pro'),
            'contactFirstName' => __('First Name', 'mailerpress-pro'),
            'contactLastName' => __('Last Name', 'mailerpress-pro'),
        ];
    }

    protected function get_custom_fields(): array
    {
        $fields = ['' => __('-- Select --', 'mailerpress-pro')];

        try {
            $customFieldsModel = \MailerPress\Core\Kernel::getContainer()->get(CustomFields::class);
            $custom_fields = $customFieldsModel->all();

            foreach ($custom_fields as $field) {
                $fields[$field->field_key] = $field->label . ' (' . $field->type . ')';
            }
        } catch (\Exception $e) {
            // Silently fail if container is not available
        }

        return $fields;
    }

    public function on_export($element)
    {
        // Remove sensitive data when exporting templates
        // Return empty array to keep all settings, or filter specific ones
        return [];
    }
}
