<?php

namespace MailerPressPro\Services\ThirdParty;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * MailerPress Email Provider for Divi Email Optin Module
 *
 * @since 1.2.0
 */
class MailerPressDiviEmailProvider extends \ET_Core_API_Email_Provider
{
    /**
     * @var string Provider name
     */
    public $name = 'MailerPress';

    /**
     * @var string Provider slug (must be lowercase)
     */
    public $slug = 'mailerpress';


    /**
     * @var string Custom fields scope
     */
    public $custom_fields_scope = 'account';


    /**
     * Constructor
     *
     * @param string $owner
     * @param string $account_name
     * @param string $api_key (not used for MailerPress)
     */
    public function __construct($owner = '', $account_name = '', $api_key = '')
    {
        parent::__construct($owner, $account_name, $api_key);

        // Check if MailerPress is active
        if (!function_exists('mailerpress_get_lists')) {
            return;
        }
    }

    /**
     * Get account fields for the Divi dashboard
     *
     * @return array
     */
    public function get_account_fields()
    {
        // MailerPress doesn't require API keys since it's a WordPress plugin
        return [
            'info' => [
                'label' => '',
                'type' => 'info',
                'value' => esc_html__(
                    'MailerPress is already installed. No additional configuration needed.',
                    'mailerpress-pro'
                ),
            ],
        ];
    }

    /**
     * Get data keymap
     *
     * @param array $keymap Key to get
     * @return array
     */
    public function get_data_keymap($keymap = [])
    {
        $keymap = [
            'list' => [
                'list_id' => 'id',
                'name' => 'name',
            ],
            'subscriber' => [
                'name' => 'first_name',
                'last_name' => 'last_name',
                'email' => 'email',
                'custom_fields' => 'custom_fields',
            ],
        ];

        return parent::get_data_keymap($keymap);
    }

    /**
     * Fetch subscriber lists from MailerPress
     *
     * @param string $account Not used
     * @return array|string
     */
    public function fetch_subscriber_lists($account = 'default')
    {
        if (!function_exists('mailerpress_get_lists')) {
            return esc_html__('MailerPress plugin is not active.', 'mailerpress-pro');
        }

        try {
            $lists = mailerpress_get_lists();


            $formatted_lists = [];

            // Add lists
            if (is_array($lists)) {
                foreach ($lists as $list) {
                    if (is_object($list) && isset($list->list_id)) {
                        $formatted_lists[] = [
                            'id' => (string)$list->list_id,
                            'name' => $list->name ?? __('List', 'mailerpress-pro') . ' ' . $list->list_id,
                        ];
                    } elseif (is_array($list) && isset($list['list_id'])) {
                        $formatted_lists[] = [
                            'id' => (string)$list['list_id'],
                            'name' => $list['name'] ?? __('List', 'mailerpress-pro') . ' ' . $list['list_id'],
                        ];
                    }
                }
            }

            if (!empty($formatted_lists)) {
                // Use _process_subscriber_lists like MailPoet does to apply keymap transformation
                $this->data['lists'] = $this->_process_subscriber_lists($formatted_lists);

                $list = is_array($formatted_lists) ? array_shift($formatted_lists) : [];
                $this->data['custom_fields'] = $this->_fetch_custom_fields('', $list);
            }

            $this->data['is_authorized'] = true;

            $this->save_data();

            return ['success' => $this->data];
        } catch (\Exception $e) {
            return esc_html__('Failed to fetch lists from MailerPress.', 'mailerpress-pro');
        }
    }

    /**
     * Fetch custom fields from MailerPress
     *
     * @param string $list_id
     * @param array $list
     * @return array
     */
    protected function _fetch_custom_fields($list_id = '', $list = [])
    {
        static $processed = null;

        if (is_null($processed)) {

            try {
                $customFieldsModel = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\CustomFields::class);
                $custom_fields = $customFieldsModel->all();

                $processed = [];

                foreach ($custom_fields as $field) {
                    $field_id = $field->field_key;

                    $processed[$field_id] = [
                        'field_id' => $field_id,
                        'name' => $field->label,
                        'type' => $this->_map_custom_field_type($field->type),
                    ];
                }
            } catch (\Exception $e) {
                $processed = [];
            }
        }

        return $processed;
    }

    /**
     * Map MailerPress field types to Divi types
     *
     * @param string $type
     * @return string
     */
    protected function _map_custom_field_type($type)
    {
        $type_map = [
            'text' => 'text',
            'textarea' => 'textarea',
            'number' => 'number',
            'email' => 'email',
            'date' => 'date',
            'select' => 'select',
            'checkbox' => 'checkbox',
            'radio' => 'radio',
        ];

        return isset($type_map[$type]) ? $type_map[$type] : 'any';
    }

    /**
     * Process custom fields before subscription
     *
     * @param array $args
     * @return array
     */
    protected function _process_custom_fields($args)
    {
        if (!isset($args['custom_fields'])) {
            return $args;
        }

        $custom_fields = $args['custom_fields'];

        unset($args['custom_fields']);

        foreach ($custom_fields as $field_id => $value) {
            if (is_array($value) && $value) {
                // This is a multiple choice field (eg. checkbox, radio, select)
                $value = array_keys($value);

                if (count($value) > 1) {
                    $value = implode(',', $value);
                } else {
                    $value = array_pop($value);
                }
            }

            self::$_->array_set($args, $field_id, $value);
        }

        return $args;
    }

    /**
     * Subscribe to MailerPress
     *
     * @param array $args Subscription data
     * @param string $url Not used
     * @return string
     */
    public function subscribe($args, $url = '')
    {
        if (!function_exists('add_mailerpress_contact')) {
            \ET_Core_Logger::error(esc_html__('MailerPress plugin is not active.', 'mailerpress-pro'));

            return esc_html__('An error occurred. Please try again later.', 'mailerpress-pro');
        }

        $args = et_core_sanitized_previously($args);



        // Get list_id from args BEFORE transformation
        // Divi uses array index, so we need to get the real list_id from the stored lists
        $lists = [];
        if (isset($args['list_id']) && $args['list_id'] !== '') {
            $list_index = intval($args['list_id']);

            // Load lists if not already loaded
            if (empty($this->data['lists'])) {
                $this->fetch_subscriber_lists();
            }


            $this->data['lists'] = array_values($this->data['lists']);


            // Get the real list_id from the list at the given index
            if (isset($this->data['lists'][$list_index]) && isset($this->data['lists'][$list_index]['list_id'])) {
                $real_list_id = intval($this->data['lists'][$list_index]['list_id']);
                if ($real_list_id > 0) {
                    $lists = [['id' => $real_list_id]];
                }
            } elseif ($list_index > 0) {
                // Fallback: use the index as list_id if structure is different
                $lists = [['id' => $list_index]];
            }
        }

        $subscriber_data = $this->transform_data_to_provider_format($args, 'subscriber');
        $subscriber_data = $this->_process_custom_fields($subscriber_data);
        $subscriber_data = self::$_->array_flatten($subscriber_data);
        $result = 'success';

        // Prepare contact data
        $contact_data = [];

        // Email (required)
        if (empty($subscriber_data['email'])) {
            return esc_html__('Email address is required.', 'mailerpress-pro');
        }

        $contact_data['contactEmail'] = $subscriber_data['email'];

        // First name
        if (!empty($subscriber_data['first_name'])) {
            $contact_data['contactFirstName'] = $subscriber_data['first_name'];
        }

        // Last name
        if (!empty($subscriber_data['last_name'])) {
            $contact_data['contactLastName'] = $subscriber_data['last_name'];
        }

        // Custom fields
        $custom_fields = [];
        foreach ($subscriber_data as $key => $value) {
            if (!in_array($key, ['email', 'first_name', 'last_name', 'custom_fields', 'contactEmail', 'contactFirstName', 'contactLastName'])) {
                $custom_fields[$key] = $value;
            }
        }
        if (!empty($custom_fields)) {
            $contact_data['custom_fields'] = $custom_fields;
        }

        /**
         * Check if the subscriber with this email already exists.
         */

        $contactModel = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Contacts::class);
        $subscriber = $contactModel->getContactByEmail($subscriber_data['email']);


        /**
         * If subscriber is not found, add as a new subscriber. Otherwise, update existing subscriber.
         */
        if (empty($subscriber)) {
            // New subscriber
            // Subscription status and double opt-in
            $subscriptionOption = get_option('mailerpress_signup_confirmation', null);
            if (null !== $subscriptionOption) {
                $subscriptionOption = json_decode($subscriptionOption, true);
                if (!is_array($subscriptionOption)) {
                    $subscriptionOption = null;
                }
            }

            $enableSignupConfirmation = false;
            if (is_array($subscriptionOption) && isset($subscriptionOption['enableSignupConfirmation'])) {
                $enableSignupConfirmation = (bool)$subscriptionOption['enableSignupConfirmation'];
            }

            // Use global setting for double opt-in
            $desiredStatus = $enableSignupConfirmation ? 'pending' : 'subscribed';
            $contact_data['contactStatus'] = $desiredStatus;
            $contact_data['subscription_status'] = $desiredStatus;

            // Set opt-in source
            $contact_data['opt_in_source'] = $desiredStatus === 'subscribed' ? 'manual' : 'divi';

            $contact_data['optin_details'] = wp_json_encode([
                'source' => 'divi_email_optin',
                'page_id' => get_the_ID(),
            ]);

            if (!empty($lists)) {
                $contact_data['lists'] = $lists;
            }

            try {
                $response = add_mailerpress_contact($contact_data);
                if (!isset($response['success']) || !$response['success']) {
                    $result = isset($response['error']) ? $response['error'] : esc_html__(
                        'Failed to add contact to MailerPress.',
                        'mailerpress-pro'
                    );
                }
            } catch (\Exception $exception) {
                $result = $exception->getMessage();
            }
        } else {
            // Existing subscriber - add to lists
            $contact_data['contactId'] = $subscriber->contact_id;

            if (!empty($lists)) {
                $contact_data['lists'] = $lists;
            }

            try {
                $response = add_mailerpress_contact($contact_data);
                if (!isset($response['success']) || !$response['success']) {
                    $result = isset($response['error']) ? $response['error'] : esc_html__(
                        'Failed to update contact in MailerPress.',
                        'mailerpress-pro'
                    );
                }
            } catch (\Exception $exception) {
                $result = $exception->getMessage();
            }
        }

        return $result;
    }
}
