<?php

namespace MailerPressPro\Services;

use WP_Error;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * MailerPress License & Update Manager
 * Workaround for HTTP 421 SNI mismatch errors
 */
class MailerPressLicence
{

    private static $instance;
    private $option_key = 'mailerpress_license';
    private $api_endpoint_value = 'wc-am-api';
    private $request_delay_seconds = 2; // Delay between requests

    public static function instance(): MailerPressLicence
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {}

    /* ---------------------------
     * Options / Instance handling
     * --------------------------- */

    public function get_options(): array
    {
        $opts = get_option($this->option_key, []);
        if (is_string($opts)) {
            $opts = json_decode($opts, true);
        }
        return is_array($opts) ? $opts : [];
    }

    public function update_options(array $data)
    {
        $opts = wp_parse_args($data, $this->get_options());
        update_option($this->option_key, $opts);
    }

    public function get_license_key(): ?string
    {
        $opts = $this->get_options();
        return !empty($opts['license_key']) ? trim($opts['license_key']) : null;
    }

    public function set_license_key(?string $key)
    {
        $opts = $this->get_options();
        $opts['license_key'] = $key;
        unset($opts['product_id']); // reset product id if key changes
        update_option($this->option_key, $opts);
    }

    public function get_instance_id(): string
    {
        $opts = $this->get_options();
        if (!empty($opts['instance'])) {
            return $opts['instance'];
        }

        $id = $this->generate_instance_id();
        $opts['instance'] = $id;
        update_option($this->option_key, $opts);
        return $id;
    }

    private function generate_instance_id(): string
    {
        if (function_exists('wp_generate_password')) {
            return wp_generate_password(24, false, false);
        }
        return bin2hex(random_bytes(12));
    }

    /* ---------------------------
     * Request delay mechanism to avoid 421
     * --------------------------- */
    private function enforce_request_delay()
    {
        $last_request = get_transient('mailerpress_last_api_call');

        if ($last_request !== false) {
            $time_since_last = time() - $last_request;

            if ($time_since_last < $this->request_delay_seconds) {
                $wait_time = $this->request_delay_seconds - $time_since_last;
                sleep($wait_time);
            }
        }

        set_transient('mailerpress_last_api_call', time(), 60);
    }

    /* ---------------------------
     * HTTP request with aggressive 421 prevention
     * --------------------------- */
    private function make_http_request(string $url, array $extra_args = []): array|WP_Error
    {
        // Enforce delay between requests
        $this->enforce_request_delay();

        // Add unique parameter to prevent any caching
        $url = add_query_arg([
            '_t' => microtime(true),
            '_r' => wp_generate_password(8, false, false)
        ], $url);

        $args = array_merge([
            'timeout' => 45,
            'httpversion' => '1.1',
            'blocking' => true,
            'headers' => [
                'Accept' => 'application/json',
                'User-Agent' => 'MailerPressPro/' . (MAILERPRESS_PRO_VERSION ?? '1.0'),
                'Connection' => 'close',
                'Cache-Control' => 'no-cache, no-store, must-revalidate',
                'Pragma' => 'no-cache',
                'Expires' => '0',
            ],
            'sslverify' => false,
            'redirection' => 5,
        ], $extra_args);

        // Use streams transport instead of cURL to avoid connection pooling
        add_filter('use_curl_transport', '__return_false', 999);
        add_filter('use_streams_transport', '__return_true', 999);

        $response = wp_remote_get($url, $args);

        // Restore default transport
        remove_filter('use_curl_transport', '__return_false', 999);
        remove_filter('use_streams_transport', '__return_true', 999);

        return $response;
    }

    /* ---------------------------
     * Product ID handling
     * --------------------------- */
    public function get_product_id(): string|WP_Error
    {
        $opts = $this->get_options();
        if (!empty($opts['product_id'])) {
            return $opts['product_id'];
        }

        $key = $this->get_license_key();
        if (empty($key)) {
            return new WP_Error('no_key', 'No license key set.');
        }

        $url = 'https://mailerpress.com/wp-json/license/v1/find?license_key=' . urlencode($key);

        $response = $this->make_http_request($url, ['timeout' => 20]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code < 200 || $code >= 300) {
           return new WP_Error('api_error', "HTTP {$code}", ['body' => $body]);
        }

        $decoded = json_decode($body, true);
        if (empty($decoded['product_id'])) {
            return new WP_Error('invalid_response', 'No product_id in response', ['body' => $body]);
        }

        $opts['product_id'] = $decoded['product_id'];
        $this->update_options($opts);

        return $decoded['product_id'];
    }

    /* ---------------------------
     * Core API call
     * --------------------------- */
    public function call_api(string $action, array $args = [])
    {
        if (!defined('MAILERPRESS_STORE_URL')) {
            return new WP_Error('missing_config', 'MAILERPRESS_STORE_URL must be defined.');
        }

        $product_id = $this->get_product_id();
        if (is_wp_error($product_id)) {
            return $product_id;
        }

        $base = untrailingslashit(MAILERPRESS_STORE_URL) . '/';
        $query = array_merge([
            'wc-api' => $this->api_endpoint_value,
            'plugin_name' => MAILERPRESS_PLUGIN_NAME,
            'wc_am_action' => $action,
            'product_id' => $product_id,
            'instance' => $this->get_instance_id(),
        ], $args);

        $url = add_query_arg($query, $base);

        $response = $this->make_http_request($url);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($code < 200 || $code >= 300) {
           return new WP_Error('api_error', "HTTP {$code}", ['body' => $body]);
        }

        $decoded = json_decode($body, true);
        if ($decoded === null) {
            return new WP_Error('invalid_json', 'API returned invalid JSON', ['body' => $body]);
        }

        if (isset($decoded['success']) && $decoded['success'] === false) {
            $errorMessage = $decoded['error'] ?? 'Unknown licensing error';
            $errorCode = $decoded['code'] ?? 'api_failure';
            return new WP_Error($errorCode, $errorMessage, ['data' => $decoded['data'] ?? []]);
        }

        return $decoded;
    }

    /* ---------------------------
     * License actions
     * --------------------------- */
    public function activate_license(): array|WP_Error
    {
        $key = $this->get_license_key();
        $parsed = parse_url(get_site_url());
        $domain = $parsed['host'];
        if (empty($key)) {
            return new WP_Error('no_key', 'No license key set.');
        }
        return $this->call_api('activate', ['api_key' => $key, 'object' => $domain, 'version' => MAILERPRESS_PRO_VERSION ?? '']);
    }

    public function deactivate_license(): array|WP_Error
    {
        $key = $this->get_license_key();
        if (empty($key)) {
            return new WP_Error('no_key', 'No license key set.');
        }
        return $this->call_api('deactivate', ['api_key' => $key]);
    }

    public function check_status(): array|WP_Error
    {
        $key = $this->get_license_key();
        if (empty($key)) {
            return new WP_Error('no_key', 'No license key set.');
        }
        return $this->call_api('status', ['api_key' => $key]);
    }

    public function verify_api_key_is_active(string $key): array|WP_Error
    {
        return $this->call_api('verify_api_key_is_active', ['api_key' => $key]);
    }

    /* ---------------------------
     * Update checking
     * --------------------------- */
    public function check_for_update(): array|WP_Error
    {
        $key = $this->get_license_key();
        if (empty($key)) {
            return new WP_Error('no_key', 'No license key set.');
        }
        return $this->call_api('update', [
            'api_key' => $key,
            'version' => MAILERPRESS_PRO_VERSION ?? ''
        ]);
    }
}
