<?php

namespace MailerPressPro\Services\AI;

use Exception;
use MailerPressPro\Abstract\AbstractAIService;
use MailerPressPro\Interfaces\AiModelInterface;

class GeminiService extends AbstractAIService implements AiModelInterface
{
    /**
     * Generate text using Gemini API
     */
    public function generate(string $prompt, array $options = []): string
    {
        $model = $options['text_ai']['model'] ?? 'gemini-2.5-pro';

        $url = sprintf(
            'https://generativelanguage.googleapis.com/v1beta/models/%s:generateContent?key=%s',
            $model,
            $this->api_key
        );

        $body = $this->createGeminiRequestBody($prompt, $options);


        $response = wp_remote_post($url, [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($body),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            throw new Exception('Gemini Request Error: ' . $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Gemini Response JSON decode error: ' . json_last_error_msg());
        }

        if ($status_code < 200 || $status_code >= 300) {
            throw new Exception('Gemini API returned HTTP ' . $status_code . ': ' . $response_body);
        }

        // Gemini returns nested structures
        if (isset($data['candidates'][0]['content']['parts'][0]['text'])) {
            return trim($data['candidates'][0]['content']['parts'][0]['text']);
        }

        throw new Exception('Gemini API returned unexpected structure: ' . $response_body);
    }

    /**
     * Suggest next words or phrases (optional helper)
     */
    public function suggest(string $prompt, array $options = [], int $limit = 3): array
    {
        if (empty($prompt)) {
            return [];
        }

        $model = $options['text_ai']['model'] ?? 'gemini-1.5-pro';

        $url = sprintf(
            'https://generativelanguage.googleapis.com/v1beta/models/%s:generateContent?key=%s',
            $model,
            $this->api_key
        );

        $body = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => [
                        ['text' => $prompt . "\nSuggest $limit possible next words as a JSON array."]
                    ]
                ]
            ],
            'generationConfig' => [
                'maxOutputTokens' => 40,
                'temperature' => 0.5,
            ],
        ];

        $response = wp_remote_post($url, [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => wp_json_encode($body),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            throw new Exception('Gemini Request Error: ' . $response->get_error_message());
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Gemini Response JSON decode error: ' . json_last_error_msg());
        }

        $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? '';

        $suggestions = json_decode(trim($text), true);
        if (!is_array($suggestions)) {
            $suggestions = preg_split('/[\s,]+/', $text);
        }

        return array_slice($suggestions, 0, $limit);
    }
}
