<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\Operators;
use MailerPressPro\Core\Segmentation\SegmentCondition;

class TagCondition implements SegmentCondition
{
    /** @var string|string[] */
    private $tags;

    private string $operator;

    private const ALLOWED_OPERATORS = [
        Operators::IS,
        Operators::IS_NOT,
        Operators::CONTAINS,
    ];

    /**
     * @param string|string[] $tags Single tag string or array of tags (for CONTAINS)
     * @param string $operator
     */
    public function __construct($tags, string $operator = Operators::IS)
    {
        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} not supported for TagCondition");
        }

        if ($operator === Operators::CONTAINS) {
            if (!is_array($tags) || empty($tags)) {
                throw new \InvalidArgumentException('Tags must be a non-empty array for CONTAINS operator.');
            }
            $this->tags = array_map('sanitize_text_field', $tags);
        } else {
            $this->tags = sanitize_text_field($tags);
        }

        $this->operator = $operator;
    }

    public function toSqlCondition(): string
    {
        global $wpdb;
        $tagsTable = $wpdb->prefix . 'mailerpress_contact_tags';

        if ($this->operator === Operators::IS) {
            return $wpdb->prepare(
                "EXISTS (SELECT 1 FROM {$tagsTable} WHERE contact_id = c.contact_id AND tag_id = %s)",
                $this->tags
            );
        } elseif ($this->operator === Operators::IS_NOT) {
            return $wpdb->prepare(
                "NOT EXISTS (SELECT 1 FROM {$tagsTable} WHERE contact_id = c.contact_id AND tag_id = %s)",
                $this->tags
            );
        } else { // CONTAINS
            $placeholders = implode(',', array_fill(0, count($this->tags), '%s'));
            $sql = "EXISTS (SELECT 1 FROM {$tagsTable} WHERE contact_id = c.contact_id AND tag_id IN ($placeholders))";

            return $wpdb->prepare($sql, ...$this->tags);
        }
    }

    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'tag',
            'label' => __('Contact Tag', 'mailerpress-pro'),
            'operators' => [
                Operators::IS => __('is', 'mailerpress-pro'),
                Operators::IS_NOT => __('is not', 'mailerpress-pro'),
                Operators::CONTAINS => __('contains', 'mailerpress-pro'),
            ],
            'datasource' => 'tags',
            'input_type' => 'select',
            'choices' => mailerpress_get_tags()
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}
