<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\Operators;
use MailerPressPro\Core\Segmentation\SegmentCondition;

class ListCondition implements SegmentCondition
{
    /** @var int|int[] */
    private $listIds;

    private string $operator;

    private const ALLOWED_OPERATORS = [
        Operators::IS,
        Operators::IS_NOT,
        Operators::CONTAINS,
    ];

    /**
     * @param int|int[] $listIds
     * @param string $operator
     */
    public function __construct($listIds, string $operator = Operators::IS)
    {
        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} not supported for ListCondition");
        }

        if ($operator === Operators::CONTAINS) {
            // Must be array of ints for CONTAINS
            if (!is_array($listIds) || empty($listIds)) {
                throw new \InvalidArgumentException('List IDs must be a non-empty array for CONTAINS operator.');
            }
            $this->listIds = array_map('absint', $listIds);
        } else {
            // For IS and IS_NOT, accept a single int
            $this->listIds = [absint($listIds)];
        }

        $this->operator = $operator;
    }

    public function toSqlCondition(): string
    {
        global $wpdb;
        $table = $wpdb->prefix . 'mailerpress_contact_lists';

        // Prepare list IDs as comma-separated string, escaped
        $ids = implode(',', array_map('intval', $this->listIds));

        if ($this->operator === Operators::IS) {
            return sprintf(
                "EXISTS (SELECT 1 FROM %s WHERE contact_id = c.contact_id AND list_id = %d)",
                esc_sql($table),
                $this->listIds[0]
            );
        } elseif ($this->operator === Operators::IS_NOT) {
            return sprintf(
                "NOT EXISTS (SELECT 1 FROM %s WHERE contact_id = c.contact_id AND list_id = %d)",
                esc_sql($table),
                $this->listIds[0]
            );
        } else { // CONTAINS
            return sprintf(
                "EXISTS (SELECT 1 FROM %s WHERE contact_id = c.contact_id AND list_id IN (%s))",
                esc_sql($table),
                $ids
            );
        }
    }

    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'list',
            'label' => __('Contact list', 'mailerpress-pro'),
            'operators' => [
                Operators::IS => __('is', 'mailerpress-pro'),
                Operators::IS_NOT => __('is not', 'mailerpress-pro'),
                Operators::CONTAINS => __('contains', 'mailerpress-pro'),
            ],
            'datasource' => 'lists',
            'input_type' => 'select',
            'choices' => mailerpress_get_lists()
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}