<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\Operators;
use MailerPressPro\Core\Segmentation\SegmentCondition;

class EngagementStatusCondition implements SegmentCondition
{
    private string $status;
    private string $operator;

    private const ALLOWED_OPERATORS = [
        Operators::IS,
        Operators::IS_NOT,
    ];

    /**
     * @param string|array $status
     * @param string $operator
     */
    public function __construct($status, string $operator = Operators::IS)
    {
        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} not supported for EngagementStatusCondition");
        }


        $this->status = sanitize_text_field($status);

        $this->operator = $operator;
    }

    public function toSqlCondition(): string
    {
        global $wpdb;

        // Map neutral and good to the same value
        $statuses = is_array($this->status) ? $this->status : [$this->status];
        $mappedStatuses = array_map(function($s) {
            return in_array($s, ['good'], true) ? 'good_or_neutral' : $s;
        }, $statuses);

        // Now handle SQL
        switch ($this->operator) {
            case Operators::IS:
                if (in_array('good_or_neutral', $mappedStatuses, true)) {
                    return "c.engagement_status IN ('good')";
                }
                return $wpdb->prepare('c.engagement_status = %s', $mappedStatuses[0] ?? 'bad');

            case Operators::IS_NOT:
                if (in_array('good_or_neutral', $mappedStatuses, true)) {
                    return "c.engagement_status NOT IN ('good'')";
                }
                return $wpdb->prepare('c.engagement_status != %s', $mappedStatuses[0] ?? 'bad');

            default:
                throw new \RuntimeException("Unsupported operator: {$this->operator}");
        }
    }


    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'engagement_status',
            'label' => __('Engagement Status', 'mailerpress-pro'),
            'operators' => [
                Operators::IS => __('is', 'mailerpress-pro'),
                Operators::IS_NOT => __('is not', 'mailerpress-pro'),
            ],
            'input_type' => 'select',
            'choices' => [
                'good' => __('Good', 'mailerpress-pro'),
                'neutral' => __('Moderate', 'mailerpress-pro'),
                'bad' => __('Bad', 'mailerpress-pro'),
            ],
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}
