<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\Operators;
use MailerPressPro\Core\Segmentation\SegmentCondition;

class CustomFieldCondition implements SegmentCondition
{
    private string $field_key;
    private $value;
    private string $operator;

    private const ALLOWED_OPERATORS = [
        Operators::EQUALS,
        Operators::NOT_EQUALS,
        Operators::CONTAINS,
        Operators::GREATER_THAN,
        Operators::LESS_THAN,
        Operators::GREATER_THAN_OR_EQUALS,
        Operators::LESS_THAN_OR_EQUALS,
        Operators::BEFORE,
        Operators::AFTER,
        Operators::ON,
    ];

    /**
     * @param string $field_key The custom field key
     * @param mixed $value The value to compare against
     * @param string $operator
     */
    public function __construct(string $field_key, $value, string $operator = Operators::EQUALS)
    {
        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} not supported for CustomFieldCondition");
        }

        $this->field_key = sanitize_key($field_key);
        $this->value = $value;
        $this->operator = $operator;
    }

    public function toSqlCondition(): string
    {
        global $wpdb;
        $customFieldsTable = $wpdb->prefix . 'mailerpress_contact_custom_fields';

        // Base subquery to check if contact has this custom field
        $baseSubquery = "SELECT 1 FROM {$customFieldsTable} WHERE contact_id = c.contact_id AND field_key = %s";

        switch ($this->operator) {
            case Operators::EQUALS:
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND field_value = %s)",
                    $this->field_key,
                    $this->value
                );

            case Operators::NOT_EQUALS:
                return $wpdb->prepare(
                    "NOT EXISTS ({$baseSubquery} AND field_value = %s)",
                    $this->field_key,
                    $this->value
                );

            case Operators::CONTAINS:
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND field_value LIKE %s)",
                    $this->field_key,
                    '%' . $wpdb->esc_like($this->value) . '%'
                );

            case Operators::GREATER_THAN:
                // For numeric comparisons, we need to cast the value
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND CAST(field_value AS DECIMAL(10,2)) > %f)",
                    $this->field_key,
                    floatval($this->value)
                );

            case Operators::LESS_THAN:
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND CAST(field_value AS DECIMAL(10,2)) < %f)",
                    $this->field_key,
                    floatval($this->value)
                );

            case Operators::GREATER_THAN_OR_EQUALS:
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND CAST(field_value AS DECIMAL(10,2)) >= %f)",
                    $this->field_key,
                    floatval($this->value)
                );

            case Operators::LESS_THAN_OR_EQUALS:
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND CAST(field_value AS DECIMAL(10,2)) <= %f)",
                    $this->field_key,
                    floatval($this->value)
                );

            case Operators::BEFORE:
                // For date fields: before end of day
                $safeDate = esc_sql($this->value);
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND CAST(field_value AS DATE) < %s)",
                    $this->field_key,
                    $safeDate
                );

            case Operators::AFTER:
                // For date fields: after start of day
                $safeDate = esc_sql($this->value);
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND CAST(field_value AS DATE) > %s)",
                    $this->field_key,
                    $safeDate
                );

            case Operators::ON:
                // For date fields: on specific date (entire day range)
                $safeDate = esc_sql($this->value);
                return $wpdb->prepare(
                    "EXISTS ({$baseSubquery} AND CAST(field_value AS DATE) = %s)",
                    $this->field_key,
                    $safeDate
                );

            default:
                throw new \InvalidArgumentException("Unsupported operator: {$this->operator}");
        }
    }

    public static function getConditionMetadata(): array
    {
        global $wpdb;
        $table = \MailerPress\Core\Enums\Tables::get(\MailerPress\Core\Enums\Tables::MAILERPRESS_CUSTOM_FIELD_DEFINITIONS);

        // Get all custom field definitions
        $custom_fields = $wpdb->get_results(
            "SELECT field_key, label, type FROM {$table} ORDER BY label ASC",
            ARRAY_A
        );

        // Format for select dropdown with type information
        $choices = array_map(function ($field) {
            return [
                'value' => $field['field_key'],
                'label' => $field['label'] . ' (' . $field['type'] . ')',
                'type' => $field['type'] // Include type for frontend filtering
            ];
        }, $custom_fields);

        return [
            'field' => 'custom_field',
            'label' => __('Custom Field', 'mailerpress-pro'),
            'operators' => [
                Operators::EQUALS => __('equals', 'mailerpress-pro'),
                Operators::NOT_EQUALS => __('does not equal', 'mailerpress-pro'),
                Operators::CONTAINS => __('contains', 'mailerpress-pro'),
                Operators::GREATER_THAN => __('greater than', 'mailerpress-pro'),
                Operators::LESS_THAN => __('less than', 'mailerpress-pro'),
                Operators::GREATER_THAN_OR_EQUALS => __('greater than or equal', 'mailerpress-pro'),
                Operators::LESS_THAN_OR_EQUALS => __('less than or equal', 'mailerpress-pro'),
                Operators::BEFORE => __('before', 'mailerpress-pro'),
                Operators::AFTER => __('after', 'mailerpress-pro'),
                Operators::ON => __('on', 'mailerpress-pro'),
            ],
            'datasource' => 'custom_fields',
            'input_type' => 'select', // For field selection
            'value_input_type' => 'text', // For value input (can be overridden based on field type)
            'choices' => $choices,
            'requires_field_key' => true, // Indicates this condition needs a field_key parameter
        ];
    }

    public static function canRun(): bool
    {
        global $wpdb;
        $table = \MailerPress\Core\Enums\Tables::get(\MailerPress\Core\Enums\Tables::MAILERPRESS_CUSTOM_FIELD_DEFINITIONS);

        // Check if custom fields table exists and has fields
        $count = $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
        return $count > 0;
    }
}
