<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use MailerPressPro\Core\EmailManager\BounceDetectionInterface;
use MailerPressPro\Core\EmailManager\BounceDetectionTrait;
use WP_Error;

class AmazonSES extends AbstractEmailService implements BounceDetectionInterface
{
    use BounceDetectionTrait;

    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        error_log('[MailerPress Amazon SES] Starting sendEmail');
        error_log('[MailerPress Amazon SES] Email data: ' . json_encode([
            'to' => $emailData['to'] ?? 'N/A',
            'subject' => $emailData['subject'] ?? 'N/A',
            'sender_to' => $emailData['sender_to'] ?? 'N/A',
            'sender_name' => $emailData['sender_name'] ?? 'N/A',
            'has_body' => !empty($emailData['body']),
        ]));

        // Récupérer les credentials depuis la configuration
        $accessKeyId = $this->config['conf']['access_key_id'] ?? '';
        $secretAccessKey = $this->config['conf']['secret_access_key'] ?? '';
        $region = $this->config['conf']['region'] ?? 'us-east-1';

        error_log('[MailerPress Amazon SES] Config check: ' . json_encode([
            'has_access_key' => !empty($accessKeyId),
            'has_secret_key' => !empty($secretAccessKey),
            'region' => $region,
            'access_key_prefix' => !empty($accessKeyId) ? substr($accessKeyId, 0, 4) . '...' : 'empty',
        ]));

        if (empty($accessKeyId) || empty($secretAccessKey)) {
            error_log('[MailerPress Amazon SES] ERROR: Missing credentials');
            return new WP_Error(
                'amazonses_missing_credentials',
                __('Amazon SES: Missing access key ID or secret access key.', 'mailerpress-pro')
            );
        }

        // Construire le message MIME
        $message = $this->buildRawMessage($emailData);
        $encodedMessage = base64_encode($message);

        error_log('[MailerPress Amazon SES] Message built: ' . json_encode([
            'message_length' => strlen($message),
            'encoded_length' => strlen($encodedMessage),
        ]));

        // Construire les paramètres de la requête
        $params = [
            'Action' => 'SendRawEmail',
            'Version' => '2010-12-01',
            'RawMessage.Data' => $encodedMessage,
        ];

        // Construire l'URL de l'endpoint
        $endpoint = sprintf('email.%s.amazonaws.com', $region);
        $url = sprintf('https://%s/', $endpoint);

        error_log('[MailerPress Amazon SES] Endpoint: ' . $url);

        // Créer la signature AWS v4
        $date = gmdate('Ymd\THis\Z');
        $dateStamp = gmdate('Ymd');
        $service = 'ses';
        $algorithm = 'AWS4-HMAC-SHA256';

        // Créer le body de la requête
        $requestBody = http_build_query($params, '', '&', PHP_QUERY_RFC3986);
        $payloadHash = hash('sha256', $requestBody);

        // Créer la requête canonique pour POST
        $canonicalHeaders = "host:{$endpoint}\n" . "x-amz-date:{$date}\n";
        $signedHeaders = 'host;x-amz-date';

        $canonicalRequest = "POST\n/\n\n{$canonicalHeaders}\n{$signedHeaders}\n{$payloadHash}";

        // Créer la string à signer
        $credentialScope = "{$dateStamp}/{$region}/{$service}/aws4_request";
        $stringToSign = "{$algorithm}\n{$date}\n{$credentialScope}\n" . hash('sha256', $canonicalRequest);

        // Calculer la signature
        $signingKey = $this->getSigningKey($secretAccessKey, $dateStamp, $region, $service);
        $signature = hash_hmac('sha256', $stringToSign, $signingKey);

        // Créer l'en-tête d'autorisation
        $authorization = sprintf(
            '%s Credential=%s/%s, SignedHeaders=%s, Signature=%s',
            $algorithm,
            $accessKeyId,
            $credentialScope,
            $signedHeaders,
            $signature
        );

        error_log('[MailerPress Amazon SES] Request prepared: ' . json_encode([
            'url' => $url,
            'method' => 'POST',
            'date' => $date,
            'credential_scope' => $credentialScope,
            'authorization_prefix' => substr($authorization, 0, 50) . '...',
        ]));

        // Envoyer la requête
        $response = wp_remote_post($url, [
            'headers' => [
                'Host' => $endpoint,
                'Authorization' => $authorization,
                'X-Amz-Date' => $date,
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => $requestBody,
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            $errorMessage = $response->get_error_message();
            $errorCode = $response->get_error_code();
            error_log('[MailerPress Amazon SES] ERROR - WP_Error: ' . json_encode([
                'code' => $errorCode,
                'message' => $errorMessage,
            ]));
            return new WP_Error(
                'amazonses_request_failed',
                sprintf(
                    __('Amazon SES request failed: %s', 'mailerpress-pro'),
                    esc_html($errorMessage)
                )
            );
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $responseBody = wp_remote_retrieve_body($response);
        $responseHeaders = wp_remote_retrieve_headers($response);

        error_log('[MailerPress Amazon SES] Response received: ' . json_encode([
            'status_code' => $statusCode,
            'body_length' => strlen($responseBody),
            'headers' => $responseHeaders->getAll(),
        ]));

        if ($statusCode !== 200) {
            error_log('[MailerPress Amazon SES] ERROR - Non-200 status: ' . json_encode([
                'status_code' => $statusCode,
                'response_body' => $responseBody,
            ]));

            // Essayer de parser la réponse XML d'Amazon SES pour obtenir plus de détails
            $errorCode = 'Unknown';
            $errorMessage = __('An unexpected error occurred while sending the email.', 'mailerpress-pro');
            $errorType = 'Unknown';

            if (!empty($responseBody)) {
                $xml = @simplexml_load_string($responseBody);
                if ($xml !== false) {
                    $errorCode = (string)($xml->Error->Code ?? 'Unknown');
                    $errorMessage = (string)($xml->Error->Message ?? 'Unknown error');
                    $errorType = (string)($xml->Error->Type ?? 'Unknown');

                    error_log('[MailerPress Amazon SES] AWS Error details: ' . json_encode([
                        'code' => $errorCode,
                        'type' => $errorType,
                        'message' => $errorMessage,
                    ]));

                    // Améliorer le message pour les erreurs courantes
                    if ($errorCode === 'MessageRejected' && strpos($errorMessage, 'not verified') !== false) {
                        $errorMessage = sprintf(
                            /* translators: %s: Original AWS error message */
                            __('Amazon SES: Email addresses are not verified. %s Please verify your email addresses in the AWS SES console or request production access to send to any email address.', 'mailerpress-pro'),
                            esc_html($errorMessage)
                        );
                    } elseif ($errorCode === 'MessageRejected') {
                        $errorMessage = sprintf(
                            /* translators: %s: Original AWS error message */
                            __('Amazon SES: Message rejected. %s', 'mailerpress-pro'),
                            esc_html($errorMessage)
                        );
                    } else {
                        $errorMessage = sprintf(
                            /* translators: %s: Original AWS error message */
                            __('Amazon SES error (%s): %s', 'mailerpress-pro'),
                            esc_html($errorCode),
                            esc_html($errorMessage)
                        );
                    }
                } else {
                    // Si on ne peut pas parser le XML, utiliser le body brut
                    $errorMessage = sprintf(
                        __('Amazon SES error (HTTP %d): %s', 'mailerpress-pro'),
                        $statusCode,
                        esc_html(substr($responseBody, 0, 200))
                    );
                }
            }

            return new WP_Error(
                'amazonses_api_error',
                $errorMessage
            );
        }

        error_log('[MailerPress Amazon SES] SUCCESS - Email sent successfully');
        return true;
    }

    /**
     * Construit un message MIME brut pour Amazon SES
     */
    private function buildRawMessage(array $emailData): string
    {
        $boundary = '----=_Part_' . uniqid();
        $to = is_array($emailData['to']) ? implode(', ', $emailData['to']) : $emailData['to'];
        $from = $emailData['sender_to'] ?? '';
        $fromName = $emailData['sender_name'] ?? '';
        $subject = $emailData['subject'] ?? '';
        $body = $emailData['body'] ?? '';

        error_log('[MailerPress Amazon SES] buildRawMessage: ' . json_encode([
            'to' => $to,
            'from' => $from,
            'from_name' => $fromName,
            'subject' => $subject,
            'body_length' => strlen($body),
            'has_reply_to' => !empty($emailData['reply_to_address']),
        ]));

        if (empty($from) || empty($to)) {
            error_log('[MailerPress Amazon SES] ERROR in buildRawMessage: Missing from or to address');
        }

        $headers = [];
        $headers[] = "From: " . ($fromName ? "{$fromName} <{$from}>" : $from);
        $headers[] = "To: {$to}";
        $headers[] = "Subject: {$subject}";
        $headers[] = "MIME-Version: 1.0";
        $headers[] = "Content-Type: multipart/alternative; boundary=\"{$boundary}\"";

        if (!empty($emailData['reply_to_address'])) {
            $replyToName = $emailData['reply_to_name'] ?? '';
            $headers[] = "Reply-To: " . ($replyToName ? "{$replyToName} <{$emailData['reply_to_address']}>" : $emailData['reply_to_address']);
        }

        $message = implode("\r\n", $headers) . "\r\n\r\n";
        $message .= "--{$boundary}\r\n";
        $message .= "Content-Type: text/html; charset=UTF-8\r\n";
        $message .= "Content-Transfer-Encoding: base64\r\n\r\n";
        $message .= chunk_split(base64_encode($body)) . "\r\n";
        $message .= "--{$boundary}--\r\n";

        error_log('[MailerPress Amazon SES] Raw message built: ' . json_encode([
            'message_length' => strlen($message),
            'headers_count' => count($headers),
        ]));

        return $message;
    }

    /**
     * Génère la clé de signature pour AWS Signature Version 4
     */
    private function getSigningKey(string $secretKey, string $dateStamp, string $region, string $service): string
    {
        $kDate = hash_hmac('sha256', $dateStamp, 'AWS4' . $secretKey, true);
        $kRegion = hash_hmac('sha256', $region, $kDate, true);
        $kService = hash_hmac('sha256', $service, $kRegion, true);
        return hash_hmac('sha256', 'aws4_request', $kService, true);
    }

    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'amazonses',
            'name' => __('Amazon SES', 'mailerpress-pro'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="4 3 92 92" fill="none"><g filter="url(#filter0_d_3582_119696)"><rect width="92" height="92" x="4" y="3" fill="#F1F1F1" rx="46"></rect><g clip-path="url(#clip0_3582_119696)"><path fill="#252F3E" d="M34.324 45.688c0 .874.094 1.583.258 2.103.188.52.423 1.087.751 1.7.117.19.164.379.164.544 0 .236-.14.473-.445.709l-1.479.992c-.211.142-.422.213-.61.213-.235 0-.47-.118-.704-.33a7.31 7.31 0 0 1-.845-1.111c-.235-.402-.47-.85-.727-1.394-1.83 2.173-4.13 3.26-6.9 3.26-1.971 0-3.543-.567-4.693-1.7-1.15-1.135-1.737-2.647-1.737-4.537 0-2.008.704-3.639 2.136-4.867 1.431-1.229 3.332-1.843 5.75-1.843.797 0 1.618.07 2.487.189.868.118 1.76.307 2.698.52V38.41c0-1.796-.375-3.048-1.102-3.78-.752-.733-2.019-1.087-3.826-1.087-.821 0-1.666.095-2.534.307a18.6 18.6 0 0 0-2.535.803 6.69 6.69 0 0 1-.821.307c-.164.048-.282.071-.375.071-.33 0-.493-.236-.493-.732v-1.158c0-.378.047-.661.164-.827.117-.165.329-.33.657-.496.821-.425 1.807-.78 2.957-1.063 1.15-.307 2.37-.449 3.66-.449 2.794 0 4.835.638 6.15 1.914 1.29 1.276 1.947 3.213 1.947 5.812v7.655h.047Zm-9.528 3.591c.775 0 1.573-.142 2.417-.425.845-.284 1.596-.803 2.23-1.512.375-.45.657-.945.798-1.512.14-.567.234-1.252.234-2.056v-.992a19.479 19.479 0 0 0-2.159-.402 17.568 17.568 0 0 0-2.206-.142c-1.572 0-2.722.308-3.496.946-.775.637-1.15 1.535-1.15 2.717 0 1.11.282 1.937.868 2.504.563.59 1.385.874 2.464.874Zm18.845 2.552c-.423 0-.704-.071-.892-.237-.188-.141-.352-.472-.493-.92L36.741 32.41c-.14-.473-.211-.78-.211-.945 0-.378.188-.59.563-.59h2.3c.446 0 .751.07.915.235.188.142.329.473.47.922l3.942 15.64 3.661-15.64c.117-.473.258-.78.446-.922.188-.141.516-.236.939-.236h1.877c.446 0 .751.071.939.236.188.142.352.473.446.922l3.708 15.83 4.06-15.83c.14-.473.304-.78.469-.922.188-.141.493-.236.915-.236h2.182c.376 0 .587.19.587.59 0 .119-.023.237-.047.379-.023.142-.07.33-.164.59l-5.656 18.264c-.14.472-.305.78-.492.921-.188.142-.493.236-.892.236h-2.019c-.445 0-.75-.07-.938-.236-.188-.165-.352-.472-.446-.945l-3.638-15.239-3.614 15.215c-.117.473-.258.78-.445.945-.188.166-.517.237-.94.237h-2.017Zm30.155.638c-1.22 0-2.44-.142-3.614-.426-1.173-.283-2.088-.59-2.698-.945-.376-.212-.634-.449-.728-.661a1.678 1.678 0 0 1-.14-.662V48.57c0-.496.187-.732.54-.732.14 0 .28.023.421.07.141.048.353.142.587.237.798.354 1.666.638 2.582.827.938.189 1.854.283 2.792.283 1.479 0 2.629-.26 3.427-.78.797-.519 1.22-1.275 1.22-2.244 0-.661-.211-1.205-.634-1.654-.422-.448-1.22-.85-2.37-1.228l-3.403-1.063c-1.713-.544-2.98-1.347-3.755-2.41-.774-1.04-1.173-2.197-1.173-3.426 0-.992.211-1.867.634-2.623a6.064 6.064 0 0 1 1.69-1.937c.703-.543 1.501-.945 2.44-1.229.939-.283 1.924-.401 2.957-.401.516 0 1.056.023 1.572.094.54.071 1.033.166 1.525.26.47.118.916.236 1.338.378.423.142.751.284.986.426.328.189.563.378.704.59.14.19.211.45.211.78v1.11c0 .496-.188.756-.54.756-.187 0-.493-.094-.892-.283-1.337-.614-2.839-.922-4.505-.922-1.338 0-2.394.213-3.121.662-.728.449-1.103 1.134-1.103 2.103 0 .661.234 1.228.704 1.677.469.45 1.337.898 2.581 1.3l3.332 1.063c1.69.543 2.91 1.3 3.638 2.268.727.969 1.08 2.08 1.08 3.308 0 1.016-.212 1.937-.61 2.74a6.344 6.344 0 0 1-1.714 2.08c-.727.59-1.596 1.015-2.605 1.322-1.056.331-2.159.497-3.356.497Z"></path><path fill="#F90" fill-rule="evenodd" d="M78.232 63.951c-7.721 5.741-18.938 8.79-28.584 8.79-13.517 0-25.697-5.033-34.896-13.397-.727-.662-.07-1.56.798-1.04 9.95 5.813 22.224 9.333 34.92 9.333 8.565 0 17.976-1.796 26.635-5.481 1.291-.591 2.394.85 1.127 1.795Z" clip-rule="evenodd"></path><path fill="#F90" fill-rule="evenodd" d="M81.447 60.265c-.986-1.275-6.524-.614-9.035-.307-.751.095-.869-.567-.188-1.063 4.412-3.119 11.663-2.22 12.508-1.181.845 1.063-.234 8.364-4.365 11.86-.633.544-1.243.26-.962-.449.939-2.339 3.027-7.607 2.042-8.86Z" clip-rule="evenodd"></path></g></g><defs><clipPath id="clip0_3582_119696"><path fill="#fff" d="M14.048 30H85.39v43H14.05z"></path></clipPath><filter id="filter0_d_3582_119696" width="100" height="100" x="0" y="0" color-interpolation-filters="sRGB" filterUnits="userSpaceOnUse"><feFlood flood-opacity="0" result="BackgroundImageFix"></feFlood><feColorMatrix in="SourceAlpha" result="hardAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"></feColorMatrix><feOffset dy="1"></feOffset><feGaussianBlur stdDeviation="2"></feGaussianBlur><feColorMatrix values="0 0 0 0 0.0687866 0 0 0 0 0.097585 0 0 0 0 0.37981 0 0 0 0.0779552 0"></feColorMatrix><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_3582_119696"></feBlend><feBlend in="SourceGraphic" in2="effect1_dropShadow_3582_119696" result="shape"></feBlend></filter></defs></svg>',
            'link' => 'https://aws.amazon.com/ses/',
            'createAccountLink' => 'https://aws.amazon.com/',
            'linkApiKey' => 'https://console.aws.amazon.com/iam/home#/security_credentials',
            'description' => __("Amazon SES offers a reliable and cost-effective service for sending and receiving emails using your own domain. It leverages Amazon's robust infrastructure, making it a powerful option for managing your email communication.", "mailerpress-pro"),
            'recommended' => true,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ],
            'bounce_doc_url' => 'https://mailerpress.com/docs/bounce-tracking/amazonses',
        ];
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function syncContacts(string $apiKey): void
    {
        // Amazon SES n'a pas de système de contacts comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
    }

    public function syncLists(string $apiKey): void
    {
        // Amazon SES n'a pas de système de listes comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
    }

    public function syncTags(string $apiKey): void {}

    public function pushContacts(string $apiKey, array $data): mixed
    {
        // Amazon SES n'a pas de système de contacts comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
        return null;
    }

    public function pushLists(string $apiKey, array $data): mixed
    {
        // Amazon SES n'a pas de système de listes comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
        return null;
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        // Amazon SES n'a pas de système de listes comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
    }

    public function pushTags(string $apiKey): void
    {
        // Amazon SES n'a pas de système de tags comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
    }

    public function deleteAllLists(string $apiKey): void
    {
        // Amazon SES n'a pas de système de listes comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
    }

    public function deleteContact(string $apiKey, array $data): void
    {
        // Amazon SES n'a pas de système de contacts comme les autres ESP
        // Cette méthode est laissée vide car Amazon SES est uniquement un service d'envoi d'emails
    }

    // ============================================
    // Bounce Detection Implementation
    // ============================================

    /**
     * Amazon SES n'utilise pas de webhooks directs comme les autres ESP
     * Il utilise plutôt Amazon SNS (Simple Notification Service) pour les notifications
     *
     * @return bool False car SES n'utilise pas de webhooks directs
     */
    public function supportsWebhookBounceDetection(): bool
    {
        return false;
    }

    /**
     * Amazon SES supporte la détection de bounces via API, mais avec des limitations
     * L'API SES ne fournit pas directement une liste de bounces comme PostMark
     * Il faut utiliser SNS pour recevoir les notifications en temps réel
     *
     * @return bool True car l'API est supportée (mais avec limitations)
     */
    public function supportsApiBounceDetection(): bool
    {
        return true;
    }

    /**
     * Configure un webhook pour recevoir les notifications de bounce
     *
     * Note: Amazon SES n'utilise pas de webhooks directs. Il utilise SNS (Simple Notification Service).
     * Cette méthode retourne false car la configuration doit être faite manuellement dans la console AWS.
     *
     * @param string $apiKey La clé API de SES (non utilisée)
     * @param string $webhookUrl L'URL du webhook (non utilisée)
     * @return array|false Retourne false car la configuration doit être faite manuellement
     */
    public function configureBounceWebhook(string $apiKey, string $webhookUrl): array|false
    {
        // Amazon SES utilise SNS, pas de webhooks directs
        // La configuration doit être faite manuellement dans la console AWS
        // On sauvegarde juste l'URL pour référence
        update_option('mailerpress_amazonses_bounce_sns_url', $webhookUrl);

        return [
            'url' => $webhookUrl,
            'description' => __('Amazon SES uses SNS (Simple Notification Service) for bounce notifications. Please configure SNS manually in your AWS console to send notifications to this URL.', 'mailerpress-pro'),
        ];
    }

    /**
     * Supprime le webhook de bounce configuré
     *
     * Note: Amazon SES utilise SNS, la suppression doit être faite manuellement dans la console AWS.
     *
     * @param string $apiKey La clé API de SES (non utilisée)
     * @param string|null $webhookId L'ID du webhook (non utilisé)
     * @return bool True si la suppression locale a réussi
     */
    public function removeBounceWebhook(string $apiKey, ?string $webhookId = null): bool
    {
        delete_option('mailerpress_amazonses_bounce_sns_url');
        return true;
    }

    /**
     * Récupère les bounces depuis l'API Amazon SES
     *
     * Note: L'API SES ne fournit pas directement une liste de bounces comme PostMark.
     * Amazon SES recommande d'utiliser SNS pour recevoir les notifications de bounce en temps réel.
     * Cette méthode retourne un tableau vide car l'API SES ne supporte pas cette fonctionnalité.
     *
     * Documentation: https://docs.aws.amazon.com/ses/latest/dg/monitor-sending-activity-using-notifications.html
     *
     * @param string $apiKey La clé API de SES (non utilisée car l'API ne supporte pas cette fonctionnalité)
     * @param array $options Options pour la récupération (non utilisées)
     * @return array Tableau vide car l'API SES ne fournit pas directement les bounces
     */
    public function fetchBouncesFromApi(string $apiKey, array $options = []): array
    {
        error_log('[MailerPress Pro Amazon SES] fetchBouncesFromApi called - Amazon SES API does not provide direct bounce retrieval. Use SNS for real-time bounce notifications.');

        // L'API SES ne fournit pas directement une liste de bounces
        // Il faut utiliser SNS pour recevoir les notifications en temps réel
        // Retourner un tableau vide avec un log pour informer
        return [];
    }

    /**
     * Traite une notification SNS depuis Amazon SES
     *
     * Amazon SES utilise SNS (Simple Notification Service) pour envoyer des notifications.
     * SNS peut envoyer trois types de messages :
     * 1. SubscriptionConfirmation - pour confirmer l'abonnement initial
     * 2. Notification - les notifications réelles de bounce/complaint
     * 3. UnsubscribeConfirmation - pour confirmer le désabonnement
     *
     * Format du message SNS depuis Amazon SES:
     * {
     *   "Type": "Notification" | "SubscriptionConfirmation" | "UnsubscribeConfirmation",
     *   "MessageId": "...",
     *   "TopicArn": "arn:aws:sns:...",
     *   "Message": "{\"notificationType\":\"Bounce\",\"bounce\":{\"bounceType\":\"Permanent\",\"bounceSubType\":\"General\",\"bouncedRecipients\":[{\"emailAddress\":\"example@example.com\",\"action\":\"failed\",\"status\":\"5.1.1\",\"diagnosticCode\":\"smtp; 550 5.1.1 user unknown\"}],\"timestamp\":\"2023-01-01T00:00:00.000Z\",\"feedbackId\":\"...\",\"reportingMTA\":\"dns; ...\"},\"mail\":{\"timestamp\":\"2023-01-01T00:00:00.000Z\",\"source\":\"sender@example.com\",\"messageId\":\"...\",\"destination\":[\"example@example.com\"]}}",
     *   "Timestamp": "2023-01-01T00:00:00.000Z",
     *   "SignatureVersion": "1",
     *   "Signature": "...",
     *   "SigningCertURL": "...",
     *   "UnsubscribeURL": "...",
     *   "SubscribeURL": "..." // Pour SubscriptionConfirmation
     * }
     *
     * @param array $payload Les données de la notification SNS
     * @return bool True si le traitement a réussi
     */
    public function processBounceWebhook(array $payload): bool
    {
        error_log(sprintf('[MailerPress Pro Amazon SES] Processing SNS notification payload - Type: %s', $payload['Type'] ?? 'unknown'));

        // Vérifier le type de message SNS
        $messageType = $payload['Type'] ?? null;

        if (!$messageType) {
            error_log('[MailerPress Pro Amazon SES] SNS notification error - Missing Type field');
            return false;
        }

        // Gérer la confirmation de souscription SNS
        if ($messageType === 'SubscriptionConfirmation') {
            error_log('[MailerPress Pro Amazon SES] Received SNS SubscriptionConfirmation - Confirming subscription...');

            if (!isset($payload['SubscribeURL'])) {
                error_log('[MailerPress Pro Amazon SES] SubscriptionConfirmation error - Missing SubscribeURL');
                return false;
            }

            // Confirmer l'abonnement en accédant à l'URL fournie
            $subscribeUrl = $payload['SubscribeURL'];
            $response = wp_remote_get($subscribeUrl, [
                'timeout' => 10,
                'sslverify' => true,
            ]);

            if (is_wp_error($response)) {
                error_log(sprintf(
                    '[MailerPress Pro Amazon SES] Failed to confirm SNS subscription - Error: %s',
                    $response->get_error_message()
                ));
                return false;
            }

            $statusCode = wp_remote_retrieve_response_code($response);
            if ($statusCode === 200) {
                error_log('[MailerPress Pro Amazon SES] SNS subscription confirmed successfully');
                return true;
            } else {
                error_log(sprintf(
                    '[MailerPress Pro Amazon SES] SNS subscription confirmation failed - Status code: %d',
                    $statusCode
                ));
                return false;
            }
        }

        // Gérer la confirmation de désabonnement SNS
        if ($messageType === 'UnsubscribeConfirmation') {
            error_log('[MailerPress Pro Amazon SES] Received SNS UnsubscribeConfirmation - Subscription is being removed');
            // On ne fait rien, juste logger
            return true;
        }

        // Traiter uniquement les notifications réelles
        if ($messageType !== 'Notification') {
            error_log(sprintf(
                '[MailerPress Pro Amazon SES] SNS notification ignored - Type is "%s" (expected "Notification", "SubscriptionConfirmation", or "UnsubscribeConfirmation")',
                $messageType
            ));
            return false;
        }

        // Parser le message JSON depuis SNS
        if (!isset($payload['Message'])) {
            error_log('[MailerPress Pro Amazon SES] Webhook error - Missing Message field');
            return false;
        }

        $message = json_decode($payload['Message'], true);
        if (!$message || !isset($message['notificationType'])) {
            error_log('[MailerPress Pro Amazon SES] Webhook error - Invalid Message format or missing notificationType');
            return false;
        }

        // Traiter uniquement les notifications de bounce
        if ($message['notificationType'] !== 'Bounce') {
            error_log(sprintf(
                '[MailerPress Pro Amazon SES] Webhook ignored - Notification type is "%s" (expected "Bounce")',
                $message['notificationType']
            ));
            return false;
        }

        if (!isset($message['bounce']) || !isset($message['bounce']['bouncedRecipients'])) {
            error_log('[MailerPress Pro Amazon SES] Webhook error - Missing bounce data or bouncedRecipients');
            return false;
        }

        $bounce = $message['bounce'];
        $bouncedRecipients = $bounce['bouncedRecipients'] ?? [];

        if (empty($bouncedRecipients)) {
            error_log('[MailerPress Pro Amazon SES] Webhook error - No bounced recipients found');
            return false;
        }

        $processedCount = 0;
        $bounceType = $bounce['bounceType'] ?? 'Unknown';
        $bounceSubType = $bounce['bounceSubType'] ?? 'Unknown';

        // Traiter chaque destinataire rebondi
        foreach ($bouncedRecipients as $recipient) {
            if (!isset($recipient['emailAddress'])) {
                continue;
            }

            $email = $recipient['emailAddress'];
            $reason = $recipient['diagnosticCode'] ?? $recipient['status'] ?? $bounceSubType;
            $action = $recipient['action'] ?? 'failed';

            $metadata = [
                'event' => 'bounce',
                'bounce_type' => $bounceType,
                'bounce_sub_type' => $bounceSubType,
                'action' => $action,
                'status' => $recipient['status'] ?? null,
                'diagnostic_code' => $recipient['diagnosticCode'] ?? null,
                'timestamp' => $bounce['timestamp'] ?? null,
                'feedback_id' => $bounce['feedbackId'] ?? null,
                'message_id' => $message['mail']['messageId'] ?? null,
            ];

            $result = $this->markContactAsBounced($email, $reason, $metadata);

            if ($result) {
                $processedCount++;
                error_log(sprintf(
                    '[MailerPress Pro Amazon SES] Bounce processed successfully - Email: %s | Type: %s | SubType: %s | Reason: %s',
                    $email,
                    $bounceType,
                    $bounceSubType,
                    $reason
                ));
            } else {
                error_log(sprintf(
                    '[MailerPress Pro Amazon SES] Failed to mark contact as bounced - Email: %s | Type: %s',
                    $email,
                    $bounceType
                ));
            }
        }

        return $processedCount > 0;
    }

    /**
     * Obtient l'URL du webhook pour Amazon SES (via SNS)
     *
     * @return string|null L'URL du webhook SNS ou null si non configuré
     */
    public function getBounceWebhookUrl(): ?string
    {
        $webhookUrl = get_option('mailerpress_amazonses_bounce_sns_url');

        if (empty($webhookUrl)) {
            // Construire l'URL du webhook MailerPress
            $webhookUrl = rest_url('mailerpress/v1/esp/bounce/amazonses');
        }

        return $webhookUrl;
    }

    /**
     * Valide la signature d'un webhook SNS depuis Amazon SES
     *
     * Amazon SNS utilise une signature RSA pour valider les messages.
     * Documentation: https://docs.aws.amazon.com/sns/latest/dg/sns-verify-signature-of-message.html
     *
     * @param string $payload Le corps de la requête
     * @param string|array $signature La signature fournie (peut être un array)
     * @param string|null $secret Le secret pour valider la signature (non utilisé pour SNS)
     * @return bool True si la signature est valide
     */
    public function verifyWebhookSignature(string $payload, string|array $signature, ?string $secret = null): bool
    {
        if (is_array($signature)) {
            $signature = $signature[0] ?? '';
        }

        if (empty($signature)) {
            error_log('[MailerPress Pro Amazon SES] Webhook signature verification failed - Signature is empty');
            return false;
        }

        // Parser le payload pour obtenir les informations SNS
        $data = json_decode($payload, true);
        if (!$data || !isset($data['SigningCertURL']) || !isset($data['Signature'])) {
            error_log('[MailerPress Pro Amazon SES] Webhook signature verification failed - Missing SNS signature data');
            return false;
        }

        // Étape 1: Télécharger et vérifier le certificat SNS
        $certPem = $this->getSNSCertificate($data['SigningCertURL']);
        if (!$certPem) {
            error_log('[MailerPress Pro Amazon SES] Webhook signature verification failed - Could not retrieve SNS certificate');
            return false;
        }

        // Étape 2: Extraire la clé publique du certificat
        $publicKey = openssl_pkey_get_public($certPem);
        if ($publicKey === false) {
            error_log('[MailerPress Pro Amazon SES] Webhook signature verification failed - Could not extract public key from certificate');
            return false;
        }

        // Étape 3: Construire la chaîne de caractères à signer selon le type de message SNS
        $stringToSign = $this->buildSNSStringToSign($data);
        if ($stringToSign === null) {
            openssl_free_key($publicKey);
            error_log('[MailerPress Pro Amazon SES] Webhook signature verification failed - Could not build string to sign');
            return false;
        }

        // Étape 4: Décoder la signature (elle est encodée en base64)
        $decodedSignature = base64_decode($data['Signature']);
        if ($decodedSignature === false) {
            openssl_free_key($publicKey);
            error_log('[MailerPress Pro Amazon SES] Webhook signature verification failed - Could not decode signature');
            return false;
        }

        // Étape 5: Vérifier la signature RSA
        // SNS peut utiliser SHA1 (version 1) ou SHA256 (version 2)
        $signatureVersion = $data['SignatureVersion'] ?? '1';
        $signatureAlgo = ($signatureVersion === '2') ? OPENSSL_ALGO_SHA256 : OPENSSL_ALGO_SHA1;

        $result = openssl_verify($stringToSign, $decodedSignature, $publicKey, $signatureAlgo);

        // Libérer la clé publique
        openssl_free_key($publicKey);

        if ($result === 1) {
            error_log('[MailerPress Pro Amazon SES] SNS signature verified successfully');
            return true;
        } elseif ($result === 0) {
            error_log('[MailerPress Pro Amazon SES] SNS signature verification failed - Invalid signature');
            return false;
        } else {
            $errorMsg = openssl_error_string();
            error_log(sprintf(
                '[MailerPress Pro Amazon SES] SNS signature verification error - OpenSSL error: %s',
                $errorMsg ?: 'Unknown error'
            ));
            return false;
        }
    }

    /**
     * Télécharge et met en cache le certificat SNS depuis l'URL fournie
     *
     * @param string $certUrl L'URL du certificat SNS
     * @return string|null Le contenu du certificat PEM ou null en cas d'échec
     */
    private function getSNSCertificate(string $certUrl): ?string
    {
        // Vérifier que l'URL provient bien d'AWS SNS (sécurité critique)
        if (!preg_match('/^https:\/\/sns\.[a-z0-9-]+\.amazonaws\.com\//', $certUrl)) {
            error_log(sprintf(
                '[MailerPress Pro Amazon SES] Certificate URL validation failed - Invalid domain: %s',
                $certUrl
            ));
            return null;
        }

        // Vérifier le cache (les certificats SNS sont valides pendant longtemps)
        $cacheKey = 'mailerpress_sns_cert_' . md5($certUrl);
        $cached = get_transient($cacheKey);

        if ($cached !== false && is_string($cached)) {
            error_log('[MailerPress Pro Amazon SES] Using cached SNS certificate');
            return $cached;
        }

        // Télécharger le certificat
        error_log(sprintf('[MailerPress Pro Amazon SES] Downloading SNS certificate from: %s', $certUrl));

        $response = wp_remote_get($certUrl, [
            'sslverify' => true,
            'timeout' => 10,
        ]);

        if (is_wp_error($response)) {
            error_log(sprintf(
                '[MailerPress Pro Amazon SES] Failed to download SNS certificate - Error: %s',
                $response->get_error_message()
            ));
            return null;
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        if ($statusCode !== 200) {
            error_log(sprintf(
                '[MailerPress Pro Amazon SES] Failed to download SNS certificate - HTTP status: %d',
                $statusCode
            ));
            return null;
        }

        $cert = wp_remote_retrieve_body($response);

        if (empty($cert)) {
            error_log('[MailerPress Pro Amazon SES] Failed to download SNS certificate - Empty response');
            return null;
        }

        // Vérifier que c'est un certificat valide
        $parsedCert = openssl_x509_parse($cert);
        if ($parsedCert === false) {
            error_log('[MailerPress Pro Amazon SES] Failed to parse SNS certificate - Invalid certificate format');
            return null;
        }

        // Mettre en cache pour 24 heures
        set_transient($cacheKey, $cert, DAY_IN_SECONDS);

        error_log('[MailerPress Pro Amazon SES] SNS certificate downloaded and cached successfully');

        return $cert;
    }

    /**
     * Construit la chaîne de caractères à signer pour la vérification de signature SNS
     *
     * L'ordre et le contenu des champs dépendent du type de message SNS.
     * Documentation: https://docs.aws.amazon.com/sns/latest/dg/sns-verify-signature-of-message.html
     *
     * @param array $message Le message SNS parsé
     * @return string|null La chaîne à signer ou null en cas d'erreur
     */
    private function buildSNSStringToSign(array $message): ?string
    {
        $messageType = $message['Type'] ?? null;

        if (!$messageType) {
            error_log('[MailerPress Pro Amazon SES] Cannot build string to sign - Missing Type field');
            return null;
        }

        // Les champs à signer dépendent du type de message
        if ($messageType === 'SubscriptionConfirmation' || $messageType === 'UnsubscribeConfirmation') {
            // Pour les confirmations d'abonnement/désabonnement
            $fields = ['Message', 'MessageId', 'SubscribeURL', 'Timestamp', 'Token', 'TopicArn', 'Type'];
        } else {
            // Pour les notifications normales
            $fields = ['Message', 'MessageId', 'Subject', 'Timestamp', 'TopicArn', 'Type'];
        }

        // Construire la chaîne dans l'ordre exact requis par SNS
        $stringToSign = '';
        foreach ($fields as $field) {
            if (isset($message[$field])) {
                $stringToSign .= $field . "\n";
                $stringToSign .= $message[$field] . "\n";
            }
        }

        if (empty($stringToSign)) {
            error_log('[MailerPress Pro Amazon SES] Cannot build string to sign - No valid fields found');
            return null;
        }

        return $stringToSign;
    }
}
