<?php

namespace MailerPressPro\Api;

use Exception;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Batch;
use MailerPressPro\Core\Attributes\Endpoint;
use MailerPressPro\Core\Kernel;
use MailerPressPro\Core\Segmentation\ConditionFactory;
use MailerPressPro\Core\Segmentation\Segment;
use MailerPressPro\Core\Segmentation\SegmentCondition;
use MailerPressPro\Core\Segmentation\SegmentQueryBuilder;
use MailerPressPro\Factories\AIFactory;
use Psr\Container\ContainerInterface;
use WP_REST_Request;
use WP_REST_Response;

class Segments
{
    private ContainerInterface $container;

    /**
     * @param ContainerInterface $container
     */
    public function __construct(ContainerInterface $container)
    {
        $this->container = $container;
    }

    #[Endpoint(
        'segments/(?P<id>\d+)',
        methods: 'PUT', // or 'PUT,PATCH' if your attribute supports comma-delimited list
        permissionCallback: [Permissions::class, 'canEdit']
    )]
    public function updateSegment(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $segmentTable = Tables::get(Tables::MAILERPRESS_SEGMENTS);

        $id = (int)$request->get_param('id');
        if ($id <= 0) {
            return new \WP_Error(
                'invalid_id',
                __('A valid segment ID is required.', 'mailerpress-pro'),
                ['status' => 400]
            );
        }

        // Does it exist?
        $row = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$segmentTable} WHERE id = %d", $id),
            ARRAY_A
        );
        if (!$row) {
            return new \WP_Error(
                'segment_not_found',
                __('Segment not found.', 'mailerpress-pro'),
                ['status' => 404]
            );
        }

        // Params
        $name = sanitize_text_field($request->get_param('name'));
        $conditions = $request->get_param('conditions');

        if (empty($name)) {
            return new \WP_Error('missing_name', __('Segment name is required.', 'mailerpress-pro'), ['status' => 400]);
        }

        if (empty($conditions)) {
            return new \WP_Error(
                'missing_conditions',
                __('Conditions are required.', 'mailerpress-pro'),
                ['status' => 400]
            );
        }

        // Duplicate name check (case‑insensitive, exclude current ID)
        $existing_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM {$segmentTable} WHERE LOWER(name) = LOWER(%s) AND id <> %d LIMIT 1",
                $name,
                $id
            )
        );
        if ($existing_id) {
            return new \WP_Error(
                'duplicate_name',
                sprintf(__('A segment named "%s" already exists.', 'mailerpress-pro'), esc_html($name)),
                ['status' => 409]
            );
        }

        // Update
        $updated = $wpdb->update(
            $segmentTable,
            [
                'name' => $name,
                'conditions' => maybe_serialize($conditions),
                'updated_at' => current_time('mysql'),
            ],
            ['id' => $id],
            ['%s', '%s', '%s'],
            ['%d']
        );

        if (false === $updated) {
            return new \WP_Error(
                'db_update_error',
                __('Failed to update segment.', 'mailerpress-pro'),
                ['status' => 500]
            );
        }

        return new \WP_REST_Response(
            [
                'id' => $id,
                'name' => $name,
                'conditions' => $conditions,
                'message' => __('Segment updated successfully.', 'mailerpress-pro'),
            ],
            200
        );
    }


    #[Endpoint(
        'segments',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function getSegments(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $segmentTable = Tables::get(Tables::MAILERPRESS_SEGMENTS);

        $search = $request->get_param('search');
        $per_page = isset($_GET['perPages']) ? (int)(wp_unslash($_GET['perPages'])) : 20;
        $page = isset($_GET['paged']) ? (int)(wp_unslash($_GET['paged'])) : 1;
        $offset = ($page - 1) * $per_page;

        $where = '1=1';
        $params = [];

        if (!empty($search)) {
            $where .= ' AND s.name LIKE %s';
            $params[] = '%' . $wpdb->esc_like($search) . '%';
        }

        $allowed_orderby = ['name', 'segment_id'];
        $allowed_order = ['ASC', 'DESC'];
        $orderby_param = $request->get_param('orderby');
        $order_param = strtoupper($request->get_param('order'));
        $orderby = in_array($orderby_param, $allowed_orderby, true) ? $orderby_param : 'name';
        $order = in_array($order_param, $allowed_order, true) ? $order_param : 'ASC';
        $orderBy = "s.$orderby $order";

        $query = $wpdb->prepare("
        SELECT
            s.*,
            s.id
        FROM {$segmentTable} s
        WHERE {$where}
        ORDER BY {$orderBy}
        LIMIT %d OFFSET %d
    ", [...$params, $per_page, $offset]);

        $segments = $wpdb->get_results($query);

        // Unserialize `conditions` if present
        foreach ($segments as &$segment) {
            if (isset($segment->conditions) && is_serialized($segment->conditions)) {
                $unserialized = maybe_unserialize($segment->conditions);
                $segment->conditions = json_encode($unserialized);
            }

            $segmentName = $segment->name ?? ''; // Or however you identify the segment

            // Build the WP_REST_Request for count only
            $request = new WP_REST_Request('GET', '/mailerpress/v1/getContactSegment');
            $request->set_param('segmentName', $segmentName);
            $request->set_param('onlyIds', true);

            // Call your endpoint method (adjust if it's in another class)
            $response = $this->contactBySegment($request);

            // Extract count from response headers or body
            if ($response instanceof \WP_REST_Response) {
                $segment->contactCount = count($response->get_data());
            } else {
                // Handle errors or WP_Error
                $segment->contactCount = 0;
            }
        }

        $total_query = $wpdb->prepare("
        SELECT COUNT(*)
        FROM {$segmentTable} s
        WHERE {$where}
    ", $params);

        $total_count = $wpdb->get_var($total_query);
        $total_pages = ceil($total_count / $per_page);

        $response = [
            'posts' => $segments,
            'pages' => $total_pages,
            'count' => $total_count,
        ];

        return new \WP_REST_Response($response, 200);
    }

    #[Endpoint(
        'segments',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canEdit']
    )]
    public function createSegment(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $segmentTable = Tables::get(Tables::MAILERPRESS_SEGMENTS);

        $name = sanitize_text_field($request->get_param('name'));
        $conditions = $request->get_param('conditions');

        if (empty($name)) {
            return new \WP_Error('missing_name', __('Segment name is required.', 'mailerpress-pro'), ['status' => 400]);
        }

        if (empty($conditions)) {
            return new \WP_Error(
                'missing_conditions',
                __('Conditions are required.', 'mailerpress-pro'),
                ['status' => 400]
            );
        }

        // --- Duplicate name check (case-insensitive) ---------------------------------
        $existing_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM {$segmentTable} WHERE LOWER(name) = LOWER(%s) LIMIT 1",
                $name
            )
        );
        if ($existing_id) {
            return new \WP_Error(
                'duplicate_name',
                sprintf(__('A segment named "%s" already exists.', 'mailerpress-pro'), esc_html($name)),
                ['status' => 409] // Conflict
            );
        }
        // -----------------------------------------------------------------------------

        $inserted = $wpdb->insert(
            $segmentTable,
            [
                'name' => $name,
                'conditions' => maybe_serialize($conditions),
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql'),
            ],
            ['%s', '%s', '%s', '%s']
        );

        if ($inserted === false) {
            return new \WP_Error('db_insert_error', __('Failed to insert segment.', 'mailerpress-pro'), ['status' => 500]);
        }

        $id = $wpdb->insert_id;

        return new \WP_REST_Response([
            'id' => $id,
            'message' => __('Segment created successfully.', 'mailerpress-pro'),
        ], 201);
    }


    #[Endpoint(
        'getContactSegment',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function contactBySegment(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $segmentTable = Tables::get(Tables::MAILERPRESS_SEGMENTS);
        $segmentName = sanitize_text_field($request->get_param('segmentName'));

        // Only fetch ids?
        $onlyIds = (bool)$request->get_param('onlyIds');
        $responseType = $request->get_param('response_type');
        // Detect whether pagination is requested (explicit page or per_page)
        $pageParam = $request->get_param('page');
        $perPageParam = $request->get_param('per_page');
        $paginate = $pageParam !== null || $perPageParam !== null;

        // Normalized pagination values (used only if $paginate == true)
        $page = max(1, (int)($pageParam ?: 1));
        $perPage = max(1, (int)($perPageParam ?: 20));
        // Optional cap to prevent abuse; adjust or remove if not desired.
        $perPage = min($perPage, 100);
        $offset = ($page - 1) * $perPage;

        // Fetch segment row
        $segment_row = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$segmentTable} WHERE name = %s", $segmentName),
            ARRAY_A
        );

        if (!$segment_row || !isset($segment_row['conditions'])) {
            return new \WP_Error('segment_not_found', 'Segment not found or invalid', ['status' => 404]);
        }

        $raw_conditions = maybe_unserialize($segment_row['conditions']);
        $segment_conditions = is_string($raw_conditions) ? json_decode($raw_conditions, true) : $raw_conditions;

        if (!is_array($segment_conditions) || !isset($segment_conditions['conditions'])) {
            return new \WP_Error('invalid_conditions', 'Segment conditions could not be parsed', ['status' => 400]);
        }

        // Build Segment object
        $segment = new Segment($segment_conditions['operator'] ?? 'AND');
        foreach ($segment_conditions['conditions'] as $cond) {
            try {
                $conditionObj = ConditionFactory::create($cond);
                $segment->addCondition($conditionObj);
            } catch (\InvalidArgumentException) {
            }
        }

        // Build SQL (supports field override)
        $queryBuilder = new SegmentQueryBuilder($segment);
        $sql = $queryBuilder->buildQuery([
            'fields' => '*',
        ]);

        // Apply pagination only if requested
        if ($paginate) {
            $sql .= $wpdb->prepare(' LIMIT %d OFFSET %d', $perPage, $offset);
        }

        $contacts = $wpdb->get_results($sql);

        if ($paginate) {
            $countSql = $queryBuilder->buildQuery(['fields' => 'COUNT(*)']);
            $total = (int)$wpdb->get_var($countSql);
            $totalPages = (int)ceil($total / $perPage);
        } else {
            $total = count($contacts);
            $totalPages = 1;
        }

        // Custom response format if requested
        if ($responseType === 'listing') {
            $paged = $page ?? 1;
            $total_rows = $total;
            $total_pages = $totalPages;
            $results = $contacts;

            return new \WP_REST_Response([
                'posts' => $results,
                'count' => $total_rows,
                'pages' => $total_pages,
                'current_page' => $paged,
            ], 200);
        }

        // Prepare response (body unchanged)
        $response = rest_ensure_response($contacts);

        // Add pagination headers only if paginated
        if ($paginate) {
            // Total (count query always counts IDs for accuracy)
            $countSql = $queryBuilder->buildQuery(['fields' => 'COUNT(*)']);
            $total = (int)$wpdb->get_var($countSql);
            $totalPages = (int)ceil($total / $perPage);

            $response->header('X-WP-Total', $total);
            $response->header('X-WP-TotalPages', $totalPages);
            $response->header('X-WP-Page', $page);
            $response->header('X-WP-Per-Page', $perPage);
        }

        return $response;
    }


    #[Endpoint(
        'conditions-metadata',
        methods: 'GET',
    )]
    public function segmentConditions(WP_REST_Request $request): WP_REST_Response
    {
        $conditionsDir = Kernel::$config['root'] . '/src/Core/Segmentation/Conditions';
        $namespacePrefix = 'MailerPressPro\\Core\\Segmentation\\Conditions\\';

        $metadata = [];

        foreach (glob($conditionsDir . '/*.php') as $file) {
            $className = basename($file, '.php');
            $fullClass = $namespacePrefix . $className;

            if (!class_exists($fullClass)) {
                continue;
            }

            if (
                !in_array('MailerPressPro\Core\Segmentation\SegmentCondition', class_implements($fullClass))
            ) {
                continue;
            }

            if (false === $fullClass::canRun()) {
                continue;
            }

            $info = $fullClass::getConditionMetadata();

            if (is_array($info['field'])) {
                foreach ($info['field'] as $field) {
                    $metadata[$field] = [
                        'operators' => $info['operators'],
                        'input_type' => $info['input_type'],
                        'label' => $info['label'],
                        'datasource' => $info['datasource'] ?? '',
                        'choices' => $info['choices'] ?? [],
                    ];
                }
            } else {
                $metadata[$info['field']] = [
                    'operators' => $info['operators'],
                    'input_type' => $info['input_type'],
                    'datasource' => $info['datasource'] ?? '',
                    'choices' => $info['choices'] ?? [],
                    'label' => $info['label'],
                    'value_input_type' => $info['value_input_type'] ?? 'text',
                    'requires_field_key' => $info['requires_field_key'] ?? false,
                ];
            }
        }


        return rest_ensure_response($metadata);
    }

    #[Endpoint(
        'segments/(?P<id>\d+)',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canEdit']
    )]
    public function deleteSegment(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $segmentTable = Tables::get(Tables::MAILERPRESS_SEGMENTS);

        // Route param captured via (?P<id>\d+) in the Endpoint path.
        $id = (int)$request->get_param('id');

        if ($id <= 0) {
            return new \WP_Error(
                'invalid_id',
                __('A valid segment ID is required.', 'mailerpress-pro'),
                ['status' => 400]
            );
        }

        // Confirm the segment exists before attempting delete.
        $exists = (int)$wpdb->get_var(
            $wpdb->prepare("SELECT COUNT(*) FROM {$segmentTable} WHERE id = %d", $id)
        );

        if ($exists === 0) {
            return new \WP_Error(
                'segment_not_found',
                __('Segment not found.', 'mailerpress-pro'),
                ['status' => 404]
            );
        }

        // Perform delete.
        $deleted = $wpdb->delete(
            $segmentTable,
            ['id' => $id],
            ['%d']
        );

        if (false === $deleted) {
            return new \WP_Error(
                'db_delete_error',
                __('Failed to delete segment.', 'mailerpress-pro'),
                ['status' => 500]
            );
        }

        // You can return 204 (no content) or 200 w/ message. I'll use 200 for clarity.
        return new \WP_REST_Response(
            [
                'id' => $id,
                'deleted' => true,
                'message' => __('Segment deleted successfully.', 'mailerpress-pro'),
            ],
            200
        );
    }
}
