<?php

namespace MailerPressPro\Api;

use MailerPressPro\Core\Attributes\Endpoint;
use MailerPressPro\Services\MailerPressLicence;
use WP_Error;
use WP_REST_Request;

class LicenceManager
{
    /**
     * Masque une clé de licence pour l'affichage sécurisé
     * Affiche seulement les 4 premiers et 4 derniers caractères
     */
    private function mask_license_key(?string $key): ?string
    {
        if (empty($key) || strlen($key) <= 8) {
            return $key;
        }

        $length = strlen($key);
        $visible_start = substr($key, 0, 4);
        $visible_end = substr($key, -4);
        $masked_length = $length - 8;
        $masked = str_repeat('*', $masked_length);

        return $visible_start . $masked . $visible_end;
    }

    #[Endpoint('license/get', methods: 'GET')]
    public function get(WP_REST_Request $request): WP_Error|\WP_REST_Response
    {
        $license = MailerPressLicence::instance();

        $status = $license->check_status();
        if (is_wp_error($status)) {
            return new WP_Error('license_error', $status->get_error_message(), ['status' => 500]);
        }

        $full_key = $license->get_license_key();

        return rest_ensure_response([
            'license_key' => $this->mask_license_key($full_key),
            'instance' => $license->get_instance_id(),
            'status' => $status,
        ]);
    }

    #[Endpoint('license/save', methods: 'POST')]
    public function save(WP_REST_Request $request): WP_Error|\WP_REST_Response
    {
        $license_key = sanitize_text_field($request->get_param('license_key'));
        if (!$license_key) {
            return new WP_Error('missing_key', 'License key is required.', ['status' => 400]);
        }

        $license = MailerPressLicence::instance();
        $license->set_license_key($license_key);

        return rest_ensure_response([
            'success' => true,
            'license_key' => $this->mask_license_key($license_key),
        ]);
    }

    #[Endpoint('license/activate', methods: 'POST')]
    public function activate(WP_REST_Request $request): WP_Error|\WP_REST_Response
    {
        $license = MailerPressLicence::instance();
        $result = $license->activate_license();

        if (is_wp_error($result)) {
            return new WP_Error('activation_failed', $result->get_error_message(), ['status' => 500]);
        }

        update_option('mailerpress_license_activated', true, false);

        return rest_ensure_response([
            'success' => true,
            'data' => $result,
        ]);
    }

    #[Endpoint('license/deactivate', methods: 'POST')]
    public function deactivate(WP_REST_Request $request): WP_Error|\WP_REST_Response
    {
        $license = MailerPressLicence::instance();
        $result = $license->deactivate_license();

        if (is_wp_error($result)) {
            return new WP_Error('deactivation_failed', $result->get_error_message(), ['status' => 500]);
        }

        // Supprimer l'option d'activation
        delete_option('mailerpress_license_activated');

        // Supprimer complètement la clé de licence de la base de données
        $license->set_license_key(null);

        return rest_ensure_response([
            'success' => true,
            'data' => $result,
        ]);
    }

    #[Endpoint('license/check-status', methods: 'GET')]
    public function status(WP_REST_Request $request): WP_Error|\WP_REST_Response
    {
        $license = MailerPressLicence::instance();
        $result = $license->check_status();


        if (is_wp_error($result)) {
            return new WP_Error('status_failed', $result->get_error_message(), ['status' => 500]);
        }

        return rest_ensure_response([
            'success' => true,
            'status' => $result,
        ]);
    }

    #[Endpoint('license/check-update', methods: 'GET')]
    public function checkUpdate(WP_REST_Request $request): WP_Error|\WP_REST_Response
    {
        $license = MailerPressLicence::instance();
        $result = $license->check_for_update();

        if (is_wp_error($result)) {
            return new WP_Error('update_failed', $result->get_error_message(), ['status' => 500]);
        }

        return rest_ensure_response([
            'success' => true,
            'update' => $result,
        ]);
    }
}
