<?php

declare(strict_types=1);

namespace MailerPressPro\Api;

\defined('ABSPATH') || exit;

use MailerPressPro\Core\Attributes\Endpoint;
use MailerPress\Core\EmailManager\EmailServiceManager;
use MailerPressPro\Api\Permissions;
use MailerPressPro\Core\Kernel;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * API endpoints pour la gestion des bounces des ESP
 */
class EspBounce
{
    /**
     * Récupère l'instance du EmailServiceManager depuis le container
     * 
     * @return EmailServiceManager
     */
    private function getEmailServiceManager(): EmailServiceManager
    {
        try {
            // Utiliser la fonction du plugin core pour récupérer le manager
            // car les services PRO sont enregistrés dans le manager du plugin core
            if (function_exists('mailerpress_get_provider_class')) {
                $manager = mailerpress_get_provider_class();
                $services = $manager->getServices();
                return $manager;
            }
            // Fallback : utiliser le container PRO
            return Kernel::getContainer()->get(EmailServiceManager::class);
        } catch (\Exception $e) {
            // Si le container n'est pas disponible, créer une nouvelle instance
            return new EmailServiceManager();
        }
    }
    /**
     * Endpoint pour recevoir les webhooks de bounce depuis Brevo
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/brevo
     */
    #[Endpoint(
        'esp/bounce/brevo',
        methods: 'POST',
        permissionCallback: '__return_true'
    )]
    public function receiveBrevoWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        return $this->processWebhook('brevo', $request);
    }

    /**
     * Endpoint pour recevoir les webhooks de bounce depuis Mailjet
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/mailjet
     */
    #[Endpoint(
        'esp/bounce/mailjet',
        methods: 'POST',
        permissionCallback: '__return_true'
    )]
    public function receiveMailjetWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        return $this->processWebhook('mailjet', $request);
    }

    /**
     * Endpoint pour recevoir les webhooks de bounce depuis PostMark
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/postmark
     */
    #[Endpoint(
        'esp/bounce/postmark',
        methods: 'POST',
        permissionCallback: '__return_true'
    )]
    public function receivePostmarkWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        return $this->processWebhook('postmark', $request);
    }

    /**
     * Endpoint pour recevoir les webhooks de bounce depuis SendGrid
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/sendgrid
     */
    #[Endpoint(
        'esp/bounce/sendgrid',
        methods: 'POST',
        permissionCallback: '__return_true'
    )]
    public function receiveSendgridWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        return $this->processWebhook('sendgrid', $request);
    }

    /**
     * Endpoint pour recevoir les webhooks de bounce depuis Mailgun
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/mailgun
     */
    #[Endpoint(
        'esp/bounce/mailgun',
        methods: 'POST',
        permissionCallback: '__return_true'
    )]
    public function receiveMailgunWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        return $this->processWebhook('mailgun', $request);
    }

    /**
     * Endpoint pour recevoir les webhooks de bounce depuis Amazon SES (via SNS)
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/amazonses
     */
    #[Endpoint(
        'esp/bounce/amazonses',
        methods: 'POST',
        permissionCallback: '__return_true'
    )]
    public function receiveAmazonSESWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        return $this->processWebhook('amazonses', $request);
    }

    /**
     * Traite un webhook de bounce
     * 
     * @param string $espName Le nom de l'ESP
     * @param WP_REST_Request $request La requête
     * @return WP_REST_Response|WP_Error
     */
    private function processWebhook(string $espName, WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {

            $serviceManager = $this->getEmailServiceManager();

            // Vérifier les services disponibles pour le debug
            $availableServices = array_keys($serviceManager->getServices());

            // Utiliser getServiceByKey au lieu de getActiveServiceByKey
            try {
                $service = $serviceManager->getServiceByKey($espName);

                // Logger le service récupéré pour vérification
                if ($service && method_exists($service, 'config')) {
                    $serviceConfig = $service->config();
                    $serviceKey = $serviceConfig['key'] ?? 'unknown';
                    $serviceName = $serviceConfig['name'] ?? 'unknown';
                }
            } catch (\Exception $e) {
                return new WP_Error(
                    'esp_not_supported',
                    sprintf(__('ESP %s is not supported', 'mailerpress-pro'), $espName),
                    ['status' => 400]
                );
            }

            if (!$service || !($service instanceof \MailerPressPro\Core\EmailManager\BounceDetectionInterface)) {
                return new WP_Error(
                    'esp_not_supported',
                    sprintf(__('ESP %s is not supported or does not implement bounce detection', 'mailerpress-pro'), $espName),
                    ['status' => 400]
                );
            }

            // Récupérer les données du webhook
            // Mailjet envoie les données dans le body JSON brut
            $body = $request->get_body();

            // Pour Amazon SES via SNS, la signature est dans le payload JSON, pas dans les en-têtes
            // On doit parser le body d'abord pour extraire la signature
            $payloadForSignature = null;
            if ($espName === 'amazonses') {
                $decodedForSignature = json_decode($body, true);
                if ($decodedForSignature && isset($decodedForSignature['Signature'])) {
                    $payloadForSignature = $decodedForSignature;
                }
            }

            // Vérifier la signature si nécessaire
            $signature = null;
            if ($espName === 'amazonses' && $payloadForSignature) {
                // Pour Amazon SES, la signature est dans le payload JSON
                $signature = $payloadForSignature['Signature'] ?? null;
            } else {
                // Pour les autres services, la signature est dans les en-têtes HTTP
                $signature = $request->get_header('X-Brevo-Signature')
                    ?? $request->get_header('X-Mailjet-Signature')
                    ?? $request->get_header('X-Postmark-Signature')
                    ?? $request->get_header('X-Twilio-Email-Event-Webhook-Signature')
                    ?? $request->get_param('signature');
            }

            if (!empty($signature)) {
                // Pour Amazon SES, on passe le payload JSON complet pour la vérification
                $bodyForVerification = $espName === 'amazonses' && $payloadForSignature
                    ? json_encode($payloadForSignature)
                    : $body;

                if (!$service->verifyWebhookSignature($bodyForVerification, $signature)) {
                    // Formater la signature pour le log (peut être string ou array)
                    $signatureLog = is_array($signature) ? json_encode($signature) : substr($signature, 0, 20) . '...';
                    return new WP_Error(
                        'invalid_signature',
                        __('Invalid webhook signature', 'mailerpress-pro'),
                        ['status' => 401]
                    );
                }
            }

            // Récupérer les données du webhook
            // Mailjet envoie les données dans le body JSON brut
            $body = $request->get_body();

            $payload = null;

            // Essayer d'abord get_json_params() (WordPress parse automatiquement le JSON)
            $payload = $request->get_json_params();

            // Si vide, essayer get_body_params() (pour les données form-urlencoded)
            if (empty($payload)) {
                $payload = $request->get_body_params();
            }

            // Si toujours vide, parser manuellement le body JSON
            if (empty($payload) && !empty($body)) {
                $decoded = json_decode($body, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                    $payload = $decoded;
                } else {
                    // Si le JSON n'est pas valide, essayer de parser comme form-data
                    parse_str($body, $parsed);
                    if (!empty($parsed)) {
                        $payload = $parsed;
                    }
                }
            }

            // Mailjet peut aussi envoyer les données dans un tableau d'événements
            // Vérifier si le payload est un tableau avec un seul élément
            if (!empty($payload) && isset($payload[0]) && is_array($payload[0])) {
                // C'est déjà un tableau d'événements, on le garde tel quel
            } elseif (!empty($payload) && !isset($payload[0]) && isset($payload['event'])) {
                // C'est un seul événement, on le garde tel quel
            } elseif (!empty($payload) && is_array($payload)) {
                // Vérifier si les données sont dans une clé spécifique
                // Mailjet peut parfois wrapper les données
                if (isset($payload['event'])) {
                    // Les données sont au niveau racine, c'est bon
                } elseif (isset($payload['data']) && is_array($payload['data'])) {
                    // Les données sont dans une clé 'data'
                    $payload = $payload['data'];
                } elseif (isset($payload['payload']) && is_array($payload['payload'])) {
                    // Les données sont dans une clé 'payload'
                    $payload = $payload['payload'];
                }
            }

            // SendGrid envoie toujours un tableau d'événements, même pour un seul événement
            // Le service processBounceWebhook gère déjà ce cas
            // On passe le payload tel quel, le service décidera comment le traiter

            if (empty($payload)) {
                return new WP_Error(
                    'invalid_payload',
                    __('Invalid webhook payload', 'mailerpress-pro'),
                    ['status' => 400]
                );
            }

            $payloadSummary = $this->getPayloadSummary($payload, $espName);

            if ($espName === 'mailjet') {
                $isTestWebhook = false;

                // Vérifier si c'est un tableau d'événements
                if (isset($payload[0])) {
                    // Si c'est un tableau, vérifier chaque événement
                    foreach ($payload as $eventPayload) {
                        if (
                            empty($eventPayload['email']) &&
                            (empty($eventPayload['MessageID']) || $eventPayload['MessageID'] == 0) &&
                            (empty($eventPayload['mj_contact_id']) || $eventPayload['mj_contact_id'] == 0)
                        ) {
                            $isTestWebhook = true;
                            break;
                        }
                    }
                } else {
                    // Vérifier si c'est un événement unique avec des données vides
                    if (
                        empty($payload['email']) &&
                        (empty($payload['MessageID']) || $payload['MessageID'] == 0) &&
                        (empty($payload['mj_contact_id']) || $payload['mj_contact_id'] == 0) &&
                        empty($payload['Message_GUID'])
                    ) {
                        $isTestWebhook = true;
                    }
                }

                if ($isTestWebhook) {
                    return new \WP_REST_Response([
                        'success' => true,
                        'message' => 'Test webhook received and ignored',
                        'event' => $payload['event'] ?? (isset($payload[0]) ? $payload[0]['event'] ?? 'unknown' : 'unknown'),
                    ], 200);
                }
            }

            $result = $service->processBounceWebhook($payload);

            // Pour Amazon SES via SNS, gérer spécialement la confirmation de souscription
            if ($espName === 'amazonses' && isset($payload['Type']) && $payload['Type'] === 'SubscriptionConfirmation') {
                if ($result) {
                    error_log('[MailerPress Pro] Amazon SES SNS subscription confirmed successfully');
                    // SNS s'attend à recevoir le contenu de la page de confirmation
                    // On retourne une réponse simple qui indique que la confirmation a été effectuée
                    return new WP_REST_Response([
                        'success' => true,
                        'message' => __('SNS subscription confirmed successfully', 'mailerpress-pro'),
                        'type' => 'SubscriptionConfirmation',
                    ], 200);
                } else {
                    error_log('[MailerPress Pro] Amazon SES SNS subscription confirmation failed');
                    return new WP_Error(
                        'subscription_confirmation_failed',
                        __('Failed to confirm SNS subscription', 'mailerpress-pro'),
                        ['status' => 500]
                    );
                }
            }

            if (!$result) {
                return new WP_Error(
                    'processing_failed',
                    __('Failed to process bounce webhook', 'mailerpress-pro'),
                    ['status' => 500]
                );
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => __('Bounce processed successfully', 'mailerpress-pro'),
            ], 200);
        } catch (\Exception $e) {
            return new WP_Error(
                'exception',
                $e->getMessage(),
                ['status' => 500]
            );
        }
    }

    /**
     * Génère un résumé du payload pour les logs
     * 
     * @param array $payload Le payload du webhook
     * @param string $espName Le nom de l'ESP
     * @return string Résumé du payload
     */
    private function getPayloadSummary(array $payload, string $espName): string
    {
        // SendGrid envoie un tableau d'événements
        if ($espName === 'sendgrid' && isset($payload[0])) {
            $count = count($payload);
            $events = [];
            foreach ($payload as $event) {
                $eventType = $event['event'] ?? 'unknown';
                $email = $event['email'] ?? 'unknown';
                $events[] = sprintf('%s:%s', $eventType, substr($email, 0, 20));
            }
            return sprintf('Array[%d events]: %s', $count, implode(', ', array_slice($events, 0, 3)));
        }

        // Pour les autres ESP, extraire les informations principales
        $event = $payload['event'] ?? $payload['RecordType'] ?? $payload['EventType'] ?? 'unknown';
        $email = $payload['email'] ?? $payload['Email'] ?? 'unknown';
        $reason = $payload['reason'] ?? $payload['Description'] ?? $payload['error'] ?? '';

        return sprintf('event:%s, email:%s, reason:%s', $event, substr($email, 0, 30), substr($reason, 0, 50));
    }


    /**
     * Endpoint pour obtenir les capacités de détection de bounce d'un ESP
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/capabilities
     */
    #[Endpoint(
        'esp/bounce/capabilities',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function getCapabilities(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {
            $espName = $request->get_param('esp');

            if (empty($espName)) {
                return new WP_Error(
                    'missing_esp',
                    __('ESP name is required', 'mailerpress-pro'),
                    ['status' => 400]
                );
            }

            $serviceManager = $this->getEmailServiceManager();

            // Utiliser getServiceByKey au lieu de getActiveServiceByKey pour éviter d'avoir besoin d'une connexion
            try {
                $service = $serviceManager->getServiceByKey($espName);
            } catch (\Exception $e) {
                // Le service n'est pas enregistré
                return new WP_REST_Response([
                    'success' => true,
                    'capabilities' => [
                        'webhook' => false,
                        'configured' => false,
                    ],
                ], 200);
            }

            if (!$service) {
                return new WP_REST_Response([
                    'success' => true,
                    'capabilities' => [
                        'webhook' => false,
                        'configured' => false,
                    ],
                ], 200);
            }

            if (!($service instanceof \MailerPressPro\Core\EmailManager\BounceDetectionInterface)) {
                return new WP_REST_Response([
                    'success' => true,
                    'capabilities' => [
                        'webhook' => false,
                        'configured' => false,
                    ],
                ], 200);
            }

            $capabilities = [
                'webhook' => $service->supportsWebhookBounceDetection(),
                'configured' => true,
                'webhook_url' => $service->getBounceWebhookUrl(),
            ];

            return new WP_REST_Response([
                'success' => true,
                'capabilities' => $capabilities,
            ], 200);
        } catch (\Exception $e) {
            return new WP_Error(
                'exception',
                $e->getMessage(),
                ['status' => 500]
            );
        }
    }

    /**
     * Endpoint pour configurer le webhook de bounce d'un ESP
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/configure
     */
    #[Endpoint(
        'esp/bounce/configure',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function configureWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {
            $espName = $request->get_param('esp');
            $webhookUrl = $request->get_param('webhook_url');

            if (empty($espName)) {
                return new WP_Error(
                    'missing_esp',
                    __('ESP name is required', 'mailerpress-pro'),
                    ['status' => 400]
                );
            }

            if (empty($webhookUrl)) {
                return new WP_Error(
                    'missing_webhook_url',
                    __('Webhook URL is required', 'mailerpress-pro'),
                    ['status' => 400]
                );
            }

            $serviceManager = $this->getEmailServiceManager();

            // Utiliser getServiceByKey au lieu de getActiveServiceByKey
            try {
                $service = $serviceManager->getServiceByKey($espName);
            } catch (\Exception $e) {
                return new WP_Error(
                    'esp_not_supported',
                    sprintf(__('ESP %s is not supported', 'mailerpress-pro'), $espName),
                    ['status' => 400]
                );
            }

            if (!$service || !($service instanceof \MailerPressPro\Core\EmailManager\BounceDetectionInterface)) {
                return new WP_Error(
                    'esp_not_supported',
                    sprintf(__('ESP %s is not supported or does not implement bounce detection', 'mailerpress-pro'), $espName),
                    ['status' => 400]
                );
            }

            if (!$service->supportsWebhookBounceDetection()) {
                return new WP_Error(
                    'webhook_not_supported',
                    sprintf(__('ESP %s does not support webhook bounce detection', 'mailerpress-pro'), $espName),
                    ['status' => 400]
                );
            }

            // Récupérer la clé API
            $servicesConfig = get_option('mailerpress_email_services', []);
            $serviceConfig = $servicesConfig['services'][$espName]['conf'] ?? [];
            $apiKey = $serviceConfig['api_key'] ?? $serviceConfig['apiKey'] ?? '';

            if (empty($apiKey)) {
                return new WP_Error(
                    'api_key_missing',
                    __('API key is missing for this ESP', 'mailerpress-pro'),
                    ['status' => 400]
                );
            }

            $result = $service->configureBounceWebhook($apiKey, $webhookUrl);

            if ($result === false) {
                return new WP_Error(
                    'configuration_failed',
                    __('Failed to configure bounce webhook. Please check your API key is valid and has the necessary permissions. Check PHP error logs for more details.', 'mailerpress-pro'),
                    ['status' => 500]
                );
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => __('Bounce webhook configured successfully', 'mailerpress-pro'),
                'webhook' => $result,
            ], 200);
        } catch (\Exception $e) {
            return new WP_Error(
                'exception',
                $e->getMessage(),
                ['status' => 500]
            );
        }
    }

    /**
     * Endpoint pour supprimer le webhook de bounce d'un ESP
     * 
     * Route: /wp-json/mailerpress/v1/esp/bounce/remove
     */
    #[Endpoint(
        'esp/bounce/remove',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function removeWebhook(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        try {
            $espName = $request->get_param('esp');

            if (empty($espName)) {
                return new WP_Error(
                    'missing_esp',
                    __('ESP name is required', 'mailerpress-pro'),
                    ['status' => 400]
                );
            }

            $serviceManager = $this->getEmailServiceManager();

            // Utiliser getServiceByKey au lieu de getActiveServiceByKey
            try {
                $service = $serviceManager->getServiceByKey($espName);
            } catch (\Exception $e) {
                return new WP_Error(
                    'esp_not_supported',
                    sprintf(__('ESP %s is not supported', 'mailerpress-pro'), $espName),
                    ['status' => 400]
                );
            }

            if (!$service || !($service instanceof \MailerPressPro\Core\EmailManager\BounceDetectionInterface)) {
                return new WP_Error(
                    'esp_not_supported',
                    sprintf(__('ESP %s is not supported or does not implement bounce detection', 'mailerpress-pro'), $espName),
                    ['status' => 400]
                );
            }

            // Récupérer la clé API
            $servicesConfig = get_option('mailerpress_email_services', []);
            $serviceConfig = $servicesConfig['services'][$espName]['conf'] ?? [];
            $apiKey = $serviceConfig['api_key'] ?? $serviceConfig['apiKey'] ?? '';

            if (empty($apiKey)) {
                return new WP_Error(
                    'api_key_missing',
                    __('API key is missing for this ESP', 'mailerpress-pro'),
                    ['status' => 400]
                );
            }

            $result = $service->removeBounceWebhook($apiKey);

            if (!$result) {
                return new WP_Error(
                    'removal_failed',
                    __('Failed to remove bounce webhook', 'mailerpress-pro'),
                    ['status' => 500]
                );
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => __('Bounce webhook removed successfully', 'mailerpress-pro'),
            ], 200);
        } catch (\Exception $e) {
            return new WP_Error(
                'exception',
                $e->getMessage(),
                ['status' => 500]
            );
        }
    }
}
