<?php

namespace MailerPressPro\Actions\ThirdParty\WooCommerce;

use MailerPressPro\Core\Attributes\Action;
use MailerPressPro\Database\Tables;

class TrackingRevenue
{
    #[Action('woocommerce_thankyou', priority: 10, acceptedArgs: 1)]
    public function saveTrackingOnThankYou($order_id)
    {
        if (!function_exists('WC') || !WC()->session) {
            return;
        }

        $tracking = WC()->session->get('mailerpress_product_click');

        if (empty($tracking) || empty($tracking['campaign_id']) || empty($tracking['product_id'])) {
            return;
        }

        $campaign_id = (int)$tracking['campaign_id'];
        $product_id = (int)$tracking['product_id'];

        // Save tracking info on the order meta
        update_post_meta($order_id, '_mailerpress_tracking', [
            'campaign_id' => $campaign_id,
            'product_id' => $product_id,
            'timestamp' => $tracking['timestamp'] ?? time(),
        ]);
    }

    #[Action('woocommerce_order_status_completed', priority: 10, acceptedArgs: 1)]
    public function trackRevenue($order_id)
    {
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        // Get tracking from order meta
        $tracking = get_post_meta($order_id, '_mailerpress_tracking', true)
            ?? WC()->session->get('mailerpress_product_click');

        if (empty($tracking) || empty($tracking['campaign_id']) || empty($tracking['product_id'])) {
            return;
        }

        $campaign_id = (int)$tracking['campaign_id'];
        $order_total = $order->get_total();
        $order_email = sanitize_email($order->get_billing_email());
        $now = current_time('mysql', 1);

        global $wpdb;
        $contactTable = $wpdb->prefix . 'mailerpress_contact';
        $statsTable = $wpdb->prefix . 'mailerpress_contact_stats';
        $campaignTable = $wpdb->prefix . 'mailerpress_campaign_stats';

        // Find contact by email
        $contact_id = (int)$wpdb->get_var(
            $wpdb->prepare("SELECT contact_id FROM {$contactTable} WHERE email = %s", $order_email)
        );

        if (!$contact_id) {
            return;
        }

        // Update or insert contact stats
        $existing = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT id, revenue FROM {$statsTable} WHERE contact_id = %d AND campaign_id = %d",
                $contact_id,
                $campaign_id
            )
        );

        if ($existing) {
            $wpdb->update(
                $statsTable,
                [
                    'revenue' => $existing->revenue + $order_total,
                    'updated_at' => $now,
                ],
                ['id' => $existing->id],
                ['%f', '%s'],
                ['%d']
            );
        } else {
            $wpdb->insert(
                $statsTable,
                [
                    'contact_id' => $contact_id,
                    'campaign_id' => $campaign_id,
                    'opened' => 0,
                    'clicked' => 0,
                    'click_count' => 0,
                    'last_click_at' => null,
                    'revenue' => $order_total,
                    'status' => 'good',
                    'created_at' => $now,
                    'updated_at' => $now,
                ],
                ['%d', '%d', '%d', '%d', '%d', '%s', '%f', '%s', '%s', '%s']
            );
        }

        // Update campaign-level revenue
        $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$campaignTable}
             SET total_revenue = total_revenue + %f, updated_at = %s
             WHERE campaign_id = %d",
                $order_total,
                $now,
                $campaign_id
            )
        );

        // Save MailerPress contact info on order for refund tracking
        update_post_meta($order_id, '_mailerpress_contact', [
            'contact_id' => $contact_id,
            'campaign_id' => $campaign_id,
        ]);

        delete_post_meta($order_id, '_mailerpress_tracking');

    }

    #[Action('woocommerce_order_refunded', priority: 10, acceptedArgs: 2)]
    public function handleRefund($order_id, $refund_id)
    {
        global $wpdb;

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        $refund = wc_get_order($refund_id);
        if (!$refund) {
            return;
        }

        $refund_amount = (float)$refund->get_amount();

        $contact_meta = get_post_meta($order_id, '_mailerpress_contact', true);
        if (empty($contact_meta)) {
            return;
        }

        $contact_id = (int)($contact_meta['contact_id'] ?? 0);
        $campaign_id = (int)($contact_meta['campaign_id'] ?? 0);

        if (!$contact_id || !$campaign_id) {
            return;
        }

        $contactTable = $wpdb->prefix . 'mailerpress_contact_stats';
        $campaignTable = $wpdb->prefix . 'mailerpress_campaign_stats';
        $now = current_time('mysql', 1);

        // ✅ Deduct refund safely
        $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$contactTable}
                 SET revenue = GREATEST(revenue - %f, 0),
                     updated_at = %s
                 WHERE contact_id = %d AND campaign_id = %d",
                $refund_amount,
                $now,
                $contact_id,
                $campaign_id
            )
        );

        $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$campaignTable}
                 SET total_revenue = GREATEST(total_revenue - %f, 0),
                     updated_at = %s
                 WHERE campaign_id = %d",
                $refund_amount,
                $now,
                $campaign_id
            )
        );

        delete_post_meta($order_id, '_mailerpress_contact');
    }
}
