<?php

namespace MailerPressPro\Actions\ThirdParty\WooCommerce;

use Exception;
use MailerPressPro\Core\Attributes\Action;
use MailerPressPro\Core\Attributes\Filter;
use Automattic\WooCommerce\Blocks\Package;
use Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields;

class CheckoutField
{
    // Mapping des positions pour le checkout classique (shortcode)
    private const CLASSIC_CHECKOUT_HOOKS = [
        'contact' => 'woocommerce_after_checkout_billing_form',
        'address' => 'woocommerce_after_order_notes',
        'order' => 'woocommerce_after_order_notes',
        'terms' => 'woocommerce_checkout_after_terms_and_conditions',
        'payment' => 'woocommerce_review_order_before_payment',
    ];

    #[Action('woocommerce_init')]
    public function mailerPress_checkout_field()
    {
        $mailerpress_settings = get_option('woocommerce_mailerpress_settings');

        if (is_string($mailerpress_settings)) {
            $mailerpress_settings = json_decode($mailerpress_settings, true);
        }


        if (false === $mailerpress_settings || empty($mailerpress_settings['enable_checkout_optin'])) {
            return;
        }

        // Support pour les blocs Gutenberg (nouveau checkout)
        // Safety: make sure the helper exists (older WooCommerce won't have it).
        if (function_exists('woocommerce_register_additional_checkout_field')) {
            try {
                woocommerce_register_additional_checkout_field(
                    array(
                        'id' => 'mailerpress/subscribe',
                        'label' => $mailerpress_settings['subscribe_label'] ?? __(
                            'Subscribe to our newsletter',
                            'mailerpress-pro'
                        ),
                        'location' => $mailerpress_settings['location'] ?? 'contact',
                        'type' => 'checkbox',
                        'required' => false,
                        'description' => __(
                            'Occasional product news & special offers. Unsubscribe anytime.',
                            'mailerpress-pro'
                        ),
                    )
                );
            } catch (\Exception $e) {
            }
        }

        // Support pour le checkout classique (shortcode)
        // Si location est "contact", on ajoute le champ directement après l'email via le filtre
        $location = $mailerpress_settings['location'] ?? 'contact';

        if ($location === 'contact') {
            // Ajouter le champ via le filtre pour qu'il apparaisse après l'email
            add_filter('woocommerce_checkout_fields', [$this, 'add_checkout_field_after_email'], 20);
        } else {
            // Pour les autres positions, utiliser les hooks
            $hook = self::CLASSIC_CHECKOUT_HOOKS[$location] ?? self::CLASSIC_CHECKOUT_HOOKS['address'];
            add_action($hook, [$this, 'render_classic_checkout_field'], 10);
        }
    }

    /**
     * Ajoute le champ opt-in directement après le champ email dans les champs de facturation
     */
    #[Filter('woocommerce_checkout_fields')]
    public function add_checkout_field_after_email($fields)
    {
        $mailerpress_settings = get_option('woocommerce_mailerpress_settings');

        if (is_string($mailerpress_settings)) {
            $mailerpress_settings = json_decode($mailerpress_settings, true);
        }

        if (false === $mailerpress_settings || empty($mailerpress_settings['enable_checkout_optin'])) {
            return $fields;
        }

        $label = $mailerpress_settings['subscribe_label'] ?? __('Subscribe to our newsletter', 'mailerpress-pro');
        $checked = !empty($_POST['mailerpress_subscribe']);

        // Ajouter le champ avec une priorité de 115 (juste après l'email qui est à 110)
        $fields['billing']['mailerpress_subscribe'] = [
            'type' => 'checkbox',
            'label' => $label,
            'required' => false,
            'class' => ['form-row-wide'],
            'priority' => 115, // Juste après billing_email (110)
            'default' => $checked ? 1 : 0,
        ];

        return $fields;
    }

    /**
     * Affiche le champ opt-in dans le checkout classique (shortcode) pour les autres positions
     */
    public function render_classic_checkout_field()
    {
        $mailerpress_settings = get_option('woocommerce_mailerpress_settings');

        if (is_string($mailerpress_settings)) {
            $mailerpress_settings = json_decode($mailerpress_settings, true);
        }

        if (false === $mailerpress_settings || empty($mailerpress_settings['enable_checkout_optin'])) {
            return;
        }

        $label = $mailerpress_settings['subscribe_label'] ?? __('Subscribe to our newsletter', 'mailerpress-pro');
        $checked = !empty($_POST['mailerpress_subscribe']);

?>
        <p class="form-row form-row-wide mailerpress-subscribe-field">
            <label class="woocommerce-form__label woocommerce-form__label-for-checkbox checkbox">
                <input
                    id="mailerpress_subscribe"
                    class="woocommerce-form__input woocommerce-form__input-checkbox input-checkbox"
                    type="checkbox"
                    name="mailerpress_subscribe"
                    value="1"
                    <?php checked($checked); ?> />
                <span><?php echo esc_html($label); ?></span>
            </label>
        </p>
<?php
    }

    /**
     * Sauvegarde la valeur du champ opt-in lors de la soumission du checkout classique
     */
    #[Action('woocommerce_checkout_update_order_meta', acceptedArgs: 2)]
    public function save_classic_checkout_optin($order_id, $data)
    {

        // Vérifier si le champ a été ajouté via woocommerce_checkout_fields (dans les champs de facturation)
        $opt_in = false;

        // Le champ peut être dans $data avec ou sans préfixe billing_
        if (isset($data['billing_mailerpress_subscribe'])) {
            $opt_in = (bool)$data['billing_mailerpress_subscribe'];
        }

        if (!$opt_in && isset($data['mailerpress_subscribe'])) {
            $opt_in = (bool)$data['mailerpress_subscribe'];
        }

        // Vérifier aussi dans $_POST (pour les champs ajoutés via hooks)
        if (!$opt_in && !empty($_POST['billing_mailerpress_subscribe'])) {
            $opt_in = true;
        }

        if (!$opt_in && !empty($_POST['mailerpress_subscribe'])) {
            $opt_in = true;
        }

        if ($opt_in) {
            $order = wc_get_order($order_id);
            if ($order) {
                // Sauvegarder dans les meta de la commande
                $order->update_meta_data('_mailerpress_subscribe', '1');
                $order->save();
            }
        }
    }

    #[Action('woocommerce_thankyou', acceptedArgs: 1)]
    public function mailepress_optin($order_id)
    {
        global $wpdb;

        $table_custom_fields = "{$wpdb->prefix}mailerpress_contact_custom_fields";

        /** @var \WC_Order $order */
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        // Vérifier l'opt-in depuis les blocs Gutenberg (nouveau checkout)
        $opt_in = false;
        try {
            if (class_exists('\Automattic\WooCommerce\Blocks\Package')) {
                $checkout_fields = Package::container()->get(CheckoutFields::class);
                $opt_in = (bool)$checkout_fields->get_field_from_object('mailerpress/subscribe', $order, 'other');
            }
        } catch (\Exception $e) {
            // Si les blocs ne sont pas disponibles, on continue avec le checkout classique
        }

        // Si pas d'opt-in depuis les blocs, vérifier le checkout classique (shortcode)
        if (!$opt_in) {
            // Vérifier toutes les clés possibles pour le champ opt-in
            $opt_in_value = null;

            // 1. Vérifier avec le préfixe _mailerpress_subscribe (méthode standard)
            $opt_in_value = $order->get_meta('_mailerpress_subscribe', true);

            // 2. Vérifier avec le préfixe _billing_mailerpress_subscribe (format WooCommerce pour champs billing)
            if (empty($opt_in_value)) {
                $opt_in_value = $order->get_meta('_billing_mailerpress_subscribe', true);
            }

            // 3. Vérifier sans préfixe (format direct)
            if (empty($opt_in_value)) {
                $opt_in_value = $order->get_meta('mailerpress_subscribe', true);
            }

            // 4. Vérifier billing_mailerpress_subscribe (sans underscore)
            if (empty($opt_in_value)) {
                $opt_in_value = $order->get_meta('billing_mailerpress_subscribe', true);
            }

            // 5. Lister toutes les meta pour déboguer
            $all_meta = $order->get_meta_data();
            $mailerpress_meta = [];
            foreach ($all_meta as $meta) {
                if (strpos($meta->key, 'mailerpress') !== false) {
                    $mailerpress_meta[$meta->key] = $meta->value;
                }
            }

            $opt_in = (bool)$opt_in_value;

        }

        if (!$opt_in) {
            return;
        }

        // CRITICAL: Use WordPress user login email, NOT billing or shipping email
        // This ensures the contact is linked to the user's WordPress account

        $customer_id = $order->get_customer_id();
        $email = '';

        // Get email from WordPress user account (user login email)
        if ($customer_id > 0) {
            $customer = get_userdata($customer_id);
            if ($customer && !empty($customer->user_email)) {
                $email = $customer->user_email;
            }
        }

        // Log all email sources for debugging
        $billing_email = $order->get_billing_email();
        $user_email = $email;
        // Validate email format
        $email = sanitize_email($email);
        if (!is_email($email)) {
            return;
        }

        if (empty($email)) {
           return;
        }

        // Load MailerPress settings
        $mailerpress_settings = get_option('woocommerce_mailerpress_settings');
        if (is_string($mailerpress_settings)) {
            $mailerpress_settings = json_decode($mailerpress_settings, true);
        }

        // Get lists from settings, or default list if none configured
        $subscribe_lists = $mailerpress_settings['subscribe_lists'] ?? [];

        // Validate that all lists exist in database
        if (!empty($subscribe_lists)) {
            $lists_table = $wpdb->prefix . 'mailerpress_lists';
            $valid_lists = [];

            foreach ($subscribe_lists as $list_id) {
                $list_id = (int)$list_id;
                $exists = $wpdb->get_var($wpdb->prepare(
                    "SELECT list_id FROM {$lists_table} WHERE list_id = %d",
                    $list_id
                ));

                if ($exists) {
                    $valid_lists[] = $list_id;
                }
            }

            $subscribe_lists = $valid_lists;
        }

        // If no valid lists, use default list
        if (empty($subscribe_lists)) {
            // Get default list from database
            $default_list_id = $wpdb->get_var(
                "SELECT list_id FROM {$wpdb->prefix}mailerpress_lists WHERE is_default = 1 LIMIT 1"
            );
            if ($default_list_id) {
                $subscribe_lists = [(int)$default_list_id];
            }
        }

        // Prepare main contact data
        $contact_data = [
            'opt_in_source' => 'woocommerce',
            'contactEmail' => $email,
            'contactFirstName' => $order->get_billing_first_name() ?: '',
            'contactLastName' => $order->get_billing_last_name() ?: '',
            'tags' => array_map(fn($id) => ['id' => $id], $mailerpress_settings['subscribe_tags'] ?? []),
            'lists' => array_map(fn($id) => ['id' => $id], $subscribe_lists),
        ];

        try {
            // Create or update main contact
            $result = add_mailerpress_contact($contact_data);

            // Get the actual contact ID
            $contact_id = $result['contact_id'] ?? ($result['data']['contact_id'] ?? 0);

            if (!$contact_id) {
               return; // stop here if contact wasn't created
            }

            // Verify the contact was created with the correct email
            $contactModel = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Contacts::class);
            $created_contact = $contactModel->getContactByEmail($email);

            // Get field mapping (Woo → MailerPress)
            $field_mapping = $mailerpress_settings['field_mapping'] ?? [];

            // Fetch all order meta + standard fields dynamically
            $order_meta_values = [];
            foreach ($order->get_meta_data() as $meta) {
                $order_meta_values[$meta->key] = $meta->value;
            }

            // Get all checkout fields
            $checkout_fields = WC()->checkout()->get_checkout_fields();

            $standard_fields = [];

            if (isset($checkout_fields['billing'])) {
                $standard_fields = array_merge($standard_fields, array_keys($checkout_fields['billing']));
            }

            if (isset($checkout_fields['shipping'])) {
                $standard_fields = array_merge($standard_fields, array_keys($checkout_fields['shipping']));
            }


            foreach ($standard_fields as $field) {
                $getter = 'get_' . $field;
                if (method_exists($order, $getter)) {
                    $order_meta_values[$field] = $order->$getter();
                }
            }

            // Insert/update MailerPress custom fields dynamically
            if ($contact_id) {
                foreach ($field_mapping as $map) {
                    $wc_field = $map['woo_field'] ?? '';
                    $custom_field_key = $map['custom_field'] ?? '';

                    if (!$wc_field || !$custom_field_key) {
                        continue;
                    }

                    $value = $order_meta_values[$wc_field] ?? '';

                    if ($value !== '') {
                        $existing = $wpdb->get_var($wpdb->prepare(
                            "SELECT field_id FROM {$table_custom_fields} WHERE contact_id = %d AND field_key = %s",
                            (int)$contact_id,
                            $custom_field_key
                        ));

                        if ($existing) {
                            $wpdb->update(
                                $table_custom_fields,
                                ['field_value' => $value],
                                ['field_id' => $existing],
                                ['%s'],
                                ['%d']
                            );
                        } else {
                            $wpdb->insert(
                                $table_custom_fields,
                                [
                                    'contact_id' => $contact_id,
                                    'field_key' => $custom_field_key,
                                    'field_value' => $value
                                ],
                                ['%d', '%s', '%s']
                            );
                        }
                    }
                }
            }
        } catch (\Exception) {
        }
    }

    /**
     * Get field dynamically from order
     * Works for:
     * - Standard WooCommerce fields
     * - Shipping/billing fields
     * - Any custom checkout fields saved in order meta
     */
    private function get_order_field_dynamic(\WC_Order $order, string $field)
    {
        // 1. Try standard getter
        $getter = 'get_' . $field;
        if (method_exists($order, $getter)) {
            return $order->{$getter}();
        }

        // 2. Try order meta
        $meta_value = $order->get_meta($field, true);
        if ($meta_value !== '') {
            return $meta_value;
        }

        // 3. Check checkout fields (for custom fields added by plugins)
        $checkout_fields = WC()->checkout->get_checkout_fields();
        foreach ($checkout_fields as $section => $fields) {
            if (isset($fields[$field])) {
                return $order->get_meta($field, true);
            }
        }

        return '';
    }
}
