<?php

namespace MailerPressPro\Actions\ThirdParty\Elementor;

use MailerPressPro\Core\Attributes\Action;
use MailerPressPro\Core\Kernel;

class Setup
{
    #[Action('elementor/widgets/register')]
    public function registerWidgets($widgets_manager)
    {
        if (!did_action('elementor/loaded')) {
            return;
        }

        require_once Kernel::$config['root'] . '/src/Services/ThirdParty/MailerPressElementorWidget.php';
        $widgets_manager->register(new \MailerPressPro\Services\ThirdParty\MailerPressElementorWidget());
    }

    #[Action('elementor_pro/init')]
    public function registerFormAction()
    {
        if (!did_action('elementor/loaded')) {
            return;
        }

        // Check if Elementor Pro is active and Forms module exists
        if (!class_exists('\ElementorPro\Plugin')) {
            return;
        }

        $forms_module = \ElementorPro\Plugin::instance()->modules_manager->get_modules('forms');
        
        if (!$forms_module) {
            return;
        }

        require_once Kernel::$config['root'] . '/src/Services/ThirdParty/MailerPressElementorFormAction.php';
        $forms_module->add_form_action('mailerpress', new \MailerPressPro\Services\ThirdParty\MailerPressElementorFormAction());
    }

    #[Action(['wp_ajax_mailerpress_elementor_submit', 'wp_ajax_nopriv_mailerpress_elementor_submit'])]
    public function handleFormSubmit()
    {
        // Verify nonce
        if (!isset($_POST['mailerpress_optin_nonce']) || !wp_verify_nonce($_POST['mailerpress_optin_nonce'], 'mailerpress_optin_submit')) {
            wp_send_json_error(['message' => __('Security check failed.', 'mailerpress-pro')]);
            return;
        }

        if (!function_exists('add_mailerpress_contact')) {
            wp_send_json_error(['message' => __('MailerPress plugin is not active.', 'mailerpress-pro')]);
            return;
        }

        $widget_id = sanitize_text_field($_POST['widget_id'] ?? '');
        $element_id = sanitize_text_field($_POST['element_id'] ?? '');
        $page_id = intval($_POST['page_id'] ?? 0);

        if (empty($widget_id) && empty($element_id)) {
            wp_send_json_error(['message' => __('Invalid widget ID.', 'mailerpress-pro')]);
            return;
        }

        // Try to get settings from POST first (sent as JSON)
        $widget_settings_json = sanitize_text_field($_POST['widget_settings'] ?? '');
        $widget_settings = null;

        if (!empty($widget_settings_json)) {
            $decoded = json_decode(stripslashes($widget_settings_json), true);
            if (is_array($decoded)) {
                $widget_settings = $decoded;
            }
        }

        // Fallback: Get widget settings from Elementor using API
        if (empty($widget_settings) && $page_id > 0 && !empty($element_id) && class_exists('\Elementor\Plugin')) {
            $document = \Elementor\Plugin::$instance->documents->get($page_id);
            if ($document) {
                $element_data = $document->get_elements_data();
                $widget_settings = $this->findWidgetSettings($element_data, $element_id);
            }
        }

        // Last fallback: Get from post meta
        if (empty($widget_settings) && $page_id > 0) {
            $elementor_data = get_post_meta($page_id, '_elementor_data', true);
            if (!empty($elementor_data)) {
                $elementor_data = json_decode($elementor_data, true);
                $widget_settings = $this->findWidgetSettings($elementor_data, $element_id ?: $widget_id);
            }
        }

        if (empty($widget_settings)) {
            wp_send_json_error(['message' => __('Widget settings not found.', 'mailerpress-pro')]);
            return;
        }

        // Prepare contact data
        $contact_data = [];
        $form_fields = $widget_settings['form_fields'] ?? [];
        $field_mapping = $widget_settings['field_mapping'] ?? [];
        $custom_fields_mapping = $widget_settings['custom_fields_mapping'] ?? [];

        // Get email (required)
        $email = '';
        foreach ($form_fields as $field) {
            $field_name = !empty($field['field_name']) ? $field['field_name'] : '';
            if ($field['field_type'] === 'email' && isset($_POST[$field_name])) {
                $email = sanitize_email($_POST[$field_name]);
                break;
            }
        }

        // Also check if email is directly in POST
        if (empty($email) && isset($_POST['email'])) {
            $email = sanitize_email($_POST['email']);
        }

        if (empty($email) || !is_email($email)) {
            wp_send_json_error(['message' => __('Valid email address is required.', 'mailerpress-pro')]);
            return;
        }

        $contact_data['contactEmail'] = $email;

        // Map standard fields
        $default_mapping = [
            'email' => 'contactEmail',
            'firstname' => 'contactFirstName',
            'first_name' => 'contactFirstName',
            'lastname' => 'contactLastName',
            'last_name' => 'contactLastName',
        ];

        foreach ($form_fields as $field) {
            $field_name = !empty($field['field_name']) ? $field['field_name'] : '';
            if (empty($field_name) || !isset($_POST[$field_name])) {
                continue;
            }

            $field_value = sanitize_text_field($_POST[$field_name]);
            if (empty($field_value)) {
                continue;
            }

            // Check custom field mapping first
            $mapped_to = null;
            foreach ($field_mapping as $mapping) {
                if ($mapping['form_field'] === $field_name && !empty($mapping['mailerpress_field'])) {
                    $mapped_to = $mapping['mailerpress_field'];
                    break;
                }
            }

            // If no custom mapping, use default mapping
            if (empty($mapped_to) && isset($default_mapping[$field_name])) {
                $mapped_to = $default_mapping[$field_name];
            }

            if ($mapped_to && $mapped_to !== 'contactEmail') {
                $contact_data[$mapped_to] = $field_value;
            }
        }

        // Map custom fields
        $custom_fields = [];
        foreach ($custom_fields_mapping as $mapping) {
            $form_field = $mapping['form_field_custom'] ?? '';
            $custom_field_key = $mapping['custom_field_key'] ?? '';

            if (empty($form_field) || empty($custom_field_key)) {
                continue;
            }

            if (isset($_POST[$form_field])) {
                $field_value = sanitize_text_field($_POST[$form_field]);
                if (!empty($field_value)) {
                    $custom_fields[$custom_field_key] = $field_value;
                }
            }
        }

        if (!empty($custom_fields)) {
            $contact_data['custom_fields'] = $custom_fields;
        }

        // Add lists (now supports multiple)
        $lists = $widget_settings['mailerpress_list'] ?? [];
        if (!empty($lists)) {
            if (is_string($lists)) {
                // Single value (backward compatibility)
                $lists = [$lists];
            }
            if (is_array($lists) && !empty($lists)) {
                $contact_data['lists'] = array_map(function ($list_id) {
                    return ['id' => intval($list_id)];
                }, array_filter($lists));
            }
        }
        
        // If no lists provided, the default list will be assigned automatically

        // Add tags
        $tags = $widget_settings['mailerpress_tags'] ?? [];
        if (!empty($tags) && is_array($tags)) {
            $contact_data['tags'] = array_map(function ($tag_id) {
                return ['id' => intval($tag_id)];
            }, array_filter($tags));
        }

        // Check if contact already exists
        $existing_contact = null;
        if (function_exists('mailerpress_get_contact_by_email')) {
            $existing_contact = mailerpress_get_contact_by_email($email);
        } else {
            // Fallback: check via model
            $contactModel = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Contacts::class);
            $existing_contact = $contactModel->getContactByEmail($email);
        }

        // Subscription status
        // Only set subscription status if contact doesn't exist or is not already subscribed
        // If contact exists and is already subscribed, preserve the status and only update lists/tags/custom fields
        if (!$existing_contact || $existing_contact->subscription_status !== 'subscribed') {
            // Check global MailerPress signup confirmation setting
            $subscriptionOption = get_option('mailerpress_signup_confirmation', null);
            if (null !== $subscriptionOption) {
                $subscriptionOption = json_decode($subscriptionOption, true);
                // Vérifier que json_decode a retourné un tableau valide
                if (!is_array($subscriptionOption)) {
                    $subscriptionOption = null;
                }
            }

            $enableSignupConfirmation = false;
            if (is_array($subscriptionOption) && isset($subscriptionOption['enableSignupConfirmation'])) {
                $enableSignupConfirmation = (bool)$subscriptionOption['enableSignupConfirmation'];
            }

            // Get double opt-in setting from widget (checkbox, checked by default)
            $doubleOptin = $widget_settings['double_optin'] ?? 'yes';
            $confirmOption = ($doubleOptin === 'yes') ? '1' : '';

            // Determine subscription status based on widget settings
            // Widget settings are prioritized: if double opt-in is checked in widget, use pending
            $desiredStatus = $confirmOption === "1" ? 'pending' : 'subscribed';
            $contact_data['contactStatus'] = $desiredStatus;

            // Also set subscription_status for compatibility (some integrations use this)
            $contact_data['subscription_status'] = $desiredStatus;

            // Add opt-in source
            // First check if a custom optin_source is set in widget settings
            $custom_optin_source = !empty($widget_settings['optin_source']) ? sanitize_text_field($widget_settings['optin_source']) : '';
            
            if (!empty($custom_optin_source)) {
                // Use custom opt-in source from settings
                $contact_data['opt_in_source'] = $custom_optin_source;
            } else {
                // Use automatic logic based on double opt-in setting
                // If widget double opt-in is unchecked (user wants subscribed directly):
                // - Use 'manual' to force the status to 'subscribed' (prevents API from overriding)
                // - This ensures widget settings are always respected
                // If widget double opt-in is checked (user wants pending with email):
                // - Use 'elementor' so that ContactCreated hook sends the confirmation email
                // - The API will respect 'pending' status if global double opt-in is enabled
                if ($desiredStatus === 'subscribed') {
                    // User wants subscribed directly, use 'manual' to force it
                    $contact_data['opt_in_source'] = 'manual';
                } else {
                    // User wants pending with confirmation email, use 'elementor' to trigger email
                    $contact_data['opt_in_source'] = 'elementor';
                }
            }
        } else {
            // Contact exists and is already subscribed - don't change status
            // Only update lists, tags, and custom fields
            // Explicitly set to null so updateContact won't change the status
            $contact_data['contactStatus'] = null;
            $contact_data['subscription_status'] = null;
        }

        $contact_data['optin_details'] = wp_json_encode([
            'widget_id' => $widget_id,
            'page_id' => get_the_ID(),
        ]);

        // Add contact to MailerPress
        $result = add_mailerpress_contact($contact_data);

        if (isset($result['success']) && $result['success']) {
            $success_message = $widget_settings['success_message'] ?? __('Thank you for subscribing!', 'mailerpress-pro');
            wp_send_json_success(['message' => $success_message]);
        } else {
            $error_message = isset($result['error']) ? $result['error'] : __('Failed to add contact to MailerPress.', 'mailerpress-pro');
            wp_send_json_error(['message' => $error_message]);
        }
    }

    private function findWidgetSettings($elementor_data, $element_id)
    {
        if (!is_array($elementor_data)) {
            return null;
        }

        foreach ($elementor_data as $element) {
            // Check if this is our widget
            if (isset($element['widgetType']) && $element['widgetType'] === 'mailerpress-optin') {
                // Match by element ID (preferred) or widget ID
                if ((isset($element['id']) && $element['id'] === $element_id) ||
                    (isset($element['settings']['_element_id']) && $element['settings']['_element_id'] === $element_id)
                ) {
                    return $element['settings'] ?? null;
                }
            }

            // Recursively search in nested elements
            if (isset($element['elements']) && is_array($element['elements'])) {
                $found = $this->findWidgetSettings($element['elements'], $element_id);
                if ($found !== null) {
                    return $found;
                }
            }
        }

        return null;
    }
}
