<?php

namespace MailerPressPro\Actions\ThirdParty\Bricks;

use MailerPressPro\Core\Attributes\Action;
use MailerPressPro\Core\Kernel;

class Setup
{
    #[Action('init', priority: 11)]
    public function registerElement()
    {
        $this->doRegisterElement();
    }

    private function doRegisterElement()
    {
        // Static flag to prevent multiple registrations
        static $registered = false;
        if ($registered) {
            return;
        }

        // Wait for Bricks to be loaded
        if (!class_exists('\Bricks\Element') || !class_exists('\Bricks\Elements')) {
            return;
        }

        // Check if Bricks is actually active
        if (!defined('BRICKS_VERSION')) {
            return;
        }

        $element_file = Kernel::$config['root'] . '/src/Services/ThirdParty/MailerPressBricksElement.php';

        // Check if file exists
        if (!file_exists($element_file)) {
            return;
        }

        // Register the element - Bricks requires the file path, not the class name
        // According to: https://academy.bricksbuilder.io/article/create-your-own-elements/
        if (method_exists('\Bricks\Elements', 'register_element')) {
            try {
                // register_element accepts: $file (required), $name (optional), $element_class (optional)
                \Bricks\Elements::register_element(
                    $element_file,
                    'mailerpress-optin', // element name
                    'MailerPressPro\Services\ThirdParty\MailerPressBricksElement' // class name
                );
                $registered = true;
            } catch (\Exception) {
                // Log error for debugging
            }
        }
    }

    #[Action(['wp_ajax_mailerpress_bricks_submit', 'wp_ajax_nopriv_mailerpress_bricks_submit'])]
    public function handleFormSubmit()
    {
        // Verify nonce
        if (!isset($_POST['mailerpress_optin_nonce']) || !wp_verify_nonce($_POST['mailerpress_optin_nonce'], 'mailerpress_optin_submit')) {
            wp_send_json_error(['message' => __('Security check failed.', 'mailerpress-pro')]);
            return;
        }

        if (!function_exists('add_mailerpress_contact')) {
            wp_send_json_error(['message' => __('MailerPress plugin is not active.', 'mailerpress-pro')]);
            return;
        }

        $element_id = sanitize_text_field($_POST['element_id'] ?? '');
        $page_id = intval($_POST['page_id'] ?? 0);

        if (empty($element_id)) {
            wp_send_json_error(['message' => __('Invalid element ID.', 'mailerpress-pro')]);
            return;
        }

        // Try to get settings from POST first (sent as JSON)
        // Use same method as Elementor for consistency
        $element_settings_json = sanitize_text_field($_POST['element_settings'] ?? '');
        $element_settings = null;

        if (!empty($element_settings_json)) {
            $decoded = json_decode(stripslashes($element_settings_json), true);
            if (is_array($decoded)) {
                $element_settings = $decoded;
            }
        }

        // IMPORTANT: Always get settings from Bricks post meta as well
        // Bricks doesn't include unchecked checkboxes in the JSON sent via form
        // But they ARE stored in the post meta, so we need to check there
        $bricks_settings_from_meta = null;
        if ($page_id > 0 && class_exists('\Bricks\Elements')) {
            $bricks_data = get_post_meta($page_id, '_bricks_page_content', true);
            if (!empty($bricks_data)) {
                $bricks_data = json_decode($bricks_data, true);
                $bricks_settings_from_meta = $this->findElementSettings($bricks_data, $element_id);
            }
        }

        // Merge settings: POST settings take priority, but add missing keys from Bricks meta
        // This is crucial for checkboxes that are unchecked (not in POST JSON but in meta)
        if (is_array($bricks_settings_from_meta)) {
            if (empty($element_settings)) {
                $element_settings = $bricks_settings_from_meta;
            } else {
                // Merge: Bricks meta first (has all settings), then POST settings override
                // This ensures we get double_optin from meta if it exists there
                $element_settings = array_merge($bricks_settings_from_meta, $element_settings);
            }
        }

        if (empty($element_settings)) {
            wp_send_json_error(['message' => __('Element settings not found.', 'mailerpress-pro')]);
            return;
        }

        // Prepare contact data
        $contact_data = [];
        $form_fields = $element_settings['form_fields'] ?? [];
        $field_mapping = $element_settings['field_mapping'] ?? [];
        $custom_fields_mapping = $element_settings['custom_fields_mapping'] ?? [];

        // Get email (required)
        $email = '';
        foreach ($form_fields as $field) {
            $field_name = !empty($field['field_name']) ? $field['field_name'] : '';
            if (($field['field_type'] ?? '') === 'email' && isset($_POST[$field_name])) {
                $email = sanitize_email($_POST[$field_name]);
                break;
            }
        }

        // Also check if email is directly in POST
        if (empty($email) && isset($_POST['email'])) {
            $email = sanitize_email($_POST['email']);
        }

        if (empty($email) || !is_email($email)) {
            wp_send_json_error(['message' => __('Valid email address is required.', 'mailerpress-pro')]);
            return;
        }

        $contact_data['contactEmail'] = $email;

        // Map standard fields
        $default_mapping = [
            'email' => 'contactEmail',
            'firstname' => 'contactFirstName',
            'first_name' => 'contactFirstName',
            'lastname' => 'contactLastName',
            'last_name' => 'contactLastName',
        ];

        foreach ($form_fields as $field) {
            $field_name = !empty($field['field_name']) ? $field['field_name'] : '';
            if (empty($field_name) || !isset($_POST[$field_name])) {
                continue;
            }

            $field_value = sanitize_text_field($_POST[$field_name]);
            if (empty($field_value)) {
                continue;
            }

            // Check custom field mapping first
            $mapped_to = null;
            foreach ($field_mapping as $mapping) {
                if (($mapping['form_field'] ?? '') === $field_name && !empty($mapping['mailerpress_field'])) {
                    $mapped_to = $mapping['mailerpress_field'];
                    break;
                }
            }

            // If no custom mapping, use default mapping
            if (empty($mapped_to) && isset($default_mapping[$field_name])) {
                $mapped_to = $default_mapping[$field_name];
            }

            if ($mapped_to && $mapped_to !== 'contactEmail') {
                $contact_data[$mapped_to] = $field_value;
            }
        }

        // Map custom fields
        $custom_fields = [];
        foreach ($custom_fields_mapping as $mapping) {
            $form_field = $mapping['form_field_custom'] ?? '';
            $custom_field_key = $mapping['custom_field_key'] ?? '';

            if (empty($form_field) || empty($custom_field_key)) {
                continue;
            }

            if (isset($_POST[$form_field])) {
                $field_value = sanitize_text_field($_POST[$form_field]);
                if (!empty($field_value)) {
                    $custom_fields[$custom_field_key] = $field_value;
                }
            }
        }

        if (!empty($custom_fields)) {
            $contact_data['custom_fields'] = $custom_fields;
        }

        // Add lists (now supports multiple)
        $lists = $element_settings['mailerpress_list'] ?? [];
        if (!empty($lists)) {
            if (is_string($lists)) {
                // Single value (backward compatibility)
                $lists = [$lists];
            }
            if (is_array($lists) && !empty($lists)) {
                $contact_data['lists'] = array_map(function ($list_id) {
                    return ['id' => intval($list_id)];
                }, array_filter($lists));
            }
        }

        // If no lists provided, the default list will be assigned automatically

        // Add tags
        $tags = $element_settings['mailerpress_tags'] ?? [];
        if (!empty($tags) && is_array($tags)) {
            $contact_data['tags'] = array_map(function ($tag_id) {
                return ['id' => intval($tag_id)];
            }, array_filter($tags));
        }

        // Check if contact already exists
        $existing_contact = null;
        if (function_exists('mailerpress_get_contact_by_email')) {
            $existing_contact = mailerpress_get_contact_by_email($email);
        } else {
            // Fallback: check via model
            $contactModel = \MailerPress\Core\Kernel::getContainer()->get(\MailerPress\Models\Contacts::class);
            $existing_contact = $contactModel->getContactByEmail($email);
        }

        // Subscription status
        // Only set subscription status if contact doesn't exist or is not already subscribed
        // If contact exists and is already subscribed, preserve the status and only update lists/tags/custom fields
        if (!$existing_contact || $existing_contact->subscription_status !== 'subscribed') {
            // Check global MailerPress signup confirmation setting
            $subscriptionOption = get_option('mailerpress_signup_confirmation', null);
            if (null !== $subscriptionOption) {
                $subscriptionOption = json_decode($subscriptionOption, true);
                // Vérifier que json_decode a retourné un tableau valide
                if (!is_array($subscriptionOption)) {
                    $subscriptionOption = null;
                }
            }

            $enableSignupConfirmation = false;
            if (is_array($subscriptionOption) && isset($subscriptionOption['enableSignupConfirmation'])) {
                $enableSignupConfirmation = (bool)$subscriptionOption['enableSignupConfirmation'];
            }

            // Get double opt-in setting from element
            // Now using SELECT control (yes/no) instead of checkbox, so value is always stored
            // Use exact same logic as Elementor
            $doubleOptin = $element_settings['double_optin'] ?? 'yes';

            // Normalize to 'yes' or 'no' (handle boolean, string, numeric)
            if ($doubleOptin === true || $doubleOptin === 'true' || $doubleOptin === 'yes' || $doubleOptin === 1 || $doubleOptin === '1') {
                $doubleOptin = 'yes';
            } elseif ($doubleOptin === false || $doubleOptin === 'false' || $doubleOptin === 'no' || $doubleOptin === 0 || $doubleOptin === '0' || $doubleOptin === '') {
                $doubleOptin = 'no';
            } else {
                // Default to 'yes' if unexpected value
                $doubleOptin = 'yes';
            }

            // Now use exact same logic as Elementor
            $confirmOption = ($doubleOptin === 'yes') ? '1' : '';

            // Determine subscription status based on element settings
            // Element settings are prioritized: if double opt-in is checked in element, use pending
            $desiredStatus = $confirmOption === "1" ? 'pending' : 'subscribed';
            // Set contact status - this is the key field
            $contact_data['contactStatus'] = $desiredStatus;

            // Also set subscription_status for compatibility (some integrations use this)
            $contact_data['subscription_status'] = $desiredStatus;

            // Add opt-in source
            // First check if a custom optin_source is set in element settings
            $custom_optin_source = !empty($element_settings['optin_source']) ? sanitize_text_field($element_settings['optin_source']) : '';

            if (!empty($custom_optin_source)) {
                // Use custom opt-in source from settings
                $contact_data['opt_in_source'] = $custom_optin_source;
            } else {
                // Use automatic logic based on double opt-in setting
                // CRITICAL: 'manual' forces subscribed status
                // If element double opt-in is unchecked (user wants subscribed directly):
                // - Use 'manual' to force the status to 'subscribed' (prevents API from overriding)
                // - This ensures element settings are always respected
                // If element double opt-in is checked (user wants pending with email):
                // - Use 'bricks' so that ContactCreated hook sends the confirmation email
                // - The API will respect 'pending' status if global double opt-in is enabled
                if ($desiredStatus === 'subscribed') {
                    // User wants subscribed directly, use 'manual' to force it
                    $contact_data['opt_in_source'] = 'bricks';
                } else {
                    // User wants pending with confirmation email, use 'bricks' to trigger email
                    $contact_data['opt_in_source'] = 'bricks';
                }
            }
        } else {
            // Contact exists and is already subscribed - don't change status
            // Only update lists, tags, and custom fields
            // Explicitly set to null so updateContact won't change the status
            $contact_data['contactStatus'] = null;
            $contact_data['subscription_status'] = null;
        }

        $contact_data['optin_details'] = wp_json_encode([
            'element_id' => $element_id,
            'page_id' => get_the_ID(),
        ]);

        // Add contact to MailerPress
        $result = add_mailerpress_contact($contact_data);

        if (isset($result['success']) && $result['success']) {
            $success_message = $element_settings['success_message'] ?? __('Thank you for subscribing!', 'mailerpress-pro');
            wp_send_json_success(['message' => $success_message]);
        } else {
            $error_message = isset($result['error']) ? $result['error'] : __('Failed to add contact to MailerPress.', 'mailerpress-pro');
            wp_send_json_error(['message' => $error_message]);
        }
    }

    private function findElementSettings($bricks_data, $element_id)
    {
        if (!is_array($bricks_data)) {
            return null;
        }

        foreach ($bricks_data as $element) {
            // Check if this is our element
            if (isset($element['name']) && $element['name'] === 'mailerpress-optin') {
                // Match by element ID
                if (isset($element['id']) && $element['id'] === $element_id) {
                    return $element['settings'] ?? null;
                }
            }

            // Recursively search in nested elements
            if (isset($element['children']) && is_array($element['children'])) {
                $found = $this->findElementSettings($element['children'], $element_id);
                if ($found !== null) {
                    return $found;
                }
            }
        }

        return null;
    }
}
