<?php
/**
 * Base Api Class
 *
 * @link       https://wpswings.com/
 * @since      1.0.0
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 */

/**
 * Base Api Class.
 *
 * This class defines all code necessary api communication.
 *
 * @since      1.0.0
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 */
class Crm_Integration_Api_Base_Keap {

	/**
	 * Keap base url.
	 *
	 * @since    1.0.0
	 * @var      string    $base_url    Keap base url.
	 */
	public $base_url;

	/**
	 * Get Request.
	 *
	 * @param string $endpoint Api endpoint of mautic.
	 * @param array  $data Data to be used in request.
	 * @param array  $headers header to be used in request.
	 */
	public function get( $endpoint, $data = array(), $headers = array() ) {

		return $this->request_get( 'GET', $endpoint, $data, $headers );
	}

	/**
	 * Post Request.
	 *
	 * @param string $endpoint Api endpoint of mautic.
	 * @param array  $request_data Data to be used in request.
	 * @param array  $headers header to be used in request.
	 */
	public function post( $endpoint, $request_data = array(), $headers = array() ) {

		$response = $this->request( 'POST', $endpoint, $request_data, $headers );
		return $response;
	}

	/**
	 * Patch Request.
	 *
	 * @param string $endpoint Api endpoint of mautic.
	 * @param array  $request_data Data to be used in request.
	 * @param array  $headers header to be used in request.
	 */
	public function patch( $endpoint, $request_data = array(), $headers = array() ) {

		$response = $this->request( 'PATCH', $endpoint, $request_data, $headers );

		return $response;
	}

	/**
	 * Put Request.
	 *
	 * @param string $endpoint Api endpoint of mautic.
	 * @param array  $request_data Data to be used in request.
	 * @param array  $headers header to be used in request.
	 */
	public function put( $endpoint, $request_data = array(), $headers = array() ) {

		$response = $this->request( 'PUT', $endpoint, $request_data, $headers );

		return $response;
	}

	/**
	 * Delete Request.
	 *
	 * @param string $endpoint Api endpoint of zoho.
	 * @param array  $data Data to be used in request.
	 * @param array  $headers header to be used in request.
	 */
	public function delete( $endpoint, $data = array(), $headers = array() ) {
		return $this->request( 'DELETE', $endpoint, $data, $headers );
	}

	/**
	 * Send api request
	 *
	 * @param string $method   HTTP method.
	 * @param string $endpoint Api endpoint.
	 * @param array  $request_data     Request data.
	 * @param array  $headers header to be used in request.
	 * @throws Exception  Exception .
	 */
	private function request( $method, $endpoint, $request_data = array(), $headers = array() ) {
		$data = '';
		$method  = strtoupper( trim( $method ) );
		$url     = $endpoint;
		if ( 'https://api.infusionsoft.com/token' == $endpoint ) {
			$headers = $headers;
		} else {

			$headers = array_merge( $headers, $this->get_headers() );
		}

		$response = wp_remote_post(
			$url,
			array(
				'method'        => $method,
				'timeout'       => 45,
				'redirection'   => 5,
				'httpversion'   => '1.0',
				'blocking'      => true,
				'headers'       => $headers,
				'body'          => $request_data,
				'cookies'       => array(),
			)
		);

		// Add better exception handling.
		try {
			$data = $this->parse_response( $response );
		} catch ( Exception $e ) {
			//throw new Exception( 'Error', 0 );
			// Got an error in api response may be due to unappropriate connection. Check logs for more information.
			return $data;
		}

		return $data;
	}

	/**
	 * Send api request
	 *
	 * @param string $method   HTTP method.
	 * @param string $endpoint Api endpoint.
	 * @param array  $request_data     Request data.
	 * @param array  $headers header to be used in request.
	 * @throws Exception  Exception .
	 */
	private function request_get( $method, $endpoint, $request_data = array(), $headers = array() ) {

		$data = array();
		$method  = strtoupper( trim( $method ) );
		$url     = $endpoint;
		$headers = array_merge( $headers, $this->get_headers() );
		if ( ! is_array( $request_data ) ) {
			$request_data = array( $request_data );
		}
		$response = wp_remote_post(
			$url,
			array(
				'method'        => $method,
				'timeout'       => 45,
				'redirection'   => 5,
				'httpversion'   => '1.0',
				'blocking'      => true,
				'headers'       => $headers,
				'body'          => $request_data,
				'cookies'       => array(),
			)
		);

		// Add better exception handling.
		try {
			$data = $this->parse_response( $response );
		} catch ( Exception $e ) {
			//throw new Exception( 'Error', 0 );
			// Got an error in api response may be due to unappropriate connection. Check logs for more information.
			return $data;
		}

		return $data;
	}

	/**
	 * Parse Api response.
	 *
	 * @param array $response Raw response.
	 *
	 * @throws Exception  Exception .
	 * @return array filtered reponse.
	 */
	private function parse_response( $response ) {

		if ( $response instanceof WP_Error ) {
			throw new Exception( 'Error', 0 );
		}
		$code    = (int) wp_remote_retrieve_response_code( $response );
		$message = wp_remote_retrieve_response_message( $response );
		$body    = wp_remote_retrieve_body( $response );
		$data    = json_decode( $body, ARRAY_A );

		return compact( 'code', 'message', 'data' );
	}

	/**
	 * Get headers.
	 *
	 * @return array headers.
	 */
	public function get_headers() {

		return array( 'Content-Type' => 'application/json' );
	}
}
