<?php
/**
 * The ajax-specific functionality of the plugin.
 *
 * @link       https://wpswings.com/
 * @since      1.0.0
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 */

if ( ! class_exists( 'Crm_Integration_Connect_Framework_Keap' ) ) {
	wp_die( 'Crm_Integration_Connect_Framework_Keap does not exists.' );
}

/**
 * The ajax-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the ajax-specific stylesheet and JavaScript.
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 */
use Automattic\WooCommerce\Utilities\OrderUtil;
class Crm_Integration_For_Keap_Ajax {

	/**
	 * Ajax Call back
	 */
	public function mwb_woo_keap_ajax_cb() {
		check_ajax_referer( 'ajax_nonce', 'nonce' );
		$event = ! empty( $_POST['event'] ) ? sanitize_text_field( wp_unslash( $_POST['event'] ) ) : '';
		$data  = $this->$event( $_POST );
		echo wp_json_encode( $data );
		wp_die();
	}

	/**
	 * [get_module_fields_for_mapping description]
	 *
	 * @param  array $posted_data [description].
	 * @return [type]              [description].
	 */
	public function get_module_fields_for_mapping( $posted_data = array() ) {

		// Get keap crm module fields.
		$module_data = array();
		$module      = ! empty( $posted_data['module'] ) ? sanitize_text_field( wp_unslash( $posted_data['module'] ) ) : '';
		$force       = ! empty( $posted_data['force'] ) ? sanitize_text_field( wp_unslash( $posted_data['force'] ) ) : false;
		$crm_integration_keap_api    = Crm_Integration_Keap_Api::get_instance();
		$module_data['crm_fields'] = $crm_integration_keap_api->get_module_fields( $module, $force );

		// Get default woo mapping.
		$module_data['default_mapping_data'] = Crm_Integration_For_Keap_Cpt::get_default_mapping_data( $module );
		$module_data['default_event'] = Crm_Integration_For_Keap_Cpt::get_default_event( $module );
		$module_data['default_primary'] = Crm_Integration_For_Keap_Cpt::get_default_primary_field( $module );

		return $module_data;
	}

	/**
	 * [get_default_status_mapping_data description]
	 *
	 * @return [type] [description]
	 */
	public function get_default_status_mapping_data() {
		check_ajax_referer( 'ajax_nonce', 'nonce' );
		$module = isset( $_POST['crm_selected_object'] ) ? sanitize_text_field( wp_unslash( $_POST['crm_selected_object'] ) ) : array();
		$field = isset( $_POST['field'] ) ? sanitize_text_field( wp_unslash( $_POST['field'] ) ) : array();
		$status_mapping = array();
		if ( ! empty( $module ) && ! empty( $field ) ) {
			$status_mapping = $this->get_default_status_mapping( $module, $field );
		}
		return $status_mapping;
	}


	/**
	 * [get_default_status_mapping description].
	 *
	 * @param  [type] $module [description].
	 * @param  [type] $field  [description].
	 * @return [type]         [description].
	 */
	public function get_default_status_mapping( $module, $field ) {

		$status_mapping = array();
		if ( ! empty( $module ) && ! empty( $field ) ) {
			if ( 'Sales_Orders' == $module && 'Status' == $field ) {
				$order_statuses = wc_get_order_statuses();
				foreach ( $order_statuses as $key => $value ) {
					switch ( $key ) {
						case 'wc-pending':
						$status_mapping[ $key ] = 'Created';
							break;
						case 'wc-processing':
						$status_mapping[ $key ] = 'Approved';
							break;
						case 'wc-on-hold':
						$status_mapping[ $key ] = 'Created';
							break;
						case 'wc-completed':
						$status_mapping[ $key ] = 'Delivered';
							break;
						case 'wc-cancelled':
						$status_mapping[ $key ] = 'Cancelled';
							break;
						case 'wc-refunded':
						$status_mapping[ $key ] = 'Cancelled';
							break;
						case 'wc-failed':
						$status_mapping[ $key ] = 'Cancelled';
							break;
						default:
						$status_mapping[ $key ] = 'Created';
							break;
					}
				}
			}
		}
		return $status_mapping;
	}


	/**
	 * Get available fields in a module.
	 *
	 * @param array $posted_data Array of ajax posted data.
	 *
	 * @return array $module_data data of specific module.
	 */
	public function get_module_fields( $posted_data = array() ) {

		$module_data = array();
		$module      = ! empty( $posted_data['module'] ) ? sanitize_text_field( wp_unslash( $posted_data['module'] ) ) : '';
		$force       = ! empty( $posted_data['force'] ) ? sanitize_text_field( wp_unslash( $posted_data['force'] ) ) : false;
		$crm_integration_keap_api    = Crm_Integration_Keap_Api::get_instance();
		$module_data = $crm_integration_keap_api->get_module_fields( $module, $force );
		return $module_data;
	}

	/**
	 * Get available modules in keap.
	 *
	 * @param array $posted_data Array of ajax posted data.
	 * @return array $module_data data of specific module.
	 */
	public function get_modules( $posted_data = array() ) {
		$module_data = array();
		$crm_integration_keap_api    = Crm_Integration_Keap_Api::get_instance();
		return $crm_integration_keap_api->get_modules_data();
	}

	/**
	 * Get available modules in keap.
	 *
	 * @return array data of specific module.
	 */
	public function delete_feed_post() {
		check_ajax_referer( 'ajax_nonce', 'nonce' );
		$feed_id = isset( $_POST['feed_id'] ) ? sanitize_text_field( wp_unslash( $_POST['feed_id'] ) ) : false;
		if ( $feed_id ) {
			wp_trash_post( $feed_id );
		}
		return array( 'success' => true );
	}


	/**
	 * Change feed status.
	 */
	public function change_feed_status() {
		check_ajax_referer( 'ajax_nonce', 'nonce' );
		$feed_id = isset( $_POST['feed_id'] ) ? sanitize_text_field( wp_unslash( $_POST['feed_id'] ) ) : false;
		$status = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : false;
		if ( $feed_id && $status ) {
			$args = array(
				'ID'          => $feed_id,
				'post_status' => $status,
			);
			wp_update_post( $args );
		}
		return array( 'success' => true );
	}

	/**
	 * [disconnect_account description]
	 *
	 * @return [type] [description]
	 */
	public function disconnect_account() {
		// Onboarding Dasboard key.

		$wipe_allowed = Crm_Integration_For_Keap_Admin::is_wipe_allowed();

		delete_option( 'mwb-woo-keap-onboarding-completed' );
		// Bulk Syncing key.
		delete_option( 'mwb_crm_Products_feed_onboarded' );
		delete_option( 'mwb_crm_Products_feed_final_reponse' );
		delete_option( 'mwb_crm_Contacts_feed_onboarded' );
		delete_option( 'mwb_crm_Contacts_feed_final_reponse' );
		delete_option( 'mwb_crm_Sales_Orders_feed_onboarded' );
		delete_option( 'mwb_crm_Sales_Orders_feed_final_reponse' );

		// Connection key.
		delete_option( 'mwb-woo-keap-authorised' );
		delete_option( 'mwb_keap_access_token' );
		delete_option( 'mwb_keap_refresh_token' );
		delete_option( 'mwb_keap_token_expiry' );
		delete_option( 'mwb_keap_token_type' );
		delete_option( 'mwb_keap_scope' );
		delete_option( 'mwb_keap_valid_client_ids_stored' );
		delete_option( 'mwb_keap_access_token_data' );
		delete_option( 'mwb_keap_client_id' );
		delete_option( 'mwb_keap_secret_id' );

		// // Default feeds events.
		if ( true == $wipe_allowed ) {
			delete_option( 'mwb_woo_keap_default_Products_feed_id' );
			delete_option( 'mwb_woo_keap_default_Contacts_feed_id' );
			delete_option( 'mwb_woo_keap_default_Sales_Orders_feed_id' );
			delete_option( 'mwb_woo_keap_default_feeds_created' );
			// Delete mwb_keap_post.
			$allposts = get_posts(
				array(
					'post_type' => 'mwb_keap_feed',
					'numberposts' => -1,
				)
			);

			foreach ( $allposts as $eachpost ) {
				wp_delete_post( $eachpost->ID, true );
				delete_post_meta( $eachpost->ID, 'crm_object' );
				delete_post_meta( $eachpost->ID, 'mapping_data' );
				delete_post_meta( $eachpost->ID, 'status_mapping' );
				delete_post_meta( $eachpost->ID, 'add_line_item' );
				delete_post_meta( $eachpost->ID, 'feed_event' );
				delete_post_meta( $eachpost->ID, 'lead_source' );
				delete_post_meta( $eachpost->ID, 'primary_field' );
				delete_post_meta( $eachpost->ID, 'default_feed' );
			}
		}
		


		return array( 'success' => true );
	}

	/**
	 * [get_reauth_url description].
	 *
	 * @return [type] [description].
	 */
	public function get_reauthorize_url() {
		$response        = array( 'success' => false );
		$keap_api        = Crm_Integration_Keap_Api::get_instance();
		$auth_url        = $keap_api->get_auth_code_url();
		if ( ! $auth_url ) {
			$response['msg'] = __( 'Something went wrong! Check your credentials and authorize again', 'integration-with-keap' );
			delete_option( 'mwb-woo-keap-authorised', false );
			return $response;
		}
		$response   = array(
			'success' => true,
			'url'     => $auth_url,
		);

		return $response;
	}

	/**
	 * [get_datatable_data_cb description].
	 */
	public function get_datatable_data_cb_keap() {

		$request = $_GET;
		$offset = $request['start'];
		$limit  = $request['length'];

		$search_data = $request['search'];
		$search_value = $search_data['value'];

		global $wpdb;
		$table_name = $wpdb->prefix . 'mwb_woo_keap_log';

		if ( '' !== $search_value ) {

			$log_data = mwb_woo_keap_get_log_data( $search_value );
			
		} else {

			$log_data = mwb_woo_keap_get_log_data( false, $limit, $offset );

		}

		$keap_api = Crm_Integration_Keap_Api::get_instance();
		$count_data  = mwb_woo_keap_get_total_log_count();
		$total_count = $count_data[0];
		$data   = array();
		foreach ( $log_data as $key => $value ) {
			if ( ! empty( $value['keap_id'] ) ) {
				$data_href = $keap_api->get_crm_link( $value['keap_id'], $value['keap_object'] );
				if ( ! empty( $data_href ) ) { // phpcs:ignore
					$link = '<a href="' . $data_href . '" target="_blank">' . $value['keap_id'] . '</a>';
				} else {
					$link = $value['keap_id'];
				}
			} else {
				$link = '-';
			}

			$temp = array(
				'',
				$value['event'],
				$value['woo_object'],
				$value['woo_id'],
				$value['keap_object'],
				$link,
				gmdate( 'd-m-Y h:i A', esc_html( $value['time'] ) ),
				json_encode( json_decode( $value['request'], true ) ),
				json_encode( json_decode( $value['response'], true ) ),
			);
			$data[] = $temp;
		}

		$json_data = array(
			'draw' => intval( $request['draw'] ),
			'recordsTotal' => $total_count,
			'recordsFiltered' => $total_count,
			'data' => $data,
		);

		echo json_encode( $json_data );

		wp_die();
	}

	/**
	 * [renew_access_token description].
	 *
	 * @return [type] [description].
	 */
	public function renew_access_token() {

		$response = array( 'success' => false );
		$keap_api = Crm_Integration_Keap_Api::get_instance();
		$response['msg'] = __( 'Something went wrong! Check your credentials and authorize again', 'integration-with-keap' );
		if ( $keap_api->renew_access_token() ) {
			$token_expiry   = Crm_Integration_For_Keap_Admin::get_keap_access_token_expiry();
			$token_message = sprintf( 'Access token will expire in %s minutes.', $token_expiry );
			$response = array(
				'success' => true,
				'msg'     => __( 'Success', 'integration-with-keap' ),
				'token_message' => $token_message,
			);
		}
		return $response;
	}

	/**
	 * Create default feeds.
	 *
	 * @return array Feed response.
	 */
	public function create_keap_default_feeds() {
		check_ajax_referer( 'ajax_nonce', 'nonce' );
		$success = false;
		$message = __( 'Something went wrong !', 'integration-with-keap' );
		$cpt_instance = new Crm_Integration_For_Keap_Cpt();
		$feeds = $cpt_instance->get_default_feeds();
		$step  = isset( $_POST['step'] ) ? sanitize_text_field( wp_unslash( $_POST['step'] ) ) : '';
		$feed  = ! empty( $feeds[ $step ] ) ? $feeds[ $step ] : false;

		if ( is_array( $feed ) ) {

			$existing_feed = get_option( 'mwb_woo_keap_default_' . $feed['crm_object'] . '_feed_id', false );

			$feed_id = $existing_feed;

			if ( false !== $existing_feed ) {
				
				$feed_id = $cpt_instance->is_valid_feed_post( $existing_feed );

			}

			if ( false === $feed_id ) {

				$feed_id = $cpt_instance->create_feed_post( $feed['title'], $feed['crm_object'] );

			}
			if ( false === $feed_id ) {
			// Check for feed post in all post.
			// Extra check if feed id is deleted from options.
				$feed_module = new Crm_Integration_For_Keap_Cpt();
				$feed_id = $feed_module->check_for_feed_post( $feed['object'] );
			} else {

				$message = sprintf(
					/* translators: 1: Name of a city 2: ZIP code */
					__( '%s feed already exists.', 'integration-with-keap' ),
					$feed['crm_object']
				);
				$success = true;
			}

			if ( ! is_wp_error( $feed_id ) && false !== $feed_id ) {

				update_option( 'mwb_woo_keap_default_' . $feed['crm_object'] . '_feed_id', $feed_id );

				$message = sprintf(
					/* translators: 1: Name of a feed.*/
					__( '%s feed created.', 'integration-with-keap' ),
					$feed['crm_object']
				);
				$success = true;
			}
		}

		$response['total_count'] = count( $feeds );
		$response['next_step']   = $step + 1;
		$response['feed']        = $feed;
		$response['feed_type']   = $feed['crm_object'];
		$response['feed_id']     = $feed_id;
		$response['message']     = $message;
		$response['success']     = $success;

		if ( $response['total_count'] == $response['next_step'] ) {
			update_option( 'mwb_woo_keap_default_feeds_created', true );
		}
		return $response;
	}

	/**
	 * Get Bulk feeds from database and send a request.
	 *
	 * @param array $posted_data Array of ajax posted data.
	 * @since  1.0.0
	 * @return array
	 */
	public function formBulkDataSync( $posted_data = array() ) {
		$tag_apply = array();
		// Get current Offset.
		$offset = ! empty( $posted_data['offset'] ) ?
		sanitize_text_field( wp_unslash( $posted_data['offset'] ) ) :
		'1';

		$is_onboarding = ! empty( $posted_data['isOnboarding'] ) ?
		sanitize_text_field( wp_unslash( $posted_data['isOnboarding'] ) ) :
		false;

		// Get current form submission.
		$formdata = array();
		! empty( $posted_data['form'] ) ? parse_str( sanitize_text_field( wp_unslash( $posted_data['form'] ) ), $formdata ) : '';

		$formdata = ! empty( $formdata ) ? map_deep( wp_unslash( $formdata ), 'sanitize_text_field' ) : false;

		// Feed Id.
		$feed_id = ! empty( $formdata['object-feed'] ) ?
		sanitize_text_field( wp_unslash( $formdata['object-feed'] ) ) :
		'';

		$keap = Woo_Crm_Connect_Keap::get_instance();
		$record_type = $keap->getFeed( $feed_id, 'crm_object' );

		if ( true == $is_onboarding ) {
			$is_onboarding_done = get_option( 'mwb_crm_' . $record_type . '_feed_onboarded', false );
			if ( ! empty( $is_onboarding_done ) && 'true' == $is_onboarding_done ) {

				return array(
					'result'    => true,
					'record_type'    => $record_type,
					'message'   => get_option( 'mwb_crm_' . $record_type . '_feed_final_reponse', array() ),
				);
			}
		}

		$keap = Woo_Crm_Connect_Keap::get_instance();
		$feed_title = $keap->getFeedTitle( $feed_id );

		switch ( $record_type ) {

			case 'Contacts':
			$woo_object = 'shop_order';
				break;
			case 'Products':
			$woo_object = 'product';
				break;
			case 'Sales_Orders':
			$woo_object = 'shop_order';
				break;
		}

		$request_module = new Crm_Integration_Request_Module_Keap();

		$bulk_data = $request_module->getBulkData( $woo_object, $offset, $feed_id );
		$single_record = array();

		$single_order_record = array();

		if ( ! empty( $bulk_data['request'] ) && is_array( $bulk_data['request'] ) ) {

			$keap_api = Crm_Integration_Keap_Api::get_instance();

			$bulk_ids = ! empty( $bulk_data['bulk_ids'] ) ? $bulk_data['bulk_ids'] : array();

			$log_data = compact( 'feed_id', 'feed_title', 'bulk_ids', 'woo_object' );
			if ( 'Products' == $record_type ) {

				foreach ( $bulk_data['request'] as $key => $single_value ) {

					foreach ( $single_value as $k => $v ) {

						if ( 'product_id' !== $k ) {

							$single_record[ $k ] = $v;
						}
					}
					$woo_id = $single_value['product_id'];

					$update_prod_id = get_post_meta( $single_value['product_id'], '_mwb_keap_product_id', true );

					if ( ! empty( $update_prod_id ) ) {

						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;

						$request_data = json_encode( $single_record );

						$result = $keap_api->update_single_products(
							$record_type,
							$request_data,
							$endpoint,
							$woo_id,
							true,
							$log_data
						);

						$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );
						$keap_id = !empty( $result['data']['id'] ) ? $result['data']['id'] : '';
						Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );

					} else {

						$request_data = json_encode( $single_record );
						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
						$result = $keap_api->create_single_record(
							$record_type,
							$request_data,
							true,
							$log_data,
							$endpoint,
							$woo_id
						);

						if ( 'Products' == $record_type ) {

							update_post_meta( $single_value['product_id'], '_mwb_keap_product_id', $result['data']['id'] );

							$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );
							$keap_id = !empty( $result['data']['id'] ) ? $result['data']['id'] : '';
							Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id , $is_success );

						}
					}
				}
			} elseif ( 'Contacts' == $record_type ) {

				if ( ! empty( $bulk_data['request'] ) ) {
					foreach ( $bulk_data['request'] as $single_key => $single_contacts_value ) {

						foreach ( $single_contacts_value as $key => $value ) {

							if ( 'contact_id' !== $key && 'tagIds' !== $key ) {

								$single_contacts_record[ $key ] = $value;
							}
						}

						$woo_id = $single_contacts_value['contact_id'];

						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/contacts';

						$request_data = $keap_api->json_contacts_data( $single_contacts_record, $feed_id );

						$result = $keap_api->create_or_update_contacts(
							$record_type,
							$request_data,
							$endpoint,
							$woo_id,
							true,
							$log_data
						);

						$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );
						$keap_id = !empty( $result['data']['id'] ) ? $result['data']['id'] : '';

						Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );

						// Apply tag to contacts    
						if ( $is_success ) {    
							$endpoint = 'https://api.infusionsoft.com/crm/rest//contacts/' . $keap_id . '/tags';    
							$tag_apply['tagIds'] = array( $single_contacts_value['tagIds'] ) ;  
							$request_data = json_encode($tag_apply, JSON_NUMERIC_CHECK);    
							$pattern = '/"/i';  
							$request_data = preg_replace($pattern, '', $request_data);  
							$pattern = '/tagIds/i'; 
							$request_data = preg_replace($pattern, '"tagIds"', $request_data);  
							$result_tag = $keap_api->apply_tag_to_contacts($request_data, $endpoint);
						}
					}
				}
			}
		}

		return array(
			'result'    => true,
			'message'   => $bulk_data,
		);
	}

	/**
	 * Get Bulk feeds which are not synced yet from database and send a request.
	 *
	 * @param array $posted_data Array of ajax posted data.
	 *
	 * @since  1.0.0
	 * @return array
	 */
	public function formOneClickSync( $posted_data = array() ) {
		$tag_apply = array();
		// Get current Offset.
		$offset = ! empty( $posted_data['offset'] ) ?
		sanitize_text_field( wp_unslash( $posted_data['offset'] ) ) :
		'1';

		// Get current form submission.
		$formdata = array();
		! empty( $posted_data['form'] ) ? parse_str( sanitize_text_field( wp_unslash( $posted_data['form'] ) ), $formdata ) : '';
		$formdata = ! empty( $formdata ) ? map_deep( wp_unslash( $formdata ), 'sanitize_text_field' ) : false;

		// Feed Id.
		$feed_id = ! empty( $formdata['object-feed'] ) ?
		sanitize_text_field( wp_unslash( $formdata['object-feed'] ) ) :
		'';

		$keap = Woo_Crm_Connect_Keap::get_instance();
		$record_type = $keap->getFeed( $feed_id, 'crm_object' );

		$keap = Woo_Crm_Connect_Keap::get_instance();
		$feed_title = $keap->getFeedTitle( $feed_id );

		switch ( $record_type ) {

			case 'Sales_Orders':
			case 'Contacts':
			$woo_object = 'shop_order';
				break;
			case 'Products':
			$woo_object = 'product';
				break;

		}

		$request_module = new Crm_Integration_Request_Module_Keap();

		$bulk_data = $request_module->getBulkData( $woo_object, $offset, $feed_id, true );
		if ( ! empty( $bulk_data['request'] ) && is_array( $bulk_data['request'] ) ) {

			$keap_api = Crm_Integration_Keap_Api::get_instance();

			$bulk_ids = ! empty( $bulk_data['bulk_ids'] ) ? $bulk_data['bulk_ids'] : array();

			$log_data = compact( 'feed_id', 'feed_title', 'bulk_ids', 'woo_object' );
			if ( 'Products' == $record_type ) {

				foreach ( $bulk_data['request'] as $key => $single_value ) {

					foreach ( $single_value as $k => $v ) {

						if ( 'product_id' !== $k ) {

							$single_record[ $k ] = $v;
						}
					}

					$woo_id = $single_value['product_id'];

					$update_prod_id = get_post_meta( $woo_id, '_mwb_keap_product_id', true );

					if ( ! empty( $update_prod_id ) ) {

						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;

						$request_data = json_encode( $single_record );

						$result = $keap_api->update_single_products(
							$record_type,
							$request_data,
							$endpoint,
							$woo_id,
							true,
							$log_data
						);

						$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );

						Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $result['data']['id'], $is_success );

					} else {

						$request_data = json_encode( $single_record );

						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
						$result = $keap_api->create_single_record(
							$record_type,
							$request_data,
							true,
							$log_data,
							$endpoint,
							$woo_id
						);

						if ( 'Products' == $record_type ) {

							update_post_meta( $single_value['product_id'], '_mwb_keap_product_id', $result['data']['id'] );

							$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );

							Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $result['data']['id'], $is_success );

						}
					}
				}
			} elseif ( 'Contacts' == $record_type ) {

				if ( ! empty( $bulk_data['request'] ) ) {

					foreach ( $bulk_data['request'] as $single_key => $single_contacts_value ) {

						foreach ( $single_contacts_value as $key => $value ) {

							if ( 'contact_id' !== $key && 'tagIds' !== $key ) {

								$single_contacts_record[ $key ] = $value;
							}
						}

						$woo_id = $single_contacts_value['contact_id'];

						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/contacts';

						$request_data = $keap_api->json_contacts_data( $single_contacts_record, $feed_id );

						$result = $keap_api->create_or_update_contacts(
							$record_type,
							$request_data,
							$endpoint,
							$woo_id,
							true,
							$log_data
						);
						if ( isset( $result['data']['id'] ) ) {
							
							$keap_id = $result['data']['id'] ; 
						}

						$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );    
						Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );    
						if ( $is_success ) {    
							$endpoint = 'https://api.infusionsoft.com/crm/rest//contacts/' . $keap_id . '/tags';    
							$tag_apply['tagIds'] = array( $single_contacts_value['tagIds'] ) ;  
							$request_data = json_encode($tag_apply, JSON_NUMERIC_CHECK);    
							$pattern = '/"/i';  
							$request_data = preg_replace($pattern, '', $request_data);  
							$pattern = '/tagIds/i'; 
							$request_data = preg_replace($pattern, '"tagIds"', $request_data);  
							$result_tag = $keap_api->apply_tag_to_contacts( $request_data, $endpoint ); 
						}
					}
				}
			}
		}

		return array(
			'result'    => true,
			'message'   => $bulk_data,
		);
	}

	/**
	 * Returns the mapping step we require.
	 *
	 * @param array $posted_data Array of ajax posted data.
	 *
	 * @since 1.0.0
	 *
	 */
	public function getObjectFeedOptions( $posted_data = array() ) {

		$feed_type = ! empty( $posted_data['syncModule'] ) ?
		sanitize_text_field( wp_unslash( $posted_data['syncModule'] ) ) :
		false;
		if ( empty( $feed_type ) ) {
			return false;
		}

		$args = array(
			'post_type' => 'mwb_keap_feed',
			'post_status' => 'publish',
			'fields' => 'ids',
			'meta_query' => array(
				'relation' => 'AND',
				array(
					'key' => 'crm_object',
					'compare' => '=',
					'value' => $feed_type,
				),
			),
		);

		$feeds = get_posts( $args );

		if ( ! empty( $feeds ) && is_array( $feeds ) ) {
			$result = array();
			foreach ( $feeds as $key => $feed_id ) {
				$result[ $feed_id ] = get_the_title( $feed_id );
			}
			wp_send_json(
				array(
					'result'    => 200,
					'options'   => $result,
				)
			);
		} else {
			wp_send_json(
				array(
					'result'    => 403,
					'options'   => '',
				)
			);
		}
	}

	/**
	 * Mark_onboarding_complete the mapping step we require.
	 *
	 * @since 1.0.0
	 */
	public function mark_onboarding_complete() {
		update_option( 'mwb-woo-keap-onboarding-completed', true );
		update_option( 'mwb-woo-keap-product-popup-notification', true );
		return array( 'success' => true );
	}

	/**
	 * Save plugin general settings
	 *
	 * @return array Response array.
	 */
	public function save_general_setting( $posted_data = array() ) {
		
		$settings = $posted_data['settings'];
		$sync_enable = isset( $settings['sync_enable'] ) ? map_deep( wp_unslash( $settings['sync_enable'] ), 'sanitize_text_field' ) : array();
		$log_enable = isset( $settings['log_enable'] ) ? map_deep( wp_unslash( $settings['log_enable'] ), 'sanitize_text_field' ) : array();
		$log_delete = isset( $settings['log_delete'] ) ? map_deep( wp_unslash( $settings['log_delete'] ), 'sanitize_text_field' ) : array();
		$wipe_data = isset( $settings['wipe_data'] ) ? map_deep( wp_unslash( $settings['wipe_data'] ), 'sanitize_text_field' ) : array();
		// 2.1.0 start.
		$error_reporting = isset( $settings['error_reporting'] ) ? map_deep( wp_unslash( $settings['error_reporting'] ), 'sanitize_text_field' ) : array();
		$error_email = isset( $settings['error_email'] ) ? map_deep( wp_unslash( $settings['error_email'] ), 'sanitize_text_field' ) : array();
		// 2.1.0 end.

		$enable_product_deletion_from_keap = isset( $settings['enable_product_deletion_from_keap'] ) ? map_deep( wp_unslash( $settings['enable_product_deletion_from_keap'] ), 'sanitize_text_field' ) : array();
		$enable_product_deletion_from_woo = isset( $settings['enable_product_deletion_from_woo'] ) ? map_deep( wp_unslash( $settings['enable_product_deletion_from_woo'] ), 'sanitize_text_field' ) : array();

		update_option( 'mwb_keap_enable_sync', $sync_enable );
		update_option( 'mwb_keap_enable_log', $log_enable );
		update_option( 'mwb_keap_delete_log', $log_delete );
		update_option( 'mwb_keap_wipe_data', $wipe_data );
		// 2.1.0 start.
		update_option( 'mwb_keap_error_reporting', $error_reporting );
		update_option( 'mwb_keap_error_email', $error_email );
		// 2.1.0 end.

		update_option( 'mwb_keap_enable_product_deletion_from_keap', $enable_product_deletion_from_keap );
		update_option( 'mwb_keap_enable_product_deletion_from_woo', $enable_product_deletion_from_woo );

		return array( 'success' => true );
	}

	/** 
	* Save plugin tags general settings 
	*   
	* @return array Response array. 
	*/  
	public function save_general_tag_setting( $posted_data = array() ) {    
		$settings = $posted_data['settings'];   
		$sync_enable = isset( $settings['sync_enable'] ) ? map_deep( wp_unslash( $settings['sync_enable'] ), 'sanitize_text_field' ) : array(); 
		$instant_sync_enable = isset( $settings['instant_sync_enable'] ) ? map_deep( wp_unslash( $settings['instant_sync_enable'] ), 'sanitize_text_field' ) : array(); 
		$products_bought = isset( $settings['products_bought'] ) ? map_deep( wp_unslash( $settings['products_bought'] ), 'sanitize_text_field' ) : array(); 
		$products_skus = isset( $settings['products_skus'] ) ? map_deep( wp_unslash( $settings['products_skus'] ), 'sanitize_text_field' ) : array();   
		$products_type = isset( $settings['products_type'] ) ? map_deep( wp_unslash( $settings['products_type'] ), 'sanitize_text_field' ) : array();   
		$order_statuses = isset( $settings['order_statuses'] ) ? map_deep( wp_unslash( $settings['order_statuses'] ), 'sanitize_text_field' ) : array();
		update_option( 'mwb_keap_enable_tag_sync', $sync_enable );
		update_option( 'mwb_keap_enable_tag_instant_sync', $instant_sync_enable );
		update_option( 'mwb_keap_enable_products_tag', $products_bought );
		update_option( 'mwb_keap_enable_skus_tag', $products_skus );
		update_option( 'mwb_keap_enable_products_type_tag', $products_type );
		update_option( 'mwb_keap_enable_order_statuses_tag', $order_statuses );
		return array( 'success' => true );
	}
	/**
	 * Clear sync log.
	 *
	 * @return array Response array.
	 */
	public function clear_sync_log() {
		Crm_Integration_For_Keap_Admin::delete_sync_log();
		return array( 'success' => true );
	}

	/**
	 * Download sync log.
	 *
	 * @return array Response array.
	 */
	public function download_sync_log() {

		$log_data = mwb_woo_keap_get_log_data( false, 25, 0, true );
		$data = array();
		$log_dir = WC_LOG_DIR . 'mwb-woo-keap-sync-log.log';
		if ( ! is_dir( $log_dir ) ) {
			$log_dir = WC_LOG_DIR . 'mwb-woo-keap-sync-log.log';
		}

		global $wp_filesystem;  // Define global object of WordPress filesystem.
		WP_Filesystem();        // Intialise new file system object.
		$file_data = '';
		$wp_filesystem->put_contents( $log_dir, '' );
		
		
		foreach ( $log_data as $key => $value ) {

			$value['keap_id'] = ! empty( $value['keap_id'] ) ? $value['keap_id'] : '-';
			$log   = 'FEED : ' . $value['event'] . PHP_EOL;
			$log  .= 'WOO ID : ' . $value['woo_id'] . PHP_EOL;
			$log  .= 'WOO OBJECT : ' . $value['woo_object'] . PHP_EOL;
			$log  .= 'KEAP ID : ' . $value['keap_id'] . PHP_EOL;
			$log  .= 'KEAP OBJECT : ' . $value['keap_object'] . PHP_EOL;
			$log  .= 'TIME : ' . gmdate( 'd-m-Y h:i A', esc_html( $value['time'] ) ) . PHP_EOL;
			$log  .= 'REQUEST : ' . wp_json_encode( json_decode( $value['request'], true ) ) . PHP_EOL;//phpcs:ignore
			$log  .= 'RESPONSE : ' . wp_json_encode( json_decode( $value['response'], true ) ) . PHP_EOL;//phpcs:ignore
			$log .= '------------------------------------' . PHP_EOL;

			$file_data .= $log;
		}

		$wp_filesystem->put_contents( $log_dir, $file_data );

		return array(
			'success' => true,
			'redirect' => admin_url( '?mwb_download=1' ),
		);
	}

	/**
	 * Sync particular object manually.
	 *
	 * @param array $posted_data Array of ajax posted data.
	 *
	 * @since 1.0.0
	 *
	 * @return bool|array feed ids with label.
	 */
	public function syncObjectManually( $posted_data = array() ) {
		$feed_id = ! empty( $posted_data['feed_id'] ) ?
		$posted_data['feed_id'] :
		false;

		$post_id = ! empty( $posted_data['post_id'] ) ?
		$posted_data['post_id'] :
		false;

		if ( empty( $feed_id ) || empty( $post_id ) ) {
			return array(
				'status'    => 404,
				'message'   => esc_html__( 'Required data not found.', 'integration-with-keap' ),
			);
		} else {

			if ( OrderUtil::custom_orders_table_usage_is_enabled() ) {
				// HPOS enabled
				$post_type = OrderUtil::get_order_type( $post_id );
				if ( empty( $post_type ) ) {
					$_post = get_post( $post_id );
					$post_type           = get_post( $post_id)->post_type;
				}
			} else {
				$_post = get_post( $post_id );
				$post_type           = get_post( $post_id )->post_type;
			}
			$keap = Woo_Crm_Connect_Keap::get_instance();
			$request_module = Crm_Integration_Request_Module_Keap::get_instance();

			$record_type = $keap->getFeed( $feed_id, 'crm_object' );
			$feed_title = $keap->getFeedTitle( $feed_id );
			switch ( $record_type ) {
				case 'Sales_Orders':
				case 'Contacts':
				$request_module->performShopOrderSync( $feed_id, $post_id );
					break;

				case 'Products':
				$request_module->createAndUpdateProduct( $post_id, $_post, array(), $feed_id );
					break;
			}

			return array(
				'status'    => 200,
				'message'   => esc_html__( 'Sync Completed. Check Logs for verification.', 'integration-with-keap' ),
			);
		}
	}

	/**
	 * Save Client and Secreat Key.
	 *
	 * @since 1.0.0
	 */
	public function mwb_keap_update_keys() {
		check_ajax_referer( 'ajax_nonce', 'nonce' );
		if ( isset( $_POST['client_id'] ) ) {
			$client_id = sanitize_text_field( wp_unslash( $_POST['client_id'] ) );
			update_option( 'mwb_keap_client_id', $client_id );
		}
		if ( isset( $_POST['secret_id'] ) ) {
			$secret_id = sanitize_text_field( wp_unslash( $_POST['secret_id'] ) );
			update_option( 'mwb_keap_secret_id', $secret_id );
		}
	}

	/** 
	* Sync Only Products.
	*   
	* @param array $posted_data Array of ajax posted data.  
	*   
	* @since 1.0.0  
	*   
	* @return bool|array feed ids with label.   
	*/
	public function form_bulk_product_data_sync( $posted_data = array() ) {

		
		$offset = ! empty( $posted_data['offset'] ) ?
		sanitize_text_field( wp_unslash( $posted_data['offset'] ) ) :
		'1';

		$is_onboarding = ! empty( $posted_data['isOnboarding'] ) ?
		sanitize_text_field( wp_unslash( $posted_data['isOnboarding'] ) ) :
		false;

		// Get current form submission.
		$formdata = array();
		! empty( $posted_data['form'] ) ? parse_str( sanitize_text_field( wp_unslash( $posted_data['form'] ) ), $formdata ) : '';

		$formdata = ! empty( $formdata ) ? map_deep( wp_unslash( $formdata ), 'sanitize_text_field' ) : false;

		// Feed Id.
		$feed_id = ! empty( $formdata['object-feed'] ) ?
		sanitize_text_field( wp_unslash( $formdata['object-feed'] ) ) :
		'';

		$keap = Woo_Crm_Connect_Keap::get_instance();
		$record_type = $keap->getFeed( $feed_id, 'crm_object' );

		if ( true == $is_onboarding ) {
			$is_onboarding_done = get_option( 'mwb_crm_' . $record_type . '_feed_onboarded', false );
			if ( ! empty( $is_onboarding_done ) && 'true' == $is_onboarding_done ) {

				return array(
					'result'    => true,
					'record_type'    => $record_type,
					'message'   => get_option( 'mwb_crm_' . $record_type . '_feed_final_reponse', array() ),
				);
			}
		}

		$keap = Woo_Crm_Connect_Keap::get_instance();
		$feed_title = $keap->getFeedTitle( $feed_id );

		switch ( $record_type ) {

			case 'Products':
			$woo_object = 'product';
				break;
		}

		$request_module = new Crm_Integration_Request_Module_Keap();

		$bulk_data = $request_module->getBulkData( $woo_object, $offset, $feed_id );
		
		$single_record = array();

		$single_order_record = array();
		if ( ! empty( $bulk_data['request'] ) && is_array( $bulk_data['request'] ) ) {

			$keap_api = Crm_Integration_Keap_Api::get_instance();

			$bulk_ids = ! empty( $bulk_data['bulk_ids'] ) ? $bulk_data['bulk_ids'] : array();

			$log_data = compact( 'feed_id', 'feed_title', 'bulk_ids', 'woo_object' );

			if ( 'Products' == $record_type ) {

				foreach ( $bulk_data['request'] as $key => $single_value ) {

					foreach ( $single_value as $k => $v ) {

						if ( 'product_id' !== $k ) {

							$single_record[ $k ] = $v;
						}
					}
					$woo_id = $single_value['product_id'];

					$update_prod_id = get_post_meta( $single_value['product_id'], '_mwb_keap_product_id', true );

					if ( ! empty( $update_prod_id ) ) {

						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;

						$request_data = json_encode( $single_record );

						$result = $keap_api->update_single_products(
							$record_type,
							$request_data,
							$endpoint,
							$woo_id,
							true,
							$log_data
						);

						$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );

						Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $result['data']['id'], $is_success );

					} else {

						$request_data = json_encode( $single_record );
						$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
						$result = $keap_api->create_single_record(
							$record_type,
							$request_data,
							true,
							$log_data,
							$endpoint,
							$woo_id
						);

						if ( 'Products' == $record_type ) {

							update_post_meta( $single_value['product_id'], '_mwb_keap_product_id', $result['data']['id'] );

							$is_success = Crm_Integration_For_Keap_Admin::is_success_res( $result );

							Crm_Integration_For_Keap_Admin::set_woo_response_meta_for_count( $feed_id, $woo_id, $result['data']['id'], $is_success );

						}
					}
				}
			} 
		}
		
		return array(
			'result'    => true,
			'message'   => $bulk_data,
		);
	}

	/** 
	* Create Categories and Tags
	*   
	* @param array $posted_data Array of ajax posted data.  
	*   
	* @since 1.0.0  
	*   
	* @return bool|array feed ids with label    
	*/  
	public function create_keap_default_tags_and_categories() {
		check_ajax_referer( 'ajax_nonce', 'nonce' );    
		$categories_data = array(); 
		$keap_api = Crm_Integration_Keap_Api::get_instance();   
		$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/tags/categories'; 
		$categories_data['description'] = ' Tag Category - Keap Integration For WooCommerce';   
		$categories_data['name'] = 'Keap Integration For WooCommerce';  
		$request_data = wp_json_encode( $categories_data ); 
		$category_id = get_option( 'mwb_keap_category_id', true );
		if ( empty( $category_id ) ) {  
			$result = $keap_api->create_category_for_tag( $request_data, $endpoint );   
			$code = isset($result['code']) ? sanitize_text_field( wp_unslash( $result['code'] ) ) : ''; 
			if ( isset( $result['code'] ) && '200' == $result['code'] ) {   
				if ( isset($result['data'] ) ) {    
					$id =  isset( $result['data']['id'] ) ? sanitize_text_field( wp_unslash( $result['data']['id'] ) ) : '';    
					update_option('mwb_keap_category_id', $id );
				}   
			} else {    
				$message =  'Created Successfully';
				$success = true;
				$code = 200;
			}   
		} elseif ( ! empty( $category_id ) ) {  
			$message =  'Already created';
			$result = $category_id;
			$code = 400;
			$success = true ;
		}   
		$response['result'] = $result;  
		$response['message']     = $message;    
		$response['success']     = $success;    
		$response['next_step']   = 1;   
		$response['total_count'] = 1;   
		return $response ;  
	}   

	public function create_keap_default_tags( $posted_data = array() ) {    
		check_ajax_referer( 'ajax_nonce', 'nonce' );    
		$tag_data = array();    
		$response = array();    
		$step  = isset( $_POST['step'] ) ? sanitize_text_field( wp_unslash( $_POST['step'] ) ) : '';    
		$cpt_instance = new Crm_Integration_For_Keap_Cpt(); 
		$keap_api = Crm_Integration_Keap_Api::get_instance();   
		$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/tags';
		$category_id = ! empty( get_option( 'mwb_keap_category_id', true ) ) ? sanitize_text_field( wp_unslash( get_option( 'mwb_keap_category_id', true ) ) ) : '';    
		$all_products_tags = $cpt_instance->get_all_produucts_for_tag();
		$order_statuses = array(
			'Pending payment',
			'Processing',
			'On hold',
			'Completed',
			'Cancelled',
			'Refunded',
			'Failed',
		);
		$all_products_tags = array_merge($all_products_tags, $order_statuses);
		if ( ! empty( $all_products_tags ) ) {
			foreach ($all_products_tags as $key => $value) {
				$tag_data['category'] = array( 'id' => '' . $category_id );
				$tag_data['description'] = 'Tag - Keap Integration For WooCommerce -' . $value; 
				$tag_data['name'] = $value ;    
				$request_data = json_encode( $tag_data );   
				$result = $keap_api->create_all_tag( $request_data, $endpoint );
				if ( isset( $result['code'] ) && '200' == $result['code'] ) {   
					$success = true;    
					$message = 'Created Successfully Tag';  
				} elseif ( isset( $result['code'] ) && '400' == $result['code'] ) { 
					$success = true;    
					$message = 'Already Created Tag';   
				}   
				$response['result'] = $result;  
				$response['message']     = $message;    
				$response['success']     = $success;    
				$response['next_step']   = count( $all_products_tags ); 
				$response['total_count'] = count( $all_products_tags ); 
			}   
		}
		return $response;   
	}

	/**
	 * Create filter field in feed form.
	 *
	 * @param    array $posted_data   Posted data.
	 * @since    1.0.0
	 * @return   array                Response data.
	 */
	public function create_feed_filters( $posted_data ) {

		$response = array(
			'success' => false,
			'msg'     => esc_html__( 'Something went wrong, Refresh and try again.', 'integration-with-quickbooks' ),
		);
		$module        = ! empty( $posted_data['module'] ) ? sanitize_text_field( wp_unslash( $posted_data['module'] ) ) : '';
		$feed_settings        = ! empty( $posted_data['feed_settings'] ) ? sanitize_text_field( wp_unslash( $posted_data['feed_settings'] ) ) : '';
		$crm_class                      = 'Crm_Integration_Keap_Api';
		$crm_api_module                 = $crm_class::get_instance();

		$module_data['selected_object'] = $module;
		$module_data['feed_settings']   = $feed_settings;
		$module_data['crm_fields']      = $crm_api_module->get_module_fields( $module, false );
		
		$filter_fields = $this->get_field_filter_options();
		return array(
			'formfields'    => $module_data['crm_fields'],
			'filter'  => $filter_fields,
			'success' => true,
		);
	}

	public function get_field_filter_options() {
		$filter = array(
			'exact_match'       => esc_html__( 'Matches exactly', 'integration-with-quickbooks' ),
			'no_exact_match'    => esc_html__( 'Does not match exactly', 'integration-with-quickbooks' ),
			'contains'          => esc_html__( 'Contains (Text)', 'integration-with-quickbooks' ),
			'not_contains'      => esc_html__( 'Does not contain (Text)', 'integration-with-quickbooks' ),
			'exist'             => esc_html__( 'Exist in (Text)', 'integration-with-quickbooks' ),
			'not_exist'         => esc_html__( 'Does not Exist in (Text)', 'integration-with-quickbooks' ),
			'starts'            => esc_html__( 'Starts with (Text)', 'integration-with-quickbooks' ),
			'not_starts'        => esc_html__( 'Does not start with (Text)', 'integration-with-quickbooks' ),
			'ends'              => esc_html__( 'Ends with (Text)', 'integration-with-quickbooks' ),
			'not_ends'          => esc_html__( 'Does not end with (Text)', 'integration-with-quickbooks' ),
			'less_than'         => esc_html__( 'Less than (Text)', 'integration-with-quickbooks' ),
			'greater_than'      => esc_html__( 'Greater than (Text)', 'integration-with-quickbooks' ),
			'less_than_date'    => esc_html__( 'Less than (Date/Time)', 'integration-with-quickbooks' ),
			'greater_than_date' => esc_html__( 'Greater than (Date/Time)', 'integration-with-quickbooks' ),
			'equal_date'        => esc_html__( 'Equals (Date/Time)', 'integration-with-quickbooks' ),
			'empty'             => esc_html__( 'Is empty', 'integration-with-quickbooks' ),
			'not_empty'         => esc_html__( 'Is not empty', 'integration-with-quickbooks' ),
			'only_number'       => esc_html__( 'Is only contains Numbers', 'integration-with-quickbooks' ),
			'only_text'         => esc_html__( 'Is only contains String', 'integration-with-quickbooks' ),
		);
		return $filter;
	}
	
	// End of class.
}
