<?php
/**
 * The plugin bootstrap file
 *
 * This file is read by WordPress to generate the plugin information in the plugin
 * admin area. This file also includes all of the dependencies used by the plugin,
 * registers the activation and deactivation functions, and defines a function
 * that starts the plugin.
 *
 * @link              https://wpswings.com/
 * @since             1.0.0
 * @package           Integration_With_keap
 *
 * @wordpress-plugin
 * Plugin Name:       Keap Integration for WooCommerce
 * Plugin URI:        https://woocommerce.com/products/integration-with-keap
 * Description:       Integrate your WooCommerce store with Keap CRM.
 * Version:           2.4.0
 * Requires at least:   5.5.0
 * Tested up to:        6.7.1
 * Woo: 8431264:c9fad46b31fca9377ee3441af463a9ea
 * WC requires at least:    5.5.0
 * WC tested up to:         9.5.1
 * Author:            WP Swings
 * Author URI:        https://wpswings.com/?utm_source=wpswings-keap-woo&utm_medium=woo-backend&utm_campaign=official
 * License:           GPL-3.0+
 * License URI:       http://www.gnu.org/licenses/gpl-3.0.txt
 * Text Domain:       integration-with-keap
 * Domain Path:       /languages
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

use Automattic\WooCommerce\Utilities\OrderUtil;

/**
 * Plugin Active Detection.
 *
 * @since    1.0.0
 * @param    string $plugin_slug index file of plugin.
 */
function mwb_keap_is_plugin_active( $plugin_slug = '' ) {

	if ( empty( $plugin_slug ) ) {

		return false;
	}

	$active_plugins = (array) get_option( 'active_plugins', array() );

	if ( is_multisite() ) {

		$active_plugins = array_merge( $active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );

	}

	return in_array( $plugin_slug, $active_plugins ) || array_key_exists( $plugin_slug, $active_plugins );
}

/**
 * The code that runs during plugin validation.
 * This action is checks for WooCommerce Dependency.
 *
 * @since    1.0.0
 */
function mwb_keap_plugin_activation() {

	$activation['status'] = true;
	$activation['message'] = '';

	// Dependant plugin.
	if ( ! mwb_keap_is_plugin_active( 'woocommerce/woocommerce.php' ) ) {

		$activation['status'] = false;
		$activation['message'] = 'woo_inactive';

	}

	return $activation;
}

$mwb_keap_plugin_activation = mwb_keap_plugin_activation();

if ( true === $mwb_keap_plugin_activation['status'] ) {

	/**
	 * Currently plugin version.
	 * Start at version 1.0.0 and use SemVer - https://semver.org
	 * Rename this for your plugin and update it as you release new versions.
	 */
	define( 'CRM_INTEGRATION_KEAP_VERSION', '2.4.0' );
	define( 'CRM_INTEGRATION_KEAP_PATH', plugin_dir_path( __FILE__ ) );
	define( 'CRM_INTEGRATION_KEAP_URL', plugin_dir_url( __FILE__ ) );
	if ( ! defined( 'WP_DEBUG' ) ) {
		define( 'WP_DEBUG', true );
	}

	add_action( 'before_woocommerce_init', 'declare_hpos_compatibility_keap' );

	function declare_hpos_compatibility_keap() {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}       
	}

	/**
	 * The code that runs during plugin activation.
	 * This action is documented in includes/class-crm-integration-for-keap-activator.php
	 */
	function activate_crm_integration_for_keap() {
		require_once plugin_dir_path( __FILE__ ) . 'includes/class-crm-integration-for-keap-activator.php';
		Crm_Integration_For_Keap_Activator::activate();
	}

	/**
	 * The code that runs during plugin deactivation.
	 * This action is documented in includes/class-crm-integration-for-keap-deactivator.php
	 */
	function deactivate_crm_integration_for_keap() {
		require_once plugin_dir_path( __FILE__ ) . 'includes/class-crm-integration-for-keap-deactivator.php';
		Crm_Integration_For_Keap_Deactivator::deactivate();
	}

	register_activation_hook( __FILE__, 'activate_crm_integration_for_keap' );
	register_deactivation_hook( __FILE__, 'deactivate_crm_integration_for_keap' );

	/**
	 * The core plugin class that is used to define internationalization,
	 * admin-specific hooks, and public-facing site hooks.
	 */
	require plugin_dir_path( __FILE__ ) . 'includes/class-crm-integration-for-keap.php';

	/**
	* Add settings links in plugin listing.
	*/
	add_filter( 'plugin_action_links', 'mwb_keap_admin_settings', 10, 2 );

	/**
	 * Add settings link in plugin listing.
	 *
	 * @since    1.0.0
	 * @param array  $actions actions.
	 * @param string $plugin_file plugin file path.
	 */
	function mwb_keap_admin_settings( $actions, $plugin_file ) {
		static $plugin;
		if ( ! isset( $plugin ) ) {
			$plugin = plugin_basename( __FILE__ );
		}
		if ( $plugin === $plugin_file ) {
			$settings = array(
				'settings' => '<a href="' . admin_url( 'admin.php?page=integration-with-keap' ) . '">' . __( 'Settings', 'integration-with-keap' ) . '</a>',
			);
			$actions  = array_merge( $settings, $actions );
		}
		return $actions;
	}

	/**
	* Add docs and support link.
	*/
	add_filter( 'plugin_row_meta', 'mwb_keap_plugin_row_links', 10, 2 );

	/**
	 * Add docs and support link.
	 *
	 * @since    1.0.0
	 * @param array  $links Other links.
	 * @param string $file plugin file path.
	 */
	function mwb_keap_plugin_row_links( $links, $file ) {

		if ( 'integration-with-keap/crm-integration-for-keap.php' == $file ) {
			unset($links[2]);
			$row_meta = array(
				'visit_plugin_site' => '<a href="' . esc_url( 'https://woocommerce.com/products/integration-with-keap' ) . '" target="_blank" aria-label="' . esc_attr__( 'Plugin Additional Links', 'crm-integration-for-zoho' ) . '" style="color:#2271b1;">' . esc_html__( 'Visit Plugin Site', 'crm-integration-for-zoho' ) . '</a>',
				'docs'    => '<a href="' . esc_url( 'https://woocommerce.com/document/integration-with-keap/' ) . '" target="_blank" aria-label="' . esc_attr__( 'Plugin Additional Links', 'domain' ) . '" style="color:green;"><img src="' . esc_url( CRM_INTEGRATION_KEAP_URL ) . 'assets/src/images/Documentation.svg" class="wps-info-img" alt="documentation image">' . esc_html__( 'Docs', 'integration-with-keap' ) . '</a>',
				'support'    => '<a href="' . esc_url( 'https://wpswings.com/submit-query/?utm_source=wpswings-keap-woo&utm_medium=woo-backend&utm_campaign=submit-query' ) . '" target="_blank" aria-label="' . esc_attr__( 'Plugin Additional Links', 'domain' ) . '" style="color:green;"><img src="' . esc_url( CRM_INTEGRATION_KEAP_URL ) . 'assets/src/images/Support.svg" class="wps-info-img" alt="support image">' . esc_html__( 'Support', 'integration-with-keap' ) . '</a>',
				'video'=> '<a href="' . esc_url( 'https://www.youtube.com/watch?v=763lo_c9pAM' ) . '" target="_blank" aria-label="' . esc_attr__( 'Plugin Additional Links', 'integration-with-quickbooks' ) . '" style="color:green;"><img src="' . esc_url( CRM_INTEGRATION_KEAP_URL ) . 'assets/src/images/video.svg" class="wps-info-img" alt="services image">' . esc_html__( 'Video', 'integration-with-keap' ) . '</a>',
				'demo'    => '<a href="' . esc_url( 'https://demo.wpswings.com/integration-with-keap/?utm_source=wpswings-keap-woo&utm_medium=woo-backend&utm_campaign=demo' ) . '" target="_blank" aria-label="' . esc_attr__( 'Plugin Additional Links', 'domain' ) . '" style="color:green;"><img src="' . esc_url( CRM_INTEGRATION_KEAP_URL ) . 'assets/src/images/Demo.svg" class="wps-info-img" alt="demo image">' . esc_html__( 'Demo', 'integration-with-keap' ) . '</a>',
				'services'=> '<a href="' . esc_url( 'https://wpswings.com/wordpress-woocommerce-solutions/' ) . '" target="_blank" aria-label="' . esc_attr__( 'Plugin Additional Links', 'integration-with-quickbooks' ) . '" style="color:green;"><img src="' . esc_url( CRM_INTEGRATION_KEAP_URL ) . 'assets/src/images/services.svg" class="wps-info-img" alt="services image">' . esc_html__( 'Services', 'integration-with-keap' ) . '</a>',
			);

			return array_merge( $links, $row_meta );
		}
		return (array) $links;
	}

	// replace get_post_meta with wps_keap_get_meta_data
	function wps_keap_get_meta_data( $id, $key, $v ) {
		if ( 'shop_order' === OrderUtil::get_order_type( $id ) && OrderUtil::custom_orders_table_usage_is_enabled() ) {
			// HPOS usage is enabled.
			$order    = wc_get_order( $id );
			if ( '_customer_user' == $key ) {
				$meta_val = $order->get_customer_id();
				return $meta_val;
			}
			$meta_val = $order->get_meta( $key );
			return $meta_val;
		} else {
			// Traditional CPT-based orders are in use.
			$meta_val = get_post_meta( $id, $key, $v );
			return $meta_val; 
		}
	}

	// replace update_post_meta with wps_keap_update_meta_data
	function wps_keap_update_meta_data( $id, $key, $value ) {
		if ( 'shop_order' === OrderUtil::get_order_type( $id ) && OrderUtil::custom_orders_table_usage_is_enabled() ) {
			// HPOS usage is enabled.
			$order = wc_get_order( $id );
			$order->update_meta_data( $key, $value );
			$order->save();
		} else {
			// Traditional CPT-based orders are in use.
			update_post_meta( $id, $key, $value );
		}
	}


	/**
	 * Begins execution of the plugin.
	 *
	 * Since everything within the plugin is registered via hooks,
	 * then kicking off the plugin from this point in the file does
	 * not affect the page life cycle.
	 *
	 * @since    1.0.0
	 */
	function run_crm_integration_for_keap() {

		$plugin = new Crm_Integration_For_Keap();
		$plugin->run();
	}


	run_crm_integration_for_keap();
} else {



	add_action( 'admin_init', 'mwb_keap_plugin_activation_failure' );

	/**
	 * Deactivate this plugin.
	 *
	 * @since    1.0.0
	 */
	function mwb_keap_plugin_activation_failure() {

		deactivate_plugins( plugin_basename( __FILE__ ) );
	}

	// Add admin error notice.
	add_action( 'admin_notices', 'mwb_keap_activation_admin_notice' );

	/**
	 * This function is used to display plugin activation error notice.
	 *
	 * @since    1.0.0
	 */
	function mwb_keap_activation_admin_notice() {

		global $mwb_keap_plugin_activation;

		?>

		<?php if ( 'woo_inactive' == $mwb_keap_plugin_activation['message'] ) : ?>

			<div class="notice notice-error is-dismissible mwb-notice">
				<p><strong><?php esc_html_e( 'WooCommerce', 'integration-with-keap' ); ?></strong><?php esc_html_e( ' is not activated, Please activate WooCommerce first to activate ', 'integration-with-keap' ); ?><strong><?php esc_html_e( 'CRM Intregation For Keap', 'integration-with-keap' ); ?></strong><?php esc_html_e( '.', 'integration-with-keap' ); ?></p>
			</div>

			<?php
		endif;
	}
}
