<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 *
 * @link  https://wpswings.com/
 * @since 1.0.0
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 */
use Automattic\WooCommerce\Utilities\OrderUtil;
use Automattic\WooCommerce\Internal\DataStores\Orders\OrdersTableDataStore;
class Crm_Integration_Connect_Framework_Keap {

	/**
	 *  The instance of this class.
	 *
	 * @since    1.0.0
	 * @var      string    $instance    The instance of this class.
	 */
	private static $instance;

	/**
	 * Main Crm_Integration_Connect_Framework_Keap Instance.
	 *
	 * Ensures only one instance of Crm_Integration_Connect_Framework_Keap is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @static
	 * @return Crm_Integration_Connect_Framework_Keap - Main instance.
	 */
	public static function get_instance() {

		if ( is_null( self::$instance ) ) {

			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Main Crm_Integration_Connect_Framework_Keap Instance.
	 *
	 * Ensures only one instance of Crm_Integration_Connect_Framework_Keap is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @static
	 */
	public function __construct() {
	}

	/**
	 * Main Crm_Integration_Connect_Framework_Keap Instance.
	 *
	 * @since 1.0.0
	 *
	 * @static
	 * @return Crm_Integration_Connect_Framework_Keap - Main instance.
	 */
	public static function supports() {
		return array(
			'product',
			'users',
			'shop_order',
			'abandon_carts',
		);
	}

	/**
	 * Shop Order Fields Mapping which are not in meta keys.
	 *
	 * @since 1.0.0
	 *
	 * @static
	 * @return array
	 */
	public static function shopOrderMetaExtras() {
		return array(
			'order_id' => 'ID',
			'order_status' => 'post_status',
			'order_items' => 'get_items',
			'order_date' => 'order_date',
			'order_note' => 'order_note',
			'payment_method' => 'payment_method',
			'payment_method_title' => 'payment_method_title',

		);
	}

	/**
	 * User Fields Mapping which are not in meta keys.
	 *
	 * @since 1.0.0
	 *
	 * @static
	 * @return array
	 */
	public static function userMetaExtras() {
		return array();
	}

	/**
	 * Products Fields Mapping which are not in meta keys.
	 *
	 * @since 1.0.0
	 *
	 * @static
	 * @return array
	 */
	public static function productMetaExtras() {
		return array(
			'product_id'    => 'ID',
			'product_name'  => 'get_name',
			'product_price' => 'get_regular_price',
			'product_sku'   => 'get_sku',
			'product_short_description' => 'get_short_description',
			'product_description'   => 'get_description',
			'product_active' => 'true',
			'product_subscription' => 'false',
		);
	}

	/**
	 * Returns the mapping index from Woo we require.
	 *
	 * @param string $post_type        The object we need index for.
	 * @param string $custom_post_type If required a custom post type.
	 *
	 * @since  1.0.0
	 * @return array|bool          The current mapping step required.
	 */
	public function get_wp_meta( $post_type = false, $custom_post_type = false ) {

		$status = 'publish';
		global $wpdb;
		switch ( $post_type ) {
			case 'product':
				$result = $wpdb->get_col( $wpdb->prepare(
					"SELECT DISTINCT( $wpdb->postmeta.meta_key )
                FROM $wpdb->posts
                LEFT JOIN $wpdb->postmeta
                ON $wpdb->posts.ID = $wpdb->postmeta.post_id
                WHERE $wpdb->posts.post_type = %s
                AND $wpdb->postmeta.meta_key != ''",
					$post_type
				));
				
				if ( ! empty( self::productMetaExtras() ) && 'product' == $post_type ) {
					$result = array_merge( $result, array_keys( self::productMetaExtras() ) );

				} elseif ( ! empty( self::shopOrderMetaExtras() ) && 'shop_order' == $post_type ) {
					$result = array_merge(
						$result,
						array_keys( self::shopOrderMetaExtras() )
					);

				}

				break;
				
			case 'shop_order':
				if ( OrderUtil::custom_orders_table_usage_is_enabled()  && 'product' != $post_type ) {
					$ord_meta = $wpdb->prefix . 'wc_orders_meta';
					$data_store = wc_get_container()->get( OrdersTableDataStore::class );
					$columns_mapping   = $data_store->get_internal_meta_keys();
					$result_order_data = array_unique($columns_mapping);
					$result = $result_order_data;
				} else {
					$result = $wpdb->get_col( $wpdb->prepare(
						"SELECT DISTINCT( $wpdb->postmeta.meta_key )
						FROM $wpdb->posts
						LEFT JOIN $wpdb->postmeta
						ON $wpdb->posts.ID = $wpdb->postmeta.post_id
						WHERE $wpdb->posts.post_type = %s
						AND $wpdb->postmeta.meta_key != ''",
						$post_type
					));
				}
				
				if ( ! empty( self::productMetaExtras() ) && 'product' == $post_type ) {
					$result = array_merge( $result, array_keys( self::productMetaExtras() ) );

				} elseif ( ! empty( self::shopOrderMetaExtras() ) && 'shop_order' == $post_type ) {
					$result = array_merge(
						$result,
						array_keys( self::shopOrderMetaExtras() )
					);

				}

				break;

			case 'users':
				$query = "SELECT distinct $wpdb->usermeta.meta_key FROM 
            $wpdb->usermeta";

				$result = $this->format_query_object(
					mwb_woo_keap_get_query_results( $query )
					
				);
				break;

			case 'abandon_carts':
				$query  = 'SELECT `session_value`, `session_id` FROM `wp_woocommerce_sessions`';
				
				$result = $this->format_query_object(
					mwb_woo_keap_get_query_results( $query )
				);
				break;

			case 'custom':
				$result = $wpdb->get_col( $wpdb->prepare(
					"SELECT DISTINCT( $wpdb->postmeta.meta_key )
                FROM $wpdb->posts
                LEFT JOIN $wpdb->postmeta
                ON $wpdb->posts.ID = $wpdb->postmeta.post_id
                WHERE $wpdb->posts.post_type = %s
                AND $wpdb->postmeta.meta_key != ''",
					$custom_post_type
				) );
				
				break;
		}
		return ! empty( $result ) ? $result : false;
	}

	/**
	 * Returns the mapping index from Woo we require.
	 *
	 * @param array $array The query object result.
	 *
	 * @since 1.0.0
	 *
	 * @return array The formatted data as raw array
	 */
	public function format_query_object( $array = false ) {

		if ( empty( $array ) ) {
			return;
		}

		$formatted_array = array();

		foreach ( $array as $key => $value ) {
			$dataset = array_values( $value );
			if ( ! empty( $dataset[0] ) ) {
				array_push( $formatted_array, $dataset[0] );
			}
		}

		return $formatted_array;
	}


	/**
	 * Returns the requested data values for required index from Woo we require.
	 *
	 * @param string $obj_type The post type.
	 * @param string $key      The post meta key.
	 * @param string $id       The post object id.
	 *
	 * @since 1.0.0
	 *
	 * @return string               The post meta values.
	 */
	public function getPropValue( $obj_type = false, $key = false, $id = false, $nonce = false ) {

		if ( false == $key || false == $id ) {
			return;
		}

		if ( empty( $obj_type ) || ! in_array( $obj_type, self::supports() ) ) {
			return;
		}

		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $nonce ) ), 'mwb-create-prop' ) ) {
			return;
		}

		switch ( $obj_type ) {

			case 'users':
				if ( array_key_exists( $key, self::userMetaExtras() ) ) {

					$callback = self::userMetaExtras()[ $key ];
					if ( 'ID' == $callback ) {
						$response = $id;
					} else {
						$user = get_user_by( 'id', $id );
					}
				} else {
					$response = ! empty( $_POST[ $key ] ) ? sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) :
					get_user_meta( $id, $key, true );
				}
				break;

			case 'shop_order':
				if ( array_key_exists( $key, self::shopOrderMetaExtras() ) ) {

					$callback = self::shopOrderMetaExtras()[ $key ];

					if ( 'ID' == $callback ) {
						$response = $id;

					} elseif ( 'post_status' == $callback ) {

						$response = get_post( $id )->post_status;

					} elseif ( 'order_date' == $callback ) {
						$order = wc_get_order( $id );
						$date = $order->order_date;
						$response = gmdate( 'Y-m-d\TH:i:s.uZ', strtotime( $date ) ) . 'Z';

					} elseif ( 'payment_method' == $callback ) {
						$order = wc_get_order( $id );
						$method = $order->get_payment_method();
						if ( 'cod' == $method ) {
							$response = 'CASH';
						} elseif ( 'cheque' == $method ) {
							$response = 'CHECK';
						} else {
							$response = 'CREDIT_CARD';
						}
					} elseif ( 'payment_method_title' == $callback ) {
						$order = wc_get_order( $id );
						$response = $order->get_payment_method_title();

					} elseif ( 'order_note' == $callback ) {
						$order = wc_get_order( $id );
						$response = $order->customer_note;
					} else {
						$response = wc_get_order( $id )->$callback();

					}
				} else {

					$response = ! empty( $_POST[ $key ] ) ? sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) : wps_keap_get_meta_data( $id, $key, true );

				}

				break;

			case 'product':
				if ( array_key_exists( $key, self::productMetaExtras() ) ) {

					$callback = self::productMetaExtras()[ $key ];

					if ( 'ID' == $callback ) {
						$response = $id;

					} else {
						$product  = wc_get_product( $id );

						if ( ! empty( $product ) ) {
							$response = $product->$callback();
						}
					}
				} else {
					$response = ! empty( $_POST[ $key ] ) ? sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) : get_post_meta( $id, $key, true );
				}

				break;

			default:
				// code...
				break;
		}
		return $response;
	}


	/**
	 * Current Woo meta keys with Labels.
	 *
	 * @param array $dataset array for woo keys.
	 *
	 * @since 1.0.0
	 *
	 * @return array - Current Woo meta keys with Labels to Woo keys.
	 */
	public function parseLabels( $dataset ) {

		if ( ! empty( $dataset ) && is_array( $dataset ) ) {
			foreach ( $dataset as $key => $value ) {
				if ( ! empty( $value ) && is_array( $value ) ) {
					foreach ( $value as $k => $v ) {

						// Initialise the key via wp index.
						$dataset[ $key ][ $v ] = false;

						// Unset the numeric index.
						unset( $dataset[ $key ][ $k ] );

						// Create Wp index.
						$k = $v;

						// Create Wp Labels.
						$v = str_replace( '_', ' ', $v );
						$v = str_replace( '-', ' ', $v );
						$v = ucwords( $v );
						$dataset[ $key ][ $k ] = $v;
					}
				}
			}

			return $dataset;
		}

		return $dataset;
	}

	/**
	 * Replace the occurence within the string only once.
	 *
	 * @param string $from    The sub-string before replace.
	 * @param string $to      The sub-string after replace.
	 * @param string $content The string from which we operate.
	 *
	 * @since  1.0.0
	 * @return array - Current Woo meta keys with Labels to Woo keys.
	 */
	public function strReplaceFirst( $from, $to, $content ) {
		$from = '/' . preg_quote( $from, '/' ) . '/';

		return preg_replace( $from, $to, $content, 1 );
	}

	/**
	 * Get other associated keap object in required format.
	 *
	 * @param string $woo_obj_type   The WP post type.
	 * @param string $lookup_feed_id The Feed id to get lookup request.
	 * @param string $lookup_type    The CRM Object required id.
	 * @param string $woo_id         The WP post id.
	 *
	 * @since  1.0.0
	 * @return array - Current Woo meta keys with Labels to Woo keys.
	 */
	public function resolveLookup(
		$woo_obj_type,
		$lookup_feed_id,
		$lookup_type,
		$woo_id
	) {

		if ( 'publish' == get_post_status( $lookup_feed_id ) ) {
			switch ( $woo_obj_type ) {
				default:
					$status_obj = 'shop_order';
					break;
			}

			$keap_association_id = wps_keap_get_meta_data(
				$woo_id,
				'mwb_keap_feed_' . $lookup_feed_id . '_association',
				true
			);

			if ( empty( $keap_association_id ) ) {

				$request = $this->getRequest(
					$status_obj,
					$lookup_feed_id,
					$woo_id
				);

				$record_type = $this->getFeed( $lookup_feed_id, 'crm_object' );

				$log_data = array(
					'woo_id'   => $woo_id,
					'feed_id'  => $lookup_feed_id,
					'woo_object' => $status_obj,
				);

				$crm_integration_keap_api = Crm_Integration_Keap_Api::get_instance();
				$endpoint = '';
				$result = $crm_integration_keap_api->create_single_record(
					$record_type,
					$request,
					false,
					$log_data
				);

				$keap_association_id = wps_keap_get_meta_data(
					$woo_id,
					'mwb_keap_feed_' . $lookup_feed_id . '_association',
					true
				);
			}
		}
		return ! empty( $keap_association_id ) ? $keap_association_id : '';
	}

	/**
	 * Returns the feed data we require.
	 *
	 * @param string $feed_id  The object id for post type feed.
	 * @param string $meta_key The object meta key for post type feed.
	 *
	 * @return array|bool The current data for required object.
	 *
	 * @since 1.0.0
	 */
	public function getFeed( $feed_id = false, $meta_key = 'mapping_data' ) {
		if ( false == $feed_id ) {
			return;
		}

		$mapping = get_post_meta( $feed_id, $meta_key, true );

		if ( empty( $mapping ) ) {
			$mapping = false;
		}

		return $mapping;
	}

	/**
	 * Returns the mapping step we require.
	 *
	 * @param string $obj_type The object post type.
	 * @param string $feed_id  The mapped feed for associated objects.
	 * @param string $obj_id   The object post id.
	 * @param string $record_type   The record_type post id.
	 *
	 * @since  1.0.0
	 * @return array The current mapping step required.
	 */
	public function getRequest( $obj_type = false, $feed_id = false, $obj_id = false, $record_type = '' ) {
		if ( false == $obj_type || false == $feed_id || false == $obj_id ) {
			return;
		}

		$keap_api = Crm_Integration_Keap_Api::get_instance();   
		$tags_id = array(); 
		$tag_data = array();

		$feed = $this->getFeed( $feed_id );

		if ( empty( $feed ) ) {
			return false;
		}

		// Process Feeds.
		$response = array();
		$tags = array();
		
		foreach ( $feed as $k => $mapping ) {

			$field_type = ! empty( $mapping['field_type'] ) ?
			$mapping['field_type'] : 'standard_field';

			switch ( $field_type ) {

				case 'standard_field':
					$is_status_mapping_required = ! empty( $mapping['use_status_mapping'] )
					&& 'yes' == $mapping['use_status_mapping']
					? true :
					false;

					$field_format = ! empty( $mapping['field_value'] ) ?
					$mapping['field_value'] : '';

					// If lookup field.
					if ( 0 === strpos( $field_format, 'feeds_' ) ) {

						$lookup_feed_id = str_replace( 'feeds_', '', $field_format );

						$field_value = $this->resolveLookup(
							$obj_type,
							$lookup_feed_id,
							$k,
							$obj_id
						);

					} elseif ( true == $is_status_mapping_required ) {

						$status_mapping = $this->getFeed( $feed_id, 'status_mapping' );

						$order_status = $this->getPropValue(
							$obj_type,
							'order_status',
							$obj_id,
							wp_create_nonce( 'mwb-create-prop' )
						);

						$field_value = ! empty( $status_mapping[ $order_status ] )
						? $status_mapping[ $order_status ]
						: '';

					} else { // Just a standard meta key.

						$obj_required = strtok( $field_format, '_' );

						$meta_key = $this->strReplaceFirst(
							$obj_type . '_',
							'',
							$field_format
						);

						$field_value = $this->getPropValue(
							$obj_type,
							$meta_key,
							$obj_id,
							wp_create_nonce( 'mwb-create-prop' )
						);
					
					}

					break;

				case 'custom_value':
					$field_key = ! empty( $mapping['custom_value'] ) ?
					$mapping['custom_value'] : '';

					preg_match_all( '/{(.*?)}/', $field_key, $dynamic_strings );

					if ( ! empty( $dynamic_strings[1] ) ) {
						$dynamic_values = $dynamic_strings[1];
						foreach ( $dynamic_values as $key => $value ) {
							$prefix = $obj_type . '_';
							$field_format = $this->strReplaceFirst(
								$prefix,
								'',
								$value
							);

							$field_value = $this->getPropValue(
								$obj_type,
								$field_format,
								$obj_id,
								wp_create_nonce( 'mwb-create-prop' )
							);

							$substr = '{' . $prefix . $field_format . '}';

							$field_key = str_replace(
								$substr,
								$field_value,
								$field_key
							);

							$field_value = $field_key;
						}
					}

					break;
			}

			$response[ $k ] = ! empty( $field_value ) ? $field_value : '';
			if ( Crm_Integration_For_Keap_Admin::is_tag_enable() ) {

				$order_items = $this->getPropValue( $obj_type, 'order_items', $obj_id, wp_create_nonce( 'mwb-create-prop' ) );  
				if ( !empty($order_items) ) {   
					foreach ( $order_items as $items_key => $items_value ) {    
						$items_id = ! empty( $items_value->get_variation_id() ) ? $items_value->get_variation_id() : $items_value->get_product_id();    
						$product = wc_get_product($items_id);   
						if ($product) { 
							if ( ! empty($product->get_sku() ) && Crm_Integration_For_Keap_Admin::is_skus_tag_enable() ) {  
								$tags[] = $product->get_sku();  
							}   
							if ( ! empty($product->get_type() ) && Crm_Integration_For_Keap_Admin::is_products_type_tag_enable() ) {    
								$tags[] = $product->get_type(); 
							}   
							if ( ! empty($product->get_name() ) && Crm_Integration_For_Keap_Admin::is_products_bought_tag_enable() ) {  
								$product_name = $product->get_name();   
								$tags[] = str_replace(',', '', $product_name);  
							}   
						}   
					}   
				}
				$ord = wc_get_order( $obj_id );
				if ( !empty( $ord ) ) {
					$order_status  = ucfirst( $ord->get_status() );
					if ( !empty( $order_status )  && Crm_Integration_For_Keap_Admin::is_order_status_tag_enable() ) {
						$tags[] = $order_status;
					}
				}
			}
		}

		$ord = wc_get_order( $obj_id );
		if ( !empty( $ord ) ) {
			$order_items  = $ord->get_items();
			if ( !empty( $order_items ) ) {
				foreach ( $order_items as $items_key => $items_value ) {
					$items_id = ! empty( $items_value->get_variation_id() ) ? $items_value->get_variation_id() : $items_value->get_product_id();
					$product = wc_get_product($items_id);
					if ($product) {
						$products_id = $product->get_id();
						$custom_tags = get_post_meta( $products_id, 'mwb_keap_custom_tags', true );
						$custom_tags = ! empty( $custom_tags ) ? explode( ',', $custom_tags ) : array();
						if ( ! empty( $custom_tags ) && is_array( $custom_tags ) ) {
							foreach ( $custom_tags as $tag ) {
								if ( ! empty( $tag ) ) {
									$tags[] = $tag;
								}
							}
						}
					}
				}
			}
		}

		$tags = array_unique($tags);
		
		if ( ! empty($tags) ) {  
			$tags = array_unique($tags);
			
			$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/tags';    
			$category_id = get_option( 'mwb_keap_category_id', true );  
			foreach ($tags as $key => $value) { 
				$tag_data['category'] = array( 'id' => '' . $category_id ); 
				$tag_data['description'] = $value;
				$tag_data['name'] = $value;
				$request_data = wp_json_encode($tag_data);
	
				$result = $keap_api->create_all_tag( $request_data, $endpoint );
				if ( isset( $result['code'] ) && '200' == $result['code'] ) {   
					if (! empty( $result['data']['id'] ) ) {    
						$tags_id[] = $result['data']['id'];   
					}   
				} elseif ( isset( $result['code'] ) && '400' == $result['code'] ) { 
					$endpoint = 'https://api.infusionsoft.com/crm/rest//tags';
					$url = $endpoint . ( parse_url($endpoint, PHP_URL_QUERY) ? '&' : '?' ) . 'name=' . $value;
					$query = parse_url($endpoint, PHP_URL_QUERY);

					// Returns a string if the URL has parameters or NULL if not
					if ($query) {
						$endpoint .= '&name=' . $value;
					} else {
						$endpoint .= '?name=' . $value;
					}
					$result = $keap_api->get_tags_id( $value , $endpoint );

					if (! empty( $result ) ) {
						$tags_id[] = $result[0]['id'];
					}
				}
			}
		}

		if ( is_array( $tags_id ) ) {
			$tag_ids = implode(',', $tags_id);
		} else {
			$tag_ids = $tags_id;
		}
// print_r($tag_ids);
// 		die('kk');
		

		$add_line_item = $this->getFeed( $feed_id, 'add_line_item' );

		if ( 'Contacts' == $record_type ) {
			$add_line_item = 'no';
			$response['tagIds'] = $tag_ids;
		}

		if ( 'Sales_Orders' == $record_type ) {
			$add_line_item = 'yes';
			if ( 'yes' == $add_line_item ) {

				$order_items = $this->getPropValue( $obj_type, 'order_items', $obj_id, wp_create_nonce( 'mwb-create-prop' ) );
				$order_date = $this->getPropValue( $obj_type, 'order_date', $obj_id, wp_create_nonce( 'mwb-create-prop' ) );
				$order_note = $this->getPropValue( $obj_type, 'order_note', $obj_id, wp_create_nonce( 'mwb-create-prop' ) );
				$payment_method = $this->getPropValue( $obj_type, 'payment_method', $obj_id, wp_create_nonce( 'mwb-create-prop' ) );
				$payment_title = $this->getPropValue( $obj_type, 'payment_method_title', $obj_id, wp_create_nonce( 'mwb-create-prop' ) );
				$line_items = array();
				$products_price = '';
				$response['tagIds'] = $tag_ids;
				$ord = wc_get_order( $obj_id );
				if ( ! empty( $ord ) ) {
					$ord_data = $ord->get_data();
					if ( !empty( $ord_data['coupon_lines'] ) ) {
						foreach ( $ord_data['coupon_lines'] as $coupon_key => $coupon_value ) {
							$coupon_code_data = $coupon_value->get_data();
							$coupon_code = $coupon_code_data['code'];
							$response['promo_codes'] = $coupon_code;
						}
					}
				}

				foreach ( $order_items as $items_key => $items_value ) {
					
					$items_id = ! empty( $items_value->get_variation_id() ) ? $items_value->get_variation_id() : $items_value->get_product_id();

					$product = wc_get_product($items_id);

					if ($product) {
						
						$products_price = $product->get_price();
					}
					
					$crm_association_id = wps_keap_get_meta_data( $items_id, 'mwb_keap_feed_' . $feed_id . '_association', true );

					// If not present sync the product instantly.
					if ( empty( $crm_association_id ) ) {

						$record_type = $this->getFeed( $feed_id, 'crm_object' );

						$log_data = array(
							'woo_id'   => $obj_id,
							'feed_id'  => $feed_id,
							'woo_object' => 'shop_order',
						);

						$crm_association_id = wps_keap_get_meta_data(
							$items_id,
							'_mwb_keap_product_id',
							true
						);
					}

					$product_description = get_post( $items_value->get_product_id() )->post_content;

					$item = array(

						'description' => $product_description,
						'price' => $products_price,
						'product_id' => $crm_association_id,
						'quantity' => $items_value->get_quantity(),
					);

					array_push( $line_items, $item );

				}
				
				$response['order_date'] = $order_date;
				$response['order_items'] = $line_items;
				$keap_api = Crm_Integration_Keap_Api::get_instance();
				$request_data = $keap_api->json_order_data( $response, $feed_id, $obj_id );
				$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/orders';
				$response['order_id'] = $obj_id;
				$response['payment_method'] = $payment_method;
				$response['payment_title'] = $payment_title;
				$response['order_note'] = $order_note;
			}
		}
		
		return $response;
	}

	/**
	 * Returns the mapping step we require.
	 *
	 * @param string $feed_type The CRM Object post type.
	 *
	 * @since 1.0.0
	 *
	 * @return bool|array single or multiple feed ids.
	 */
	public function getObjectFeed( $feed_type = false ) {
		if ( empty( $feed_type ) ) {
			return false;
		}

		$args = array(
			'post_type' => 'mwb_keap_feed',
			'post_status' => 'publish',
			'fields' => 'ids',
			'meta_query' => array(
				'relation' => 'AND',
				array(
					'key' => 'crm_object',
					'compare' => '=',
					'value' => $feed_type,
				),
			),
		);

		$feeds = get_posts( $args );

		if ( ! empty( $feeds ) ) {
			if ( count( $feeds ) > 1 ) {
				$feed_id = $feeds;
			} else {
				$feed_id = reset( $feeds );
			}

			return $feed_id;
		}
		return false;
	}

	/**
	 * Get title of a particur feed.
	 *
	 * @param string $feed_id Id of feed.
	 *
	 * @return array.
	 */
	public function getFeedTitle( $feed_id ) {
		$title   = esc_html( 'Feed #' ) . $feed_id;
		$feed    = get_post( $feed_id );
		$title   = ! empty( $feed->post_title ) ? $feed->post_title : $title;
		return $title;
	}

	/**
	 * Returns the hooks/callbacks/priority for each active feed.
	 *
	 * @since  1.0.0
	 * @return string The current object event Hook.
	 */
	public function getHookRequestsForFeed() {
		$feed_ids = $this->getAllFeed();

		if ( empty( $feed_ids ) ) {
			return;
		}

		$requested_hooks = array();

		foreach ( $feed_ids as $key => $feed_id ) {
			if ( ! empty( $feed_id ) ) {

				$event = $this->getFeed( $feed_id, 'feed_event' );

				$record_type = $this->getFeed(
					$feed_id,
					'crm_object'
				);
				$requested_hooks[ $feed_id ] = array(
					'hook'      => $this->getHook( $event, $record_type ),
					'callback'  => $this->getCallback( $event, $record_type ),
				);
			}
		}

		return $requested_hooks;
	}

	/**
	 * Returns the callbacks for each active feed.
	 *
	 * @param string $event  The event for feed.
	 * @param string $object The CRM object for feed.
	 *
	 * @since  1.0.0
	 * @return string The current object event Hook.
	 */
	public function getCallback( $event = '', $object = false ) {
		if ( empty( $event ) || empty( $object ) ) {
			return;
		}

		switch ( $event ) {
			case 'new-order':
				$callback = 'shopOrderUpdated';
				break;

			case 'wc-pending':
				$callback = 'shopOrderUpdated';
				break;

			case 'wc-processing':
				$callback = 'shopOrderUpdated';
				break;

			case 'wc-on-hold':
				$callback = 'shopOrderUpdated';
				break;

			case 'wc-completed':
				$callback = 'shopOrderUpdated';
				break;

			case 'wc-cancelled':
				$callback = 'shopOrderUpdated';
				break;

			case 'wc-refunded':
				$callback = 'shopOrderRefunded';
				break;

			case 'wc-failed':
				$callback = 'shopOrderUpdated';
				break;

			case 'status_change':
				$callback = 'shopOrderStatusChanged';
				break;

			case 'product_update_create':
				$callback = 'createAndUpdateProduct';
				break;

			case 'user_update_create':
				$callback = 'createAndUpdateContact';
				break;
				
			case 'send_manually':
				switch ( $object ) {

					case 'Sales_Orders':
					case 'Contacts':
						$callback = 'createAndUpdateShopOrder';
						break;

					case 'Products':
						$callback = 'createAndUpdateProduct';
						break;

				}
				break;

			default:
				break;
		}
		return $callback;
	}

	/**
	 * Returns the hooks for each active feed.
	 *
	 * @param string $event  The event for feed.
	 * @param string $object The CRM object for feed.
	 *
	 * @since  1.0.0
	 * @return string The current object event Hook.
	 */
	public function getHook( $event = '', $object = false ) {

		$hook = '';
		switch ( $event ) {

			case 'new-order':
				$hook = 'woocommerce_new_order';
				break;

			case 'wc-pending':
				$hook = 'woocommerce_order_status_pending';
				break;

			case 'wc-processing':
				$hook = 'woocommerce_order_status_processing';
				break;

			case 'wc-on-hold':
				$hook = 'woocommerce_order_status_on-hold';
				break;

			case 'wc-completed':
				$hook = 'woocommerce_order_status_completed';
				break;

			case 'wc-cancelled':
				$hook = 'woocommerce_order_status_cancelled';
				break;

			case 'wc-refunded':
				$hook = array(
					'woocommerce_order_fully_refunded',
					'woocommerce_order_partially_refunded',
				);
				break;

			case 'wc-failed':
				$hook = 'woocommerce_order_status_failed';
				break;

			case 'status_change':
				$hook = array(
					'woocommerce_order_status_pending',
					'woocommerce_order_status_failed',
					'woocommerce_order_fully_refunded',
					'woocommerce_order_partially_refunded',
					'woocommerce_order_status_cancelled',
					'woocommerce_order_status_completed',
					'woocommerce_order_status_on-hold',
					'woocommerce_order_status_processing',
				);
				break;

			case 'product_update_create':
				$hook = 'save_post_product';
				break;

			case 'user_update_create':
				$hook = 'woocommerce_new_order';
				break;

			case 'send_manually':
				switch ( $object ) {
					case 'Sales_Orders':
					case 'Deals':
					case 'Contacts':
						$hook = 'save_post_shop_order';
						break;

					case 'Products':
						$hook = 'save_post_product';
						break;

				}
				break;

			default:
				break;
		}

		return $hook;
	}

	/**
	 * Returns the hooks for each active feed.
	 *
	 * @param string $hook The hooks for feed.
	 *
	 * @since  1.0.0
	 * @return string The current hooks param count.
	 */
	public function getHookParamCount( $hook = '' ) {

		$count = '0';
		if ( is_array( $hook ) ) {
			if ( in_array( 'woocommerce_order_fully_refunded', $hook ) ) {
				$hook = 'woocommerce_order_fully_refunded';
			}
		}

		switch ( $hook ) {

			case 'woocommerce_new_order':
				$count = '2';
				break;

			case 'woocommerce_order_status_pending':
				$count = '1';
				break;

			case 'woocommerce_order_status_processing':
				$count = '1';
				break;

			case 'woocommerce_order_status_on-hold':
				$count = '1';
				break;

			case 'woocommerce_order_status_completed':
				$count = '1';
				break;

			case 'woocommerce_order_status_cancelled':
				$count = '1';
				break;

			case 'woocommerce_order_status_failed':
				$count = '1';
				break;

			case 'woocommerce_order_fully_refunded':
			case 'woocommerce_order_partially_refunded':
				$count = '2';
				break;

			case 'save_post_shop_order':
				$count = '3';
				break;

			case 'save_post_product':
				$count = '3';
				break;

			default:
				break;
		}

		return $count;
	}

	/**
	 * Returns the mapping step we require.
	 *
	 * @since 1.0.0
	 *
	 * @return bool|array single or multiple feed ids.
	 */
	public function getAllFeed() {

		$args = array(
			'post_type' => 'mwb_keap_feed',
			'post_status' => 'publish',
			'fields' => 'ids',
			'posts_per_page' => -1,
		);

		$feeds = get_posts( $args );

		if ( ! empty( $feeds ) ) {
			return $feeds;
		}
		return false;
	}


	/**
	 * Feeds Condtional html.
	 *
	 * @param     string $and_condition  The and condition of current html.
	 * @param     string $and_index      The and offset of current html.
	 * @param     string $or_index       The or offset of current html.
	 * @since     1.0.0
	 * @return    mixed
	 */
	public function render_and_conditon( $and_condition = array(), $and_index = '1', $or_index = '' ) {

		if ( empty( $and_index ) || empty( $and_condition ) || empty( $or_index ) ) {
			return;
		}

		?>
		<div class="and-condition-filter" data-and-index=<?php echo esc_attr( $and_index ); ?> >
			<select name="condition[<?php echo esc_html( $or_index ); ?>][<?php echo esc_html( $and_index ); ?>][field]"  class="condition-form-field">
				<option value="-1" ><?php esc_html_e( 'Select Field', 'integration-with-quickbooks' ); ?></option>
				<?php foreach ( $and_condition['form_fields'] as $key => $value ) : ?>
						<?php foreach ( $value as $index => $field ) : ?>
							<option value="<?php echo esc_html( $key ); ?>" <?php selected( $and_condition['field'], $key ); ?> ><?php echo esc_html( $field ); ?></option>
						<?php 
						break;
					   endforeach; 
						?>
				<?php endforeach; ?>
			</select>
			<select name="condition[<?php echo esc_html( $or_index ); ?>][<?php echo esc_html( $and_index ); ?>][option]" class="condition-option-field">
				<option value="-1"><?php esc_html_e( 'Select Condition', 'integration-with-quickbooks' ); ?></option>
				<?php foreach ( $this->get_avialable_form_filters() as $key => $value ) : ?>
					<option value="<?php echo esc_html( $key ); ?>" <?php selected( $and_condition['option'], $key ); ?>><?php echo esc_html( $value ); ?></option>
				<?php endforeach; ?>
			</select>
			<input type="text" name="condition[<?php echo esc_html( $or_index ); ?>][<?php echo esc_html( $and_index ); ?>][value]" class="condition-value-field" value="<?php echo esc_html( ! empty( $and_condition['value'] ) ? $and_condition['value'] : '' ); ?>" placeholder="<?php esc_html_e( 'Enter value', 'integration-with-quickbooks' ); ?>" >
			<?php if ( 1 != $and_index ) : // @codingStandardsIgnoreLine ?>
				<span class="dashicons dashicons-no"></span>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Feeds conditional filter options.
	 *
	 * @since     1.0.0
	 * @return    array
	 */
	public function get_avialable_form_filters() {

		$filter = array(
			'exact_match'       => esc_html__( 'Matches exactly', 'integration-with-quickbooks' ),
			'no_exact_match'    => esc_html__( 'Does not match exactly', 'integration-with-quickbooks' ),
			'contains'          => esc_html__( 'Contains (Text)', 'integration-with-quickbooks' ),
			'not_contains'      => esc_html__( 'Does not contain (Text)', 'integration-with-quickbooks' ),
			'exist'             => esc_html__( 'Exist in (Text)', 'integration-with-quickbooks' ),
			'not_exist'         => esc_html__( 'Does not Exist in (Text)', 'integration-with-quickbooks' ),
			'starts'            => esc_html__( 'Starts with (Text)', 'integration-with-quickbooks' ),
			'not_starts'        => esc_html__( 'Does not start with (Text)', 'integration-with-quickbooks' ),
			'ends'              => esc_html__( 'Ends with (Text)', 'integration-with-quickbooks' ),
			'not_ends'          => esc_html__( 'Does not end with (Text)', 'integration-with-quickbooks' ),
			'less_than'         => esc_html__( 'Less than (Text)', 'integration-with-quickbooks' ),
			'greater_than'      => esc_html__( 'Greater than (Text)', 'integration-with-quickbooks' ),
			'less_than_date'    => esc_html__( 'Less than (Date/Time)', 'integration-with-quickbooks' ),
			'greater_than_date' => esc_html__( 'Greater than (Date/Time)', 'integration-with-quickbooks' ),
			'equal_date'        => esc_html__( 'Equals (Date/Time)', 'integration-with-quickbooks' ),
			'empty'             => esc_html__( 'Is empty', 'integration-with-quickbooks' ),
			'not_empty'         => esc_html__( 'Is not empty', 'integration-with-quickbooks' ),
			'only_number'       => esc_html__( 'Is only contains Numbers', 'integration-with-quickbooks' ),
			'only_text'         => esc_html__( 'Is only contains String', 'integration-with-quickbooks' ),
		);

		return $filter;
	}

	// End of class.
}
