<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://wpswings.com/
 * @since      1.0.0
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/includes
 */
class Crm_Integration_For_Keap {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @var      Crm_Integration_For_Keap_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'CRM_INTEGRATION_KEAP_VERSION' ) ) {
			$this->version = CRM_INTEGRATION_KEAP_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'integration-with-keap-for-woocommerce';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->initialize_cpt();
		$this->define_woo_hooks();
		$this->define_ajax_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Crm_Integration_For_Keap_Loader. Orchestrates the hooks of the plugin.
	 * - Crm_Integration_For_Keap_I18n. Defines internationalization functionality.
	 * - Crm_Integration_For_Keap_Admin. Defines all hooks for the admin area.
	 * - Crm_Integration_For_Keap_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 */
	private function load_dependencies() {

		/**
		 * The file reponsible for defining all gobal functions used in plugin.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'woo-includes/woo-functions.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'woo-crm-fw/api/class-crm-integration-api-base.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'woo-crm-fw/api/class-crm-integration-keap-api.php';

		/**
		 * The class responsible for defining all woo concerned data retrival.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'woo-crm-fw/woo/class-crm-integration-connect-framework.php';

		/**
		 * The class responsible for defining all woo concerned requests.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'woo-crm-fw/woo/class-crm-integration-request-module.php';

		/**
		 * The class responsible for defining all actions that occur in the ajax request.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-crm-integration-for-keap-ajax.php';

		/**
		 * The class responsible for defining all actions related to cpt.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-crm-integration-for-keap-cpt.php';

		/**
		 * The class responsible for defining api modules respective to keap.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-crm-integration-for-keap-fw.php';

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-crm-integration-for-keap-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'includes/class-crm-integration-for-keap-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( __DIR__ ) . 'admin/class-crm-integration-for-keap-admin.php';

		$this->loader = new Crm_Integration_For_Keap_Loader();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Crm_Integration_For_Keap_I18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 */
	private function set_locale() {

		$plugin_i18n = new Crm_Integration_For_Keap_I18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Crm_Integration_For_Keap_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

			// Add plugins menu page.
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu_page_keap', 99 );
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'remove_default_submenu_keap', 99 );
		// All admin actions and filters after License Validation goes here.
		$this->loader->add_filter( 'wps_add_plugins_menus_array', $plugin_admin, 'crm_admin_submenu_page_keap', 15 );
		$this->loader->add_filter( 'wps_add_crm_plugins_keap_menus_array', $plugin_admin, 'crm_admin_submenu_page_keap', 15 );
		$this->loader->add_action( 'parent_file', $plugin_admin, 'prefix_highlight_taxonomy_parent_menu' );
		$this->loader->add_filter( 'submenu_file', $plugin_admin, 'crm_bfw_set_submenu_file_to_handle_menu_for_wp_pages', 10, 2 );

		// Initiate all functionality of crm.
		$this->loader->add_action( 'woocommerce_init', $plugin_admin, 'woo_crm_init' );
		$this->loader->add_action( 'admin_init', $plugin_admin, 'admin_init_action' );

		// Specify components.
		$this->loader->add_filter( 'mwb_crm_woo_tabs', $plugin_admin, 'return_tabs' );

		// Includes all the templates.
		$this->loader->add_action( 'mwb_crm_connect_render_tab_keap', $plugin_admin, 'render_tabs' );
		$this->loader->add_action( 'mwb_crm_connect_render_tabcontent_keap', $plugin_admin, 'render_tab_contents' );

		// Clear log callback.
		$this->loader->add_action( 'mwb_woo_keap_clear_log', $plugin_admin, 'clear_sync_log' );

		// Real time tag sync.
		$this->loader->add_action( 'mwb_woo_keap_real_time_tag_sync', $plugin_admin, 'mwb_woo_keap_real_time_tag_sync', 1, 5 );

		// Authorisation.
		$this->loader->add_action( 'admin_init', $plugin_admin, 'keap_check_for_authorization', 10 );
		
		// Get access token.
		$this->loader->add_action( 'admin_init', $plugin_admin, 'keap_get_access_token' );

		$this->loader->add_action( 'woocommerce_product_options_general_product_data', $plugin_admin, 'add_nonce_product_nonce' );
		// Deactivation form 
		$this->loader->add_action( 'admin_footer', $plugin_admin, 'init_deactivation_keap' );

		// Add cron schedule for 5 mins.
		$this->loader->add_filter( 'cron_schedules', $plugin_admin, 'add_background_sync_schedule' );

		$this->loader->add_action( 'wps_woo_keap_background_sync', $plugin_admin, 'perform_background_sync' );

		// Schedule background sync.
		$this->loader->add_filter( 'admin_init', $plugin_admin, 'schedule_background_sync_event' );

		// add filter to add a select custom field to the product data tab.
		$this->loader->add_filter( 'woocommerce_product_data_tabs', $plugin_admin, 'add_custom_product_data_tab', 10, 1 );

		// add action to display the custom field in the product data tab.
		$this->loader->add_action( 'woocommerce_product_data_panels', $plugin_admin, 'add_custom_product_data_fields' );

		$this->loader->add_action( 'woocommerce_process_product_meta', $plugin_admin, 'save_custom_product_data_fields', 10, 2 );

		if ( 'yes' == get_option( 'mwb_keap_enable_product_deletion_from_keap', 'no' ) ) {
			$this->loader->add_action( 'wp_trash_post', $plugin_admin, 'action_wp_trash_post_keap', 10, 1 );

			$this->loader->add_action( 'untrash_post', $plugin_admin, 'action_wp_untrash_post_keap', 10, 1 );

			$this->loader->add_action( 'before_delete_post', $plugin_admin, 'custom_function_on_product_delete' );

			$this->loader->add_action( 'woocommerce_before_delete_product_variation', $plugin_admin, 'custom_function_on_product_delete',10,1 );
		}
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Crm_Integration_For_Keap_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Check Client id store.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function is_keap_valid_client_ids_stored() {

		$is_valid_client_ids_store = get_option( 'mwb_keap_valid_client_ids_stored', false );

		return $is_valid_client_ids_store;
	}

	/**
	 * Register all of the hooks related to the ajax requests.
	 *
	 * @since  1.0.0
	 */
	private function define_ajax_hooks() {

		$ajax_module = new Crm_Integration_For_Keap_Ajax();

		// On new/update products.
		$this->loader->add_action(
			'wp_ajax_mwb_woo_keap_ajax',
			$ajax_module,
			'mwb_woo_keap_ajax_cb'
		);

		$this->loader->add_action(
			'wp_ajax_get_datatable_data_keap',
			$ajax_module,
			'get_datatable_data_cb_keap'
		);
	}

	/**
	 * Register all of the hooks related to the woocommerce objects add/updates.
	 *
	 * @since 1.0.0
	 * @throws  Exception Exception.
	 * @return null
	 */
	private function define_woo_hooks() {

		if ( ! Crm_Integration_For_Keap_Admin::is_instant_sync_enable() ) {
			return;
		}

		$request_module = new Crm_Integration_Request_Module_Keap();

		// Get all Feeds and then extend the hooks.
		$keap = Woo_Crm_Connect_Keap::get_instance();

		// Add all callbacks.
		$hookable_callbacks = $keap->getHookRequestsForFeed();

		if ( empty( $hookable_callbacks ) ) {
			return;
		}

		/**
		 * Remove all order status change sync feeds. Run them once.
		 * 'woocommerce_order_status_pending'
		 * 'woocommerce_order_status_failed'
		 * 'woocommerce_order_fully_refunded'
		 * 'woocommerce_order_partially_refunded'
		 * 'woocommerce_order_status_cancelled'
		 * 'woocommerce_order_status_completed'
		 * 'woocommerce_order_status_on-hold'
		 * 'woocommerce_order_status_processing'
		 */
		$feeds = $request_module->getFeedIdByRequest();
		$hook_delay = 100;

		if ( ! empty( $feeds ) && is_array( $feeds ) ) {

			$hooks = array(
				'woocommerce_order_status_pending',
				'woocommerce_order_status_failed',
				'woocommerce_order_fully_refunded',
				'woocommerce_order_partially_refunded',
				'woocommerce_order_status_cancelled',
				'woocommerce_order_status_completed',
				'woocommerce_order_status_on-hold',
				'woocommerce_order_status_processing',
			);

			foreach ( $hooks as $key => $hook ) {
				$this->loader->add_action(
					$hook,
					$request_module,
					'shopOrderStatusChanged',
					$hook_delay,
					$keap->getHookParamCount( $hook )
				);
			}

			foreach ( $feeds as $key => $feeds_id ) {
				unset( $hookable_callbacks[ $feeds_id ] );
			}
		}

		// Same events and callbacks entires will be handled in the feeds.
		$hookable_callbacks = array_unique( $hookable_callbacks, SORT_REGULAR );

		// Now add single event callbacks.
		if ( ! empty( $hookable_callbacks ) && is_array( $hookable_callbacks ) ) {
			foreach ( $hookable_callbacks as $feed_id => $hookable_callback ) {

				try {

					if ( is_array( $hookable_callback['hook'] ) ) {
						foreach ( $hookable_callback['hook'] as $key => $single_hook ) {
							$this->loader->add_action(
								$single_hook,
								$request_module,
								$hookable_callback['callback'],
								$hook_delay,
								$keap->getHookParamCount(
									$single_hook
								)
							);

							$hook_delay++;
						}
					} else {
						$this->loader->add_action(
							$hookable_callback['hook'],
							$request_module,
							$hookable_callback['callback'],
							$hook_delay,
							$keap->getHookParamCount(
								$hookable_callback['hook']
							)
						);
					}

					$hook_delay++;

				} catch ( \Throwable $th ) {
					throw new Exception( esc_html( $th->getMessage() ), 1 );
				}
			}
		}
	}

	 /**
	  * Initialize custom post type class.
	  *
	  * @since  1.0.0
	  */
	private function initialize_cpt() {

		$cpt_module = new Crm_Integration_For_Keap_Cpt();

		// Register custom posttypes.
		$this->loader->add_action( 'init', $cpt_module, 'register_plugin_cpt' );
		// Add meta box for feeds.
		$this->loader->add_action( 'add_meta_boxes', $cpt_module, 'add_feed_meta_boxes', 10, 2 );

		// Save meta box data.
		$this->loader->add_action( 'save_post', $cpt_module, 'save_metabox_data' );
	}
}
