<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://wpswings.com/
 * @since      1.0.0
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Integration_With_keap
 * @subpackage Integration_With_keap/admin
 */
use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;
class Crm_Integration_For_Keap_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string $plugin_name       The name of this plugin.
	 * @param      string $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Crm_Integration_For_Keap_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Crm_Integration_For_Keap_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		$screen = get_current_screen();

		if ( ! empty( $screen ) && in_array( $screen->id, $this->get_keap_plugin_admin_screens() ) ) {

			wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/crm-integration-for-keap-admin.css', array(), time(), 'all' );

		}

		wp_enqueue_style( $this->plugin_name . '-global-css', plugin_dir_url( __FILE__ ) . 'css/crm-integartion-for-keap-admin-global.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Crm_Integration_For_Keap_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Crm_Integration_For_Keap_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		$screen = get_current_screen();

		$shop_order_screen = wc_get_container()->get( CustomOrdersTableController::class )->custom_orders_table_usage_is_enabled()
		? wc_get_page_screen_id( 'shop-order' )
		: 'shop_order';

		if ( ! empty( $screen ) && ( $shop_order_screen === $screen->id ) ) {

			wp_enqueue_script( 'crm-connect-sweetalert', plugin_dir_url( __FILE__ ) . 'js/sweet-alert.js', array( 'jquery' ), $this->version, false );

			wp_enqueue_script( 'crm-connnect-meta-box', plugin_dir_url( __FILE__ ) . 'js/order-meta-box-script.js', array( 'jquery' ), time(), false );
			$ajax_data = array(
				'crm'         => 'keap',
				'ajax_url'    => admin_url( 'admin-ajax.php' ),
				'ajax_action' => 'mwb_woo_keap_ajax',
				'ajax_nonce'  => wp_create_nonce( 'ajax_nonce' ),
			);
			wp_localize_script( 'crm-connnect-meta-box', 'ajax_data', $ajax_data );
		}

		if ( ! empty( $screen ) && in_array( $screen->id, $this->get_keap_plugin_admin_screens() ) ) {

			wp_enqueue_script( 'crm-connect-sweetalert', plugin_dir_url( __FILE__ ) . 'js/sweet-alert.js', array( 'jquery' ), $this->version, false );

			wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/crm-integration-for-keap-admin.js', array( 'jquery' ), time(), false );

			$ajax_data = array(
				'crm'         => 'keap',
				'ajax_url'    => admin_url( 'admin-ajax.php' ),
				'ajax_action' => 'mwb_woo_keap_ajax',
				'ajax_nonce'  => wp_create_nonce( 'ajax_nonce' ),
				'feed_back_link' => admin_url( 'admin.php?page=integration-with-keap' ),
				'feed_back_text' => __( 'Back to feeds', 'integration-with-keap' ),
				'keap_api_name'  => __('API Name', 'integration-with-keap'),
				'keap_api_type'  => __('Type', 'integration-with-keap'),
				'keap_api_length' => __('Max Length', 'integration-with-keap'),
				'delete_icon'    => CRM_INTEGRATION_KEAP_URL . 'assets/src/images/trash.svg',
				'notice'         => $this->get_admin_notice(),
				'post_id'        => get_the_ID(),
				'keap_disconnect'     => array(
					'title'      => __( 'Attempt to disconnect Account! Are you sure?', 'integration-with-keap' ),
					'yes'        => __( 'Disconnect', 'integration-with-keap' ),
					'no'         => __( 'Cancel', 'integration-with-keap' ),
					'processing' => __( 'Disconnecting...', 'integration-with-keap' ),
				),
				'keap_prod_popup'           => array(
					'title'      => __( '
						In case you skip the "Bulk Sync" option, it is essential to sync the "Products" (since "Instant Order" and "Historical Order" are dependent on synced products).', 'integration-with-keap' ),
					'yes'        => __( 'Skip', 'integration-with-keap' ),
				),
				'keap_skip'           => array(
					'title'      => __( 'Your have more entries to be synced. We recommend you to skip for now. Please click on Initiate Sync to continue with the sync.', 'integration-with-keap' ),
					'yes'        => __( 'Initiate Sync', 'integration-with-keap' ),
					'no'         => __( 'Skip', 'integration-with-keap' ),
					'processing' => __( 'Skipping...', 'integration-with-keap' ),
				),
				'keap_connect' => array(
					'authorize'        => __( 'Authorize', 'integration-with-keap' ),
					'login'        => __( 'Login & Authorize', 'integration-with-keap' ),
				),
			);

			if ( 'mwb_keap_feed' == $screen->post_type ) {
				$ajax_data['feed_form_settings'] = $this->get_feed_form_settings();
			}

			$this->clear_admin_notice();
			wp_localize_script( $this->plugin_name, 'ajax_data', $ajax_data );
		}

		//Deactivation form script
		wp_enqueue_script( 'crm-keap-connect-hubspot-sdk', '//js-eu1.hsforms.net/forms/shell.js', array(), $this->version, false );
		
		wp_enqueue_script( $this->plugin_name . '-global-js', plugin_dir_url( __FILE__ ) . 'js/crm-integration-for-keap-admin-global.js', array( 'jquery' ), $this->version, true );
	}

	/**
	 * Ge plugin admin screens.
	 *
	 * @return array Array of admin screens.
	 */
	public function get_keap_plugin_admin_screens() {

		return array( 'wp-swings_page_integration-with-keap', 'mwb_keap_feed' );
	}

	/**
	 * Check if mwb boilerplate is active.
	 *
	 * @param  string $submenu Submenu array.
	 * @return bool
	 */
	public function is_boilerplate_active_keap( $submenu = '' ) {
		if ( ! empty( $submenu ) ) {
			return ( is_array( $submenu ) && array_key_exists( 'wps-plugins', $submenu ) );
		} else {
			return false;
		}
	}

	/**
	 * Check if wps boilerplate is active.
	 *
	 * @param  string $submenu Submenu array.
	 * @return bool
	 */
	public function is_any_crm_plugins_active( $submenu = '' ) {

		if ( ! empty( $submenu ) ) {
			return ( is_array( $submenu ) && array_key_exists( 'wps-crm-plugins', $submenu ) );
		} else {
			return false;
		}
	}
	/**
	 * Include admin menu
	 */
	public function add_admin_menu_page_keap() {

		global $submenu;

		if ( false == $this->is_boilerplate_active_keap( $submenu ) ) { //phpcs:ignore
			if ( true == $this->is_any_crm_plugins_active( $submenu ) ) {//phpcs:ignore

			   /**
				* Plugin menus filter hook.
				*
				* @since    1.0.0
				*/
				$wdc_menus = apply_filters( 'wps_add_crm_plugins_keap_menus_array', array() );

				if ( is_array( $wdc_menus ) && ! empty( $wdc_menus ) ) {
					foreach ( $wdc_menus as $wdc_key => $wdc_value ) {
						add_submenu_page( 'wps-crm-plugins', $wdc_value['name'], $wdc_value['name'], 'manage_options', $wdc_value['menu_link'], array( $wdc_value['instance'], $wdc_value['function'] ) );
					}
				}
			} else {
				add_menu_page(
					'WP Swings',
					'WP Swings',
					'manage_options',
					'wps-crm-plugins',
					'',
					CRM_INTEGRATION_KEAP_URL . '/assets/src/images/wpswings_logo.png',
					15
				);

			   /**
				* Plugin menus filter hook.
				*
				* @since    1.0.0
				*/
				$wdc_menus = apply_filters( 'wps_add_crm_plugins_keap_menus_array', array() );

				if ( is_array( $wdc_menus ) && ! empty( $wdc_menus ) ) {
					foreach ( $wdc_menus as $wdc_key => $wdc_value ) {
						add_submenu_page( 'wps-crm-plugins', $wdc_value['name'], $wdc_value['name'], 'manage_options', $wdc_value['menu_link'], array( $wdc_value['instance'], $wdc_value['function'] ) );
					}
				}
			}
		}
	}

	/**
	 * WooCommerce Delivery Checker crm_admin_submenu_page_keap.
	 *
	 * @since 1.0.0
	 * @param array $menus Marketplace menus.
	 */
	public function crm_admin_submenu_page_keap( $menus = array() ) {

		$menus[] = array(
			'name'      => __( 'CRM connect - Keap', 'integration-with-keap' ),
			'slug'      => 'integration-with-keap',
			'menu_link' => 'integration-with-keap',
			'instance'  => $this,
			'function'  => 'include_admin_menu_display',
		);

		return $menus;
	}

	/**
	 * Removing default submenu of parent menu in backend dashboard
	 *
	 * @since   1.0.0
	 */
	public function remove_default_submenu_keap() {

		global $submenu;

		if ( is_array( $submenu ) && array_key_exists( 'wps-crm-plugins', $submenu ) ) {
			if ( isset( $submenu['wps-crm-plugins'][0] ) ) {
				unset( $submenu['wps-crm-plugins'][0] );
			}
		}
	}

	/**
	 * Include_settings_display function
	*/
	public function include_admin_menu_display() {
	
		$file_path = '/partials/crm-integration-for-keap-admin-display.php';
		self::load_template( $file_path );
	}

	/**
	 * Initiate deactivation screen.
	*/
	public function init_deactivation_keap() {
		
		$file_path = '/partials/templates/deactivation-screen.php';
		self::load_template( $file_path);
	}

	/**
	 * Return mwb-crm-plugins menu at custom post type page.
	 *
	 * @param mixed $parent_file parent_file.
	 * @return mixed
	 */
	public function prefix_highlight_taxonomy_parent_menu( $parent_file ) {
		global $submenu_file, $current_screen, $pagenow;
		$mwb_bfw_taxonomy_array = 'mwb_keap_feed';

		if ( get_current_screen()->post_type == 'mwb_keap_feed' ) {
			$parent_file = 'wps-crm-plugins';
		}

		return $parent_file;
	}

	/**
	 * Return salesforce sub-menu at custom post type page.
	 *
	 * @param mixed $submenu_file submenu_file.
	 * @param mixed $parent_file parent_file.
	 * @return mixed
	 */
	public function crm_bfw_set_submenu_file_to_handle_menu_for_wp_pages( $submenu_file, $parent_file ) {
		$mwb_bfw_taxonomy_array = 'mwb_keap_feed';
		if ( get_current_screen()->post_type == $mwb_bfw_taxonomy_array ) {
			$submenu_file = 'integration-with-keap';
		}

		return $submenu_file;
	}

	/**
	 * Check and include admin view file
	 *
	 * @param string $file_path Relative path of file.
	 * @param array  $params Array of extra params.
	 * @param string $base String of extra params.
	 */
	public static function load_template( $file_path, $params = array(), $base = '' ) {

		try {

			$result = wc_get_template(
				$file_path,
				$params,
				'',
				'' != $base ? $base : plugin_dir_path( __FILE__ )
			);

		} catch ( \Throwable $th ) {
			echo esc_html( $th->getMessage() );
			wp_die();
		}
	}

	/**
	 * Initiate all functionality of crm.
	 */
	public function woo_crm_init() {

		// Add authorisation screen.
		add_action( 'mwb-woo-connect-authorisation-screen', array( $this, 'init_authorisation' ) );

		// Add dashbo screen.
		add_action( 'mwb-woo-connect-dashboard-screen', array( $this, 'init_dashboard' ) );
	}

	/**
	 * Initiate authorisation screen.
	 */
	public function init_authorisation() {

		$settings = array();
		$file_path = 'partials/templates/authorisation-screen.php';
		self::load_template( $file_path );
	}

	/**
	 * Initiate all functionality of crm.
	 */
	public function init_dashboard() {

		$settings = array();
		$file_path = 'partials/templates/dashboard-screen.php';
		self::load_template( $file_path );
	}

	/**
	 * Map all redirects on admin portal.
	 */
	public function admin_init_action() {

		// Get refesh token using authcode and api keys.
		if ( isset( $_GET['code'] ) && isset( $_GET['accounts-server'] ) && isset( $_GET['location'] ) ) {
			$code     = sanitize_text_field( wp_unslash( $_GET['code'] ) );
			$keap_api = Crm_Integration_Keap_Api::get_instance();
			$success  = $keap_api->get_refresh_token_data( $code );
			update_option( 'mwb-woo-keap-authorised', $success );
			if ( ! $success ) {
				$message = __( 'Check your credentials !!', 'integration-with-keap' );
				$this->add_admin_notice( $message, $success );
			}
			wp_safe_redirect( admin_url( 'admin.php?page=integration-with-keap' ) );

		}

		// Refresh access token on expiry.
		if ( isset( $_GET['mwb_refresh_keap_access_token'] ) && '1' == $_GET['mwb_refresh_keap_access_token'] ) {
			if ( isset( $_GET['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) ) ) {
				$keap_api = Crm_Integration_Keap_Api::get_instance();
				$success  = $keap_api->renew_access_token() ? '1' : '0';
				wp_safe_redirect( admin_url( 'admin.php?page=integration-with-keap&renew_token=' . $success ) );
			}
		}

		// Download log file.
		if ( isset( $_GET['mwb_download'] ) ) {
			$filename = WC_LOG_DIR . 'mwb-woo-keap-sync-log.log';
			header( 'Content-type: text/plain' );
			header( 'Content-Disposition: attachment; filename="' . basename( $filename ) . '"' );
			readfile( $filename );
			exit;
		}
	}

	/**
	 * Expiry for current Token.
	 */
	public static function get_keap_access_token_expiry() {

		$token_expire_in = get_option( 'mwb_keap_token_expiry' );
		if ( $token_expire_in > time() ) {

			return ( $token_expire_in - time() ) / 60;
		}
		return false;
	}


	/**
	 * Get Tabs labels.
	 */
	public function render_tabs() {
	   /**
		* Plugin tabs filter hook.
		*
		* @since    1.0.0
		*/
		$tabs = apply_filters( 'mwb_crm_woo_tabs', array() );
		$file_path = 'woo-crm-fw/templates/crm-woo-tabs.php';
		self::load_template( $file_path, array( 'tabs'  => $tabs ), CRM_INTEGRATION_KEAP_PATH );
	}

	public function mwb_woo_keap_real_time_tag_sync( $post_type, $feed_id, $post_id, $record_type, $keap_id ) { 

		$crm_connect_manager      = Woo_Crm_Connect_Keap::get_instance();
		$request = $crm_connect_manager->getRequest(
			$post_type, 
			$feed_id,   
			$post_id,   
			$record_type    
		);
		$endpoint = 'https://api.infusionsoft.com/crm/rest/contacts/' . $keap_id . '/tags'; 
		$tag_apply['tagIds'] = array( $request['tagIds'] ); 
		$request_data = json_encode($tag_apply, JSON_NUMERIC_CHECK);    
		$pattern = '/"/i';  
		$request_data = preg_replace($pattern, '', $request_data);  
		$pattern = '/tagIds/i'; 
		$request_data = preg_replace($pattern, '"tagIds"', $request_data);  
		$crm_integration_keap_api = Crm_Integration_Keap_Api::get_instance();   
		$result_tag               = $crm_integration_keap_api->apply_tag_to_contacts($request_data, $endpoint); 
	}

	/**
	 * Get Tabs Contents.
	 */
	public function render_tab_contents() {
	   /**
		* Plugin tabs filter hook.
		*
		* @since    1.0.0
		*/
		$tabs = apply_filters( 'mwb_crm_woo_tabs', array() );
		if ( ! empty( $tabs ) && is_array( $tabs ) ) {
			foreach ( $tabs as $file_name => $value ) {

				switch ( $file_name ) {
					case 'mwb-data-sync':
					$settings[]  = array(
						'title' => esc_html__( 'Bulk Data Sync', 'integration-with-keap' ),
						'description' => esc_html__( 'Sync all the WooCommerce Objects of your site to the CRM in One Click.', 'integration-with-keap' ),
						'button' => array(
							'title' => esc_html__( 'Bulk Objects', 'integration-with-keap' ),
							'href' => '',
							'newtab' => false,
							'class' => 'bulk-data-sync',
						),
					);
					$settings[]  = array(
						'title' => esc_html__( 'One Click Sync', 'integration-with-keap' ),
						'description' => esc_html__( 'Sync all the New WooCommerce Objects of your site to the CRM in One Click.', 'integration-with-keap' ),
						'button' => array(
							'title' => esc_html__( 'One Click Sync', 'integration-with-keap' ),
							'href' => '',
							'newtab' => false,
							'class' => 'one-click-sync',
						),
					);

					$params = array( 'settings' => $settings );

						break;

					case 'mwb-dashboard':
					$connection_status = self::get_connection_status();
					;
					$token_expiry      = self::get_keap_access_token_expiry();
					$params            = compact( 'connection_status', 'token_expiry' );
						break;

					case 'mwb-logs':
					$params = array();
						break;

					case 'mwb-tags':
					$params = array();
						break;

					case 'mwb-feeds':
					$args         = array( 'publish', 'draft' );
					$cpt_instance = new Crm_Integration_For_Keap_Cpt();
					$feeds        = $cpt_instance->get_available_feeds( $args );
					$feed_options = $cpt_instance->get_feed_event_options();
					$lead_options = $cpt_instance->get_lead_source_options();
					$params       = compact( 'feeds', 'cpt_instance', 'feed_options', 'lead_options' );
						break;

					default:
					$params = array();
						break;
				}

				$file_path = 'woo-crm-fw/templates/tabs-contents/' . $file_name . '.php';
				self::load_template( $file_path, $params, CRM_INTEGRATION_KEAP_PATH );
			}
		}
	}

	/**
	 * Get connection status.
	 */
	public static function get_connection_status() {
		return get_option( 'mwb-woo-keap-authorised', false );
	}


	/**
	 * Specify Tabs Lables/Contents.
	 *
	 * @param array $tabs Array of data.
	 */
	public function return_tabs( $tabs = array() ) {

		return array(
			'mwb-dashboard' => __( 'Dashboard', 'integration-with-keap' ),
			'mwb-feeds'     => __( 'Feeds', 'integration-with-keap' ),
			'mwb-data-sync' => __( 'Data Sync', 'integration-with-keap' ),
			'mwb-logs'      => __( 'Logs', 'integration-with-keap' ),
			'mwb-tags'      => __( 'Tags', 'integration-with-keap' ),
			'mwb-settings'  => __( 'Settings', 'integration-with-keap' ),
		);
	}

	/**
	 * Get authorization settings.
	 *
	 * @return array $settings Settings array.
	 */
	public static function get_keap_authorization_settings() {

		$settings[] = array(
			'title' => esc_html__( 'Enter your API credentials here', 'integration-with-keap' ),
			'class'    => 'mwb-wfw-sub-heading',
			'type'  => 'title',
		);
		$settings[] = array(
			'title'             => esc_html__( 'Client ID', 'integration-with-keap' ),
			'id'                => 'mwb_keap_client_id',
			'type'              => 'text',
			'placeholder'       => 'Enter your Client ID here',
			'value'             => get_option( 'mwb_keap_client_id', false ),
			'custom_attributes' => array( 'required' => 'required' ),
		);
		$settings[] = array(
			'title'             => esc_html__( 'Secret ID', 'integration-with-keap' ),
			'id'                => 'mwb_keap_secret_id',
			'type'              => 'text',
			'placeholder'       => 'Enter your Secret ID here',
			'value'             => get_option( 'mwb_keap_secret_id', false ),
			'custom_attributes' => array( 'required' => 'required' ),
		);

		$settings[] = array(
			'type' => 'sectionend',
		);
		return $settings;
	}

	/**
	 * Get available keap objects
	 */
	public static function get_available_crm_objects() {
		$keap_api = Crm_Integration_Keap_Api::get_instance();
		$module_data = $keap_api->get_modules_data();
		return !empty( $module_data ) ? $module_data : array();
	}

	/**
	 * Get feed field types.
	 */
	public function get_feed_form_settings() {

		$class   = Woo_Crm_Connect_Keap::get_instance();
		$options = $class->getMappingDataset();
		$cpt_instance = new Crm_Integration_For_Keap_Cpt();
		$available_feeds = $cpt_instance->get_available_feeds_for_mapping();
		if ( count( $available_feeds ) ) {
			$options['feeds'] = $available_feeds;
		}
		$settings['field_type'] = array(
			'label' => __( 'Field Type', 'integration-with-keap' ),
			'options' => array(
				'standard_field' => __( 'Standard Fields', 'integration-with-keap' ),
				'custom_value'   => __( 'Custom Value', 'integration-with-keap' ),
			),
		);
		$settings['field_value'] = array(
			'label' => __( 'Field Value', 'integration-with-keap' ),
			'options' => $options,
		);

		$settings['custom_value'] = array(
			'label' => __( 'Custom Value', 'integration-with-keap' ),
		);

		return $settings;
	}

	/**
	 * Get general settings.
	 *
	 * @return array Array of settings.
	 */
	public static function get_general_settings() {

		$settings[] = array(
			'type' => 'title',
		);

		$settings[] = array(
			'title'             => esc_html__( 'Enable instant sync', 'integration-with-keap' ),
			'id'                => 'mwb-keap-enable-sync',
			'type'              => 'checkbox',
			'class'             => 'mwb-crm-feed-switch',
			'value'             => get_option( 'mwb_keap_enable_sync', 'yes' ),
		);

		$settings[] = array(
			'title'             => esc_html__( 'Enable logging', 'integration-with-keap' ),
			'id'                => 'mwb-keap-enable-log',
			'type'              => 'checkbox',
			'class'             => 'mwb-crm-feed-switch',
			'value'             => get_option( 'mwb_keap_enable_log', 'yes' ),
		);

		$settings[] = array(
			'title'             => esc_html__( 'Delete log after x days', 'integration-with-keap' ),
			'id'                => 'mwb-keap-delete-log',
			'type'              => 'number',
			'value'             => get_option( 'mwb_keap_delete_log', '30' ),
			'custom_attributes' => array( 'min' => 7 ),
		);


		$settings[] = array(
			'title' => esc_html__( 'Wipe data on disconnect', 'integration-with-keap' ),
			'id'    => 'mwb-keap-wipe-data',
			'type'  => 'checkbox',
			'class' => 'mwb-crm-feed-switch',
			'value' => get_option( 'mwb_keap_wipe_data', 'no' ),
			'desc_tip'  => __( 'Enable to delete the sync history on disconnecting the account.', 'integration-with-keap' ),
		);

		// 2.1.0 start.
		$settings[] = array(
			'title' => esc_html__( 'Enable error reporting', 'integration-with-keap' ),
			'id'    => 'mwb-keap-error-reporting',
			'type'  => 'checkbox',
			'class' => 'mwb-crm-feed-switch',
			'value' => get_option( 'mwb_keap_error_reporting', 'no' ),
			'desc_tip'  => __( 'Enable to send email when error occured.', 'integration-with-keap' ),
		);

		$settings[] = array(
			'title'             => esc_html__( 'Enter email', 'integration-with-keap' ),
			'id'                => 'mwb-keap-error-email',
			'type'              => 'text',
			'value'             => get_option( 'mwb_keap_error_email', get_bloginfo( 'admin_email' ) ),
		);
		// 2.1.0 end.

		$settings[] = array(
			'title' => esc_html__( 'Product deletion from Woo to keap', 'integration-with-keap' ),
			'id'    => 'mwb-keap-enable-product-deletion-from-keap',
			'type'  => 'checkbox',
			'class' => 'mwb-crm-feed-switch',
			'value' => get_option( 'mwb_keap_enable_product_deletion_from_keap', 'no' ),
			'desc_tip'  => __( 'Enable this option to inactive, active or delete products on Keap when any product go to trash, restore or permanently deleted in WooCommerce.', 'integration-with-keap' ),
		);

		$settings[] = array(
			'title' => esc_html__( 'Product deletion from keap to Woo', 'integration-with-keap' ),
			'id'    => 'mwb-keap-enable-product-deletion-from-woo',
			'type'  => 'checkbox',
			'class' => 'mwb-crm-feed-switch',
			'value' => get_option( 'mwb_keap_enable_product_deletion_from_woo', 'no' ),
			'desc_tip'  => __( 'Enable this option to trash, restore or permanently delete products on WooCommerce when any product go to inactive, active or deleted in Keap.', 'integration-with-keap' ),
		);

		/**
		* Add extra setting filter hook.
		*
		* @since    1.0.0
		*/
		$settings = apply_filters( 'wps_woo_add_extra_setting', $settings );

		$settings[] = array(
			'type' => 'sectionend',
		);
		return $settings;
	}

	/**
	 * Clear sync log callback.
	 */
	public function clear_sync_log() {

		$last_delete     = get_option( 'mwb_woo_keap_log_last_delete', time() );
		$delete_duration = get_option( 'mwb_keap_delete_log', 30 );
		if ( time() > ( ( $delete_duration * 24 * 60 * 60 ) + $last_delete ) ) {
			self::delete_sync_log();
			update_option( 'mwb_woo_keap_log_last_delete', time() );
		}
	}

	/** 
	* Get general settings. 
	*   
	* @return array Array of settings.  
	*/  
	public static function get_general_tab_settings() {
		$settings[] = array(
		'type' => 'title',  
		);  
		$settings[] = array(    
		'title'             => esc_html__( 'Enable Tag sync', 'integration-with-keap' ),    
		'id'                => 'mwb-keap-enable-tag-sync',  
		'type'              => 'checkbox',  
		'class'             => 'mwb-crm-feed-switch',   
		'value'             => get_option( 'mwb_keap_enable_tag_sync', 'no' ),  
		);  
		$settings[] = array(    
		'title'             => esc_html__( 'Enable Tag on Instant sync', 'integration-with-keap' ), 
		'id'                => 'mwb-keap-enable-tag-instant-sync',  
		'type'              => 'checkbox',  
		'class'             => 'mwb-crm-feed-switch',   
		'value'             => get_option( 'mwb_keap_enable_tag_instant_sync', 'no' ),  
		);  
		$settings[] = array(    
		'title'             => esc_html__( 'Enable Products Bought Tags', 'integration-with-keap' ),    
		'id'                => 'mwb-keap-enable-products-tag',  
		'type'              => 'checkbox',  
		'class'             => 'mwb-crm-feed-switch',   
		'value'             => get_option( 'mwb_keap_enable_products_tag', 'no' ),
		);  
		$settings[] = array(    
		'title'             => esc_html__( 'Enable Products SKUs Tags', 'integration-with-keap' ),
		'id'                => 'mwb-keap-enable-skus-tag',  
		'type'              => 'checkbox',  
		'class'             => 'mwb-crm-feed-switch',   
		'value'             => get_option( 'mwb_keap_enable_skus_tag', 'no' ),
		);  
		$settings[] = array(    
		'title'             => esc_html__( 'Enable Products Type Tags', 'integration-with-keap' ),
		'id'                => 'mwb-keap-enable-products-type-tag', 
		'type'              => 'checkbox',  
		'class'             => 'mwb-crm-feed-switch',   
		'value'             => get_option( 'mwb_keap_enable_products_type_tag', 'no' ),
		);  
		$settings[] = array(    
		'title'             => esc_html__( 'Enable Order Statuses Tags', 'integration-with-keap' ),
		'id'                => 'mwb-keap-enable-order-statuses-tag',    
		'type'              => 'checkbox',
		'class'             => 'mwb-crm-feed-switch',   
		'value'             => get_option( 'mwb_keap_enable_order_statuses_tag', 'no' ),
		);
		$settings[] = array(    
		'type' => 'sectionend', 
		);  
		return $settings;
	}

	/**
	 * Clear log table.
	 */
	public static function delete_sync_log() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'mwb_woo_keap_log';
		$query      = "TRUNCATE TABLE {$table_name}";
		mwb_woo_keap_execute_db_query( $query );
	}


	/**
	 * Clear admin notices.
	 */
	public function clear_admin_notice() {
		delete_option( 'mwb-woo-keap-admin-notice' );
	}

	/**
	 * Get admin notices to show.
	 *
	 * @return array notice array.
	 */
	public function get_admin_notice() {
		return get_option( 'mwb-woo-keap-admin-notice', array() );
	}

	/**
	 * Add notice to options.
	 *
	 * @param string $message Notice error.
	 * @param bool   $success success.
	 */
	public function add_admin_notice( $message, $success ) {
		update_option( 'mwb-woo-keap-admin-notice', compact( 'message', 'success' ) );
	}

	/**
	 * Check if instant sync is enable.
	 *
	 * @return boolean
	 */
	public static function is_instant_sync_enable() {
		$enable = get_option( 'mwb_keap_enable_sync', 'yes' );
		$enable = ( 'yes' === $enable );
		return $enable;
	}

	/**
	 * Check if instant sync is enable.
	 *
	 * @return boolean
	 */
	public static function is_products_popup() {
		$enable = get_option( 'mwb-woo-keap-product-popup-notification', 'yes' );
		$enable = ( 'yes' === $enable );
		return $enable;
	}

	/**
	 * Check if instant sync is enable.
	 *
	 * @return bool
	 */
	public static function is_wipe_allowed() {
		$enable = get_option( 'mwb_keap_wipe_data', 'no' );
		$enable = ( 'yes' === $enable );
		
		return $enable;
	}
	/** 
	* Check if log is enable.   
	*   
	* @return boolean   
	*/  
	public static function is_log_enable() {    
		$enable = get_option( 'mwb_keap_enable_log', 'yes' );   
		$enable = ( 'yes' === $enable );    
		return $enable; 
	}   
	/** 
	* Check if Products tags is enable. 
	*   
	* @return boolean   
	*/  
	public static function is_tag_enable() {    
		$enable = get_option( 'mwb_keap_enable_tag_sync', 'no' );   
		$enable = ( 'yes' === $enable );    
		return $enable; 
	}   
	/** 
	* Check if Products tags is enable. 
	*   
	* @return boolean   
	*/  
	public static function is_instant_tag_enable() {    
		$enable = get_option( 'mwb_keap_enable_tag_instant_sync', 'no' );   
		$enable = ( 'yes' === $enable );    
		return $enable; 
	}   
	/** 
	* Check if Products tags is enable. 
	*   
	* @return boolean   
	*/  
	public static function is_products_bought_tag_enable() {    
		$enable = get_option( 'mwb_keap_enable_products_tag', 'no' );   
		$enable = ( 'yes' === $enable );    
		return $enable; 
	}   
	/** 
	* Check if SKUs tags is enable. 
	*   
	* @return boolean   
	*/  
	public static function is_skus_tag_enable() {   
		$enable = get_option( 'mwb_keap_enable_skus_tag', 'no' );   
		$enable = ( 'yes' === $enable );    
		return $enable; 
	}   
	/** 
	* Check if Products Type tags is enable.    
	*   
	* @return boolean   
	*/  
	public static function is_products_type_tag_enable() {  
		$enable = get_option( 'mwb_keap_enable_products_type_tag', 'no' );  
		$enable = ( 'yes' === $enable );    
		return $enable; 
	}
	/** 
	* Check if Products Type tags is enable.    
	*   
	* @return boolean   
	*/  
	public static function is_order_status_tag_enable() {   
		$enable = get_option( 'mwb_keap_enable_order_statuses_tag', 'no' ); 
		$enable = ( 'yes' === $enable );    
		return $enable; 
	}


	/**
	 * Redirecting to oauth for keap.
	 *
	 * @since    1.0.0
	 */
	public function keap_check_for_authorization() {

		if ( ( ! empty( $_GET['mwb_tabs_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['mwb_tabs_nonce'] ) ), 'admin_save_data' ) )) {

			if ( isset( $_GET['action'] ) && 'keap_authorize' == $_GET['action'] && isset( $_GET['page'] ) && 'integration-with-keap' == $_GET['page'] ) {

				$keap_api = Crm_Integration_Keap_Api::get_instance();
	
				$keap_url = $keap_api->get_auth_code_url();
	
				add_filter('allowed_redirect_hosts', function ( $hosts ) {
					$keap_api = Crm_Integration_Keap_Api::get_instance();
					$keap_url = $keap_api->get_auth_code_url();
					$host = str_replace('https://', '', $keap_url);
					$host = explode('/', $host);
					$hosts[] = $host[0];
					return $hosts;
				});
				wp_safe_redirect( $keap_url );
				exit();
			}
		}
	}

	/**
	 * Get access token from keap CRM
	 *
	 * @since    1.0.0
	 */
	public function keap_get_access_token() {

		if ( isset( $_GET['code'] ) ) {
			$client_id     = get_option( 'mwb_keap_client_id', false );
			$client_secret = get_option( 'mwb_keap_secret_id', false );

			$keap_api = Crm_Integration_Keap_Api::get_instance();
			$keap_inatance = new Crm_Integration_For_Keap();

			if ( $client_id && $client_secret ) {

				if ( ! $keap_inatance->is_keap_valid_client_ids_stored() ) {

					$response = $keap_api->keap_fetch_access_token_from_code( $client_id, $client_secret );

				} else {

					$response = $keap_api->keap_fetch_access_token_from_code( $client_id, $client_secret );
				}

				wp_safe_redirect( admin_url( 'admin.php' ) . '?page=integration-with-keap' );

			}
		}
	}

	/**
	 * Get is_success_res  from keap CRM
	 *
	 * @param array $response Response.
	 * @since    1.0.0
	 */
	public static function is_success_res( $response ) {

		if ( ! empty( $response['code'] ) && ( 200 == $response['code'] ) ) {
			$is_success = 'true';

		} elseif ( ! empty( $response['message'] ) && ( 'OK' == $response['message'] ) ) {

			$is_success = 'true';

		} elseif ( ! empty( $response['message'] ) && ( 'Created' == $response['message'] ) ) {
			$is_success = 'true';
		} else {

			$is_success = 'false';
		}

		return $is_success;
	}

	/**
	 * Get set_woo_response_meta_for_count  from keap CRM
	 *
	 * @param string $feed_id feed_id.
	 * @param string $woo_id woo_id.
	 * @param string $keap_id keap_id.
	 * @param string $is_success is_success.
	 * @since    1.0.0
	 */
	public static function set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success ) {

		if ( $is_success ) {

			wps_keap_update_meta_data( $woo_id, '_keap_associated_object_status', true );
			wps_keap_update_meta_data(
				$woo_id,
				'mwb_keap_feed_' . $feed_id . '_association',
				$keap_id
			);

		} else {

			wps_keap_update_meta_data( $woo_id, '_keap_associated_object_status', false );

		}
	}
	
	/**
	 * Add product field nonce.
	 */
	public function add_nonce_product_nonce() {
		?>
		<input type="hidden" name="meta_box_nonce" value="<?php echo esc_attr( wp_create_nonce( 'meta_box_nonce' ) ); ?>">
		<?php
	}

	/**
	* Add support screen.   
	*   
	* @since 1.0.1  
	*/  
	public static function add_skype_connect() { 
		?>
		<section class="mwb-keap-features">	
		<div class="mwb-keap-features-container">
		<div class="mwb-keap-features-row">	
		<div class="mwb-keap__header">	
		<h2 class="mwb-keap-title"><?php esc_html_e( 'Need Instant Assistance ?', 'integration-with-keap' ); ?></h2>	
		<p><?php esc_html_e( 'Regarding any issue, query or feature request for Keap Integration For Woocommerce', 'integration-with-keap' ); ?></p>	
		</div>	
		<div class="mwb-keap__contact-wrapper">	
		<div class="mwb-keap__contact">	
		<a target="_blank" href="https://join.skype.com/invite/xCmwbfxx8MCX"><img src="<?php echo esc_url( CRM_INTEGRATION_KEAP_URL ); ?>assets/src/images/skype.svg"></a>	
		</div>	
		</div>	
		</div>	
		</div>	
		</section>
		<?php	
	}

	/**
	 * Add background sync schedule.
	 *
	 * @param array $schedules Array of available schedules.
	 * @return array $schedules
	 */
	public function add_background_sync_schedule( $schedules ) {
		if ( ! isset( $schedules['wps_woo_bg_sync'] ) ) {

			$schedules['wps_woo_bg_sync'] = array(
				'interval' => 5 * 60,
				'display'  => __( 'Once every 5 minutes', 'crm-integration-for-zoho' ),
			);
		}

		return $schedules;
	}
	/**
	 * Schedule background sync event.
	 */
	public function schedule_background_sync_event() {

		// Scheduled event for background syncing.
		if ( ! wp_next_scheduled( 'wps_woo_keap_background_sync' ) ) {
			wp_schedule_event( time(), 'wps_woo_bg_sync', 'wps_woo_keap_background_sync' );
		}
	}

	/**
	 * Perform background sync.
	 */
	public function perform_background_sync() {

		if ( 'yes' == get_option( 'mwb_keap_enable_product_deletion_from_woo', 'no' ) ) {
			$connect_framework = new Crm_Integration_Connect_Framework_Keap();

			$feed_ids = $connect_framework->getObjectFeed( 'Products' );
			$feed_id  = is_array( $feed_ids ) ? reset( $feed_ids ) : $feed_ids;

			if ( empty( $feed_id ) ) {
				return;
			}
			if ( ! empty( $feed_id ) && is_numeric( $feed_id ) ) {
				$product_args = array(
					'post_type'      => array( 'product', 'product_variation' ),
					'posts_per_page' => -1,
					'post_status'    => 'all',
					'meta_query'     => array(  //phpcs:ignore
						array(
							'key'     => 'mwb_keap_feed_' . $feed_id . '_association',
							'compare' => 'EXISTS',
						),
					),
				);

				$product_query = new WP_Query( $product_args );
				while ( $product_query->have_posts() ) :
					$product_query->the_post();
					$prd_id[] = get_the_ID();
					$zoho_prod_id = get_post_meta( get_the_ID(), 'mwb_keap_feed_' . $feed_id . '_association', true );
					$this->delete_product_from_wordpress( $zoho_prod_id, get_the_ID() );
					$product = wc_get_product(get_the_ID());
					if ($product->is_type('variable')) {
						$variations = $product->get_children(); // Get IDs of variations
						foreach ($variations as $variation_id) {
							$prd_id[] = $variation_id;
							$qb_prod_id = get_post_meta( $variation_id, 'mwb_keap_feed_' . $feed_id . '_association', true );
							if ( !empty( $qb_prod_id ) ) {
								$this->delete_product_from_wordpress( $qb_prod_id, $variation_id );
							}
						}
					}
				endwhile;
			}
		}
	}

	public function delete_product_from_wordpress( $zoho_prod_id, $woo_prod_id ) {

		$api_class = new Crm_Integration_Keap_Api();
		$response = $api_class->get_record( $zoho_prod_id, 'products', array() );

		if ( isset( $response['code'] ) && 'Unable to find this Product' == $response['data']['message'] && '404' == $response['code'] ) {
			wp_delete_post( $woo_prod_id );
		} else if ( isset( $response['code'] ) && '200' == $response['code'] ) {
			if ( !empty( $response['data'] ) ) {
				$data = $response['data'];
				if ( true == $data['status'] ) {
					if ( 'trash' == wc_get_product($woo_prod_id)->get_data()['status'] ) {
						wp_publish_post( $woo_prod_id );
					}
				} else if ( 'publish' == wc_get_product($woo_prod_id)->get_data()['status'] ) {
					wp_trash_post( $woo_prod_id );
				}
			}
		}
	}

	public function custom_function_on_product_delete( $post_id ) {
		// Check if the deleted post is a product
		if ( 'product' === get_post_type($post_id) || 'product_variation' === get_post_type($post_id)) {
			// Perform your actions here
			
			$post = get_post($post_id);
			$connect_framework = new Crm_Integration_Connect_Framework_Keap();
			$request_module = new Crm_Integration_Request_Module_Keap();

			$feed_ids = $connect_framework->getObjectFeed( 'Products' );
			$feed_id  = is_array( $feed_ids ) ? reset( $feed_ids ) : $feed_ids;

			if ( empty( $feed_id ) ) {
				return;
			}
			if ( ! empty( $feed_id ) && is_numeric( $feed_id ) ) {
				$record_id = get_post_meta( $post_id, 'mwb_keap_feed_' . $feed_id . '_association', true );
				$api_class = new Crm_Integration_Keap_Api();
				$response = $api_class->delete_record( $record_id, 'products', array() );
			}

		}
	}

	/**
	 * Do inactive products on keap.
	 *
	 * @param string $post_id post_id.
	 * 
	 */
	public function action_wp_trash_post_keap( $post_id ) {
		if (get_post_type($post_id) === 'product') {
			$post = get_post($post_id);
			$connect_framework = new Crm_Integration_Connect_Framework_Keap();
			$request_module = new Crm_Integration_Request_Module_Keap();

			$feed_ids = $connect_framework->getObjectFeed( 'Products' );
			$feed_id  = is_array( $feed_ids ) ? reset( $feed_ids ) : $feed_ids;

			if ( empty( $feed_id ) ) {
				return;
			}
			if ( ! empty( $feed_id ) && is_numeric( $feed_id ) ) {

				$keap = Woo_Crm_Connect_Keap::get_instance();
				$feed_title = $keap->getFeedTitle( $feed_id );
				$record_type = $keap->getFeed( $feed_id, 'crm_object' );
				// Get product.
				$product = wc_get_product( $post_id );
				$bulk_request = array();
	
				$bulk_ids = array();
				if ( 'variable' == $product->get_type() ) {
					$available_variations = $product->get_available_variations();
					if ( ! empty( $available_variations ) && is_array( $available_variations ) ) {
	
						$bulk_ids = array();
						foreach ( $available_variations as $key => $variation ) {
							$request = $connect_framework->getRequest(
								$post->post_type,
								$feed_id,
								$variation['variation_id']
							);
	
							$bulk_ids[] = $variation['variation_id'];
	
						   /**
							* Keap product request filter hook.
							*
							* @since    1.0.0
							*/
							$request = apply_filters(
								'woo_crm_woo_keap_product_request',
								$request,
								$post_id
							);
	
							$request['product_id'] = $variation['variation_id'];
							array_push( $bulk_request, $request );
						}
	
					}
				} else {
					$request = $connect_framework->getRequest(
						$post->post_type,
						$feed_id,
						$post_id
					);
					$bulk_ids[] = $post_id;
	
				   /**
					* Keap product request filter hook.
					*
					* @since    1.0.0
					*/
					$request = apply_filters(
						'woo_crm_woo_keap_product_request',
						$request,
						$post_id
					);
					$request['product_id'] = $post_id;
				}
				$request['active'] = 'false';
				$request['subscription_only'] = 'false';
				$request = ! empty( $bulk_request ) ?
				$bulk_request :
				$request;
				
				$record_type = $connect_framework->getFeed(
					$feed_id,
					'crm_object'
				);
	
				$feed_title = $keap->getFeedTitle( $feed_id );
	
				$log_data = array(
					'woo_id'   => $post_id,
					'bulk_ids'   => $bulk_ids,
					'feed_id'  => $feed_id,
					'woo_object' => $post->post_type,
					'feed_title' => $feed_title,
				);
	
				$single_record = array();
				$filter_exist = $request_module->maybe_check_filter( $feed_id );
				if ( ( isset( $filter_exist[2] ) && ! empty( $filter_exist[2][1]['field'] ) && '-1' != $filter_exist[2][1]['field'] ) || ( isset( $filter_exist[1] ) && ! empty( $filter_exist[1][1]['field'] ) && '-1' != $filter_exist[1][1]['field'] ) || ( isset( $filter_exist[3] ) && ! empty( $filter_exist[3][1]['field'] ) && '-1' != $filter_exist[3][1]['field'] ) || ( isset( $filter_exist[4] ) && ! empty( $filter_exist[4][1]['field'] ) && '-1' != $filter_exist[4][1]['field'] ) ) {
					$filter_result = $request_module->wps_validate_filter( $filter_exist, $request );
					if ( true === $filter_result ) {
						$crm_integration_keap_api = Crm_Integration_Keap_Api::get_instance();
						if ( is_array( $request) && array_key_exists(0, $request) ) {
			
							foreach ($request as $key => $single_request) {
								
								$woo_id = $single_request['product_id'];
			
								$update_prod_id = get_post_meta( $woo_id, '_mwb_keap_product_id', true );
			
								foreach ( $single_request as $k => $v ) {
			
									if ( 'product_id' !== $k ) {
			
										$single_record[ $k ] = $v;
									}
								}
								if ( ! empty( $update_prod_id ) ) {
			
									$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
									$request_data = json_encode( $single_record );
			
									$result = $crm_integration_keap_api->update_single_products(
										$record_type,
										$request_data,
										$endpoint,
										$woo_id,
										! empty( $bulk_request ) ? true : false,
										$log_data
									);
									if (! empty( $result['data']['id'])) {
			
										$keap_id = $result['data']['id'];
										
										$is_success = self::is_success_res( $result );
				
										self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
									}
			
									
								} else {
									$request_data = json_encode( $single_record );
			
									$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
									$result = $crm_integration_keap_api->create_single_record(
										$record_type,
										$request_data,
										! empty( $bulk_request ) ? true : false,
										$log_data,
										$endpoint,
										$woo_id
									);
			
									if ( 'Products' == $record_type ) {
										if (! empty( $result['data']['id'])) {
			
											$keap_id = $result['data']['id'];

											update_post_meta( $single_request['product_id'], '_mwb_keap_product_id', $keap_id );
				
											$is_success = self::is_success_res( $result );
				
											self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
										}
			
									}
								}
							}
						} else {
			
							$woo_id = $request['product_id'];
			
							$update_prod_id = get_post_meta( $request['product_id'], '_mwb_keap_product_id', true );
			
							foreach ( $request as $k => $v ) {
			
								if ( 'product_id' !== $k ) {
			
									$single_record[ $k ] = $v;
								}
							}
							
							if ( ! empty( $update_prod_id ) ) {
			
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
			
								$request_data = json_encode( $single_record );
			
								$result = $crm_integration_keap_api->update_single_products(
									$record_type,
									$request_data,
									$endpoint,
									$woo_id,
									! empty( $bulk_request ) ? true : false,
									$log_data
								);
								if (! empty( $result['data']['id'])) {
			
									$keap_id = $result['data']['id'];
								}
								$is_success = self::is_success_res( $result );
			
								self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
			
							} else {
			
								$request_data = json_encode( $single_record );
			
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
								$result = $crm_integration_keap_api->create_single_record(
									$record_type,
									$request_data,
									! empty( $bulk_request ) ? true : false,
									$log_data,
									$endpoint,
									$woo_id
								);
			
								if ( 'Products' == $record_type ) {
									if (! empty( $result['data']['id'])) {
			
										$keap_id = $result['data']['id'];
									}
									update_post_meta( $request['product_id'], '_mwb_keap_product_id', $keap_id );
			
									$is_success = self::is_success_res( $result );
			
									self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
			
								}
							}
						}
					}
				} else {
					$crm_integration_keap_api = Crm_Integration_Keap_Api::get_instance();
					if ( is_array( $request) && array_key_exists(0, $request) ) {
	
						foreach ($request as $key => $single_request) {
							
							$woo_id = $single_request['product_id'];
	
							$update_prod_id = get_post_meta( $woo_id, '_mwb_keap_product_id', true );
	
							foreach ( $single_request as $k => $v ) {
	
								if ( 'product_id' !== $k ) {
	
									$single_record[ $k ] = $v;
								}
							}
							if ( ! empty( $update_prod_id ) ) {
	
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
								$request_data = json_encode( $single_record );
	
								$result = $crm_integration_keap_api->update_single_products(
									$record_type,
									$request_data,
									$endpoint,
									$woo_id,
									! empty( $bulk_request ) ? true : false,
									$log_data
								);
								if (! empty( $result['data']['id'])) {
	
									$keap_id = $result['data']['id'];
								}
								$is_success = self::is_success_res( $result );
	
								self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
								
							} else {
								$request_data = json_encode( $single_record );
	
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
								$result = $crm_integration_keap_api->create_single_record(
									$record_type,
									$request_data,
									! empty( $bulk_request ) ? true : false,
									$log_data,
									$endpoint,
									$woo_id
								);
	
								if ( 'Products' == $record_type ) {
									if (! empty( $result['data']['id'])) {
	
										$keap_id = $result['data']['id'];
									}
									update_post_meta( $single_request['product_id'], '_mwb_keap_product_id', $keap_id );
	
									$is_success = self::is_success_res( $result );
	
									self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
								}
							}
						}
					} else {
	
						$woo_id = $request['product_id'];
	
						$update_prod_id = get_post_meta( $request['product_id'], '_mwb_keap_product_id', true );
	
						foreach ( $request as $k => $v ) {
	
							if ( 'product_id' !== $k ) {
	
								$single_record[ $k ] = $v;
							}
						}
						
						if ( ! empty( $update_prod_id ) ) {
	
							$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
	
							$request_data = json_encode( $single_record );
	
							$result = $crm_integration_keap_api->update_single_products(
								$record_type,
								$request_data,
								$endpoint,
								$woo_id,
								! empty( $bulk_request ) ? true : false,
								$log_data
							);
							if (! empty( $result['data']['id'])) {
	
								$keap_id = $result['data']['id'];
							}
							$is_success = self::is_success_res( $result );
	
							self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
						} else {
	
							$request_data = json_encode( $single_record );
	
							$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
							$result = $crm_integration_keap_api->create_single_record(
								$record_type,
								$request_data,
								! empty( $bulk_request ) ? true : false,
								$log_data,
								$endpoint,
								$woo_id
							);
	
							if ( 'Products' == $record_type ) {
								if (! empty( $result['data']['id'])) {
	
									$keap_id = $result['data']['id'];
								}
								update_post_meta( $request['product_id'], '_mwb_keap_product_id', $keap_id );
	
								$is_success = self::is_success_res( $result );
	
								self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Do active products on keap.
	 *
	 * @param string $post_id post_id.
	 * 
	 */
	public function action_wp_untrash_post_keap( $post_id ) {
		if (get_post_type($post_id) === 'product') {
			$post = get_post($post_id);
			$connect_framework = new Crm_Integration_Connect_Framework_Keap();
			$request_module = new Crm_Integration_Request_Module_Keap();

			$feed_ids = $connect_framework->getObjectFeed( 'Products' );
			$feed_id  = is_array( $feed_ids ) ? reset( $feed_ids ) : $feed_ids;

			if ( empty( $feed_id ) ) {
				return;
			}
			if ( ! empty( $feed_id ) && is_numeric( $feed_id ) ) {

				$keap = Woo_Crm_Connect_Keap::get_instance();
				$feed_title = $keap->getFeedTitle( $feed_id );
				$record_type = $keap->getFeed( $feed_id, 'crm_object' );
				// Get product.
				$product = wc_get_product( $post_id );
				$bulk_request = array();
	
				if ( 'variable' == $product->get_type() ) {
					$available_variations = $product->get_available_variations();
					if ( ! empty( $available_variations ) && is_array( $available_variations ) ) {
	
						$bulk_ids = array();
						foreach ( $available_variations as $key => $variation ) {
							$request = $connect_framework->getRequest(
								$post->post_type,
								$feed_id,
								$variation['variation_id']
							);
	
							$bulk_ids[] = $variation['variation_id'];
	
						   /**
							* Keap product request filter hook.
							*
							* @since    1.0.0
							*/
							$request = apply_filters(
								'woo_crm_woo_keap_product_request',
								$request,
								$post_id
							);
	
							$request['product_id'] = $variation['variation_id'];
							array_push( $bulk_request, $request );
						}
	
					}
				} else {
					$request = $connect_framework->getRequest(
						$post->post_type,
						$feed_id,
						$post_id
					);
					$bulk_ids[] = $post_id;
	
				   /**
					* Keap product request filter hook.
					*
					* @since    1.0.0
					*/
					$request = apply_filters(
						'woo_crm_woo_keap_product_request',
						$request,
						$post_id
					);
					$request['product_id'] = $post_id;
				}
				$request['active'] = 'true';
				$request['subscription_only'] = 'false';
				$request = ! empty( $bulk_request ) ?
				$bulk_request :
				$request;
				
				$record_type = $connect_framework->getFeed(
					$feed_id,
					'crm_object'
				);
	
				$feed_title = $keap->getFeedTitle( $feed_id );
	
				$log_data = array(
					'woo_id'   => $post_id,
					'bulk_ids'   => $bulk_ids,
					'feed_id'  => $feed_id,
					'woo_object' => $post->post_type,
					'feed_title' => $feed_title,
				);
	
				$filter_exist = $request_module->maybe_check_filter( $feed_id );
				if ( ( isset( $filter_exist[2] ) && ! empty( $filter_exist[2][1]['field'] ) && '-1' != $filter_exist[2][1]['field'] ) || ( isset( $filter_exist[1] ) && ! empty( $filter_exist[1][1]['field'] ) && '-1' != $filter_exist[1][1]['field'] ) || ( isset( $filter_exist[3] ) && ! empty( $filter_exist[3][1]['field'] ) && '-1' != $filter_exist[3][1]['field'] ) || ( isset( $filter_exist[4] ) && ! empty( $filter_exist[4][1]['field'] ) && '-1' != $filter_exist[4][1]['field'] ) ) {
					$filter_result = $request_module->wps_validate_filter( $filter_exist, $request );
					if ( true === $filter_result ) {
						$crm_integration_keap_api = Crm_Integration_Keap_Api::get_instance();
						if ( is_array( $request) && array_key_exists(0, $request) ) {
			
							foreach ($request as $key => $single_request) {
								
								$woo_id = $single_request['product_id'];
			
								$update_prod_id = get_post_meta( $woo_id, '_mwb_keap_product_id', true );
			
								foreach ( $single_request as $k => $v ) {
			
									if ( 'product_id' !== $k ) {
			
										$single_record[ $k ] = $v;
									}
								}
								if ( ! empty( $update_prod_id ) ) {
			
									$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
									$request_data = json_encode( $single_record );
			
									$result = $crm_integration_keap_api->update_single_products(
										$record_type,
										$request_data,
										$endpoint,
										$woo_id,
										! empty( $bulk_request ) ? true : false,
										$log_data
									);
									if (! empty( $result['data']['id'])) {
			
										$keap_id = $result['data']['id'];
									}
									$is_success = self::is_success_res( $result );
			
									self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
			
									
								} else {
									$request_data = json_encode( $single_record );
			
									$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
									$result = $crm_integration_keap_api->create_single_record(
										$record_type,
										$request_data,
										! empty( $bulk_request ) ? true : false,
										$log_data,
										$endpoint,
										$woo_id
									);
			
									if ( 'Products' == $record_type ) {
										if (! empty( $result['data']['id'])) {
			
											$keap_id = $result['data']['id'];
										}
										update_post_meta( $single_request['product_id'], '_mwb_keap_product_id', $keap_id );
			
										$is_success = self::is_success_res( $result );
			
										self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
			
									}
								}
							}
						} else {
			
							$woo_id = $request['product_id'];
			
							$update_prod_id = get_post_meta( $request['product_id'], '_mwb_keap_product_id', true );
			
							foreach ( $request as $k => $v ) {
			
								if ( 'product_id' !== $k ) {
			
									$single_record[ $k ] = $v;
								}
							}
							
							if ( ! empty( $update_prod_id ) ) {
			
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
			
								$request_data = json_encode( $single_record );
			
								$result = $crm_integration_keap_api->update_single_products(
									$record_type,
									$request_data,
									$endpoint,
									$woo_id,
									! empty( $bulk_request ) ? true : false,
									$log_data
								);
								if (! empty( $result['data']['id'])) {
			
									$keap_id = $result['data']['id'];
								}
								$is_success = self::is_success_res( $result );
			
								self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
			
							} else {
			
								$request_data = json_encode( $single_record );
			
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
								$result = $crm_integration_keap_api->create_single_record(
									$record_type,
									$request_data,
									! empty( $bulk_request ) ? true : false,
									$log_data,
									$endpoint,
									$woo_id
								);
			
								if ( 'Products' == $record_type ) {
									if (! empty( $result['data']['id'])) {
			
										$keap_id = $result['data']['id'];
									}
									update_post_meta( $request['product_id'], '_mwb_keap_product_id', $keap_id );
			
									$is_success = self::is_success_res( $result );
			
									self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
			
								}
							}
						}
					}
				} else {
					$crm_integration_keap_api = Crm_Integration_Keap_Api::get_instance();
					if ( is_array( $request) && array_key_exists(0, $request) ) {
	
						foreach ($request as $key => $single_request) {
							
							$woo_id = $single_request['product_id'];
	
							$update_prod_id = get_post_meta( $woo_id, '_mwb_keap_product_id', true );
	
							foreach ( $single_request as $k => $v ) {
	
								if ( 'product_id' !== $k ) {
	
									$single_record[ $k ] = $v;
								}
							}
							if ( ! empty( $update_prod_id ) ) {
	
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
								$request_data = json_encode( $single_record );
	
								$result = $crm_integration_keap_api->update_single_products(
									$record_type,
									$request_data,
									$endpoint,
									$woo_id,
									! empty( $bulk_request ) ? true : false,
									$log_data
								);
								if (! empty( $result['data']['id'])) {
	
									$keap_id = $result['data']['id'];
								}
								$is_success = self::is_success_res( $result );
	
								self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
								
							} else {
								$request_data = json_encode( $single_record );
	
								$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
								$result = $crm_integration_keap_api->create_single_record(
									$record_type,
									$request_data,
									! empty( $bulk_request ) ? true : false,
									$log_data,
									$endpoint,
									$woo_id
								);
	
								if ( 'Products' == $record_type ) {
									if (! empty( $result['data']['id'])) {
	
										$keap_id = $result['data']['id'];
									}
									update_post_meta( $single_request['product_id'], '_mwb_keap_product_id', $keap_id );
	
									$is_success = self::is_success_res( $result );
	
									self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
								}
							}
						}
					} else {
	
						$woo_id = $request['product_id'];
	
						$update_prod_id = get_post_meta( $request['product_id'], '_mwb_keap_product_id', true );
	
						foreach ( $request as $k => $v ) {
	
							if ( 'product_id' !== $k ) {
	
								$single_record[ $k ] = $v;
							}
						}
						
						if ( ! empty( $update_prod_id ) ) {
	
							$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/' . $update_prod_id;
	
							$request_data = json_encode( $single_record );
	
							$result = $crm_integration_keap_api->update_single_products(
								$record_type,
								$request_data,
								$endpoint,
								$woo_id,
								! empty( $bulk_request ) ? true : false,
								$log_data
							);
							if (! empty( $result['data']['id'])) {
	
								$keap_id = $result['data']['id'];
							}
							$is_success = self::is_success_res( $result );
	
							self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
						} else {
	
							$request_data = json_encode( $single_record );
	
							$endpoint = 'https://api.infusionsoft.com/crm/rest/v1/products/';
							$result = $crm_integration_keap_api->create_single_record(
								$record_type,
								$request_data,
								! empty( $bulk_request ) ? true : false,
								$log_data,
								$endpoint,
								$woo_id
							);
	
							if ( 'Products' == $record_type ) {
								if (! empty( $result['data']['id'])) {
	
									$keap_id = $result['data']['id'];
								}
								update_post_meta( $request['product_id'], '_mwb_keap_product_id', $keap_id );
	
								$is_success = self::is_success_res( $result );
	
								self::set_woo_response_meta_for_count( $feed_id, $woo_id, $keap_id, $is_success );
	
							}
						}
					}
				}
			}
		}
	}
	
	// add a function with name add_custom_product_data_tab to add a select custom field in product data tab
	public function add_custom_product_data_tab( $tabs ) {
		$tabs['mwb_keap'] = array(
			'label'    => __( 'Custom tags - keap', 'integration-with-keap' ),
			'target'   => 'mwb_keap_product_data',
			'class'    => array( 'show_if_keap' ),
			'priority' => 21,
		);
		return $tabs;
	}

	// add a multiselect custom field in the newly added product data tab
	public function add_custom_product_data_fields() {
		?>
		<div id="mwb_keap_product_data" class="panel woocommerce_options_panel">
			<div class="options_group">	
				<?php
				
				woocommerce_wp_textarea_input(
					array(
						'id'          => 'mwb_keap_custom_tags',
						'name'        => 'mwb_keap_custom_tags',
						'label'       => __( 'Add custom tags to Product in keap', 'integration-with-keap' ),
						'desc_tip'    => true,
						'description' => __( 'Add custom tags for this product. Separate tags with commas.', 'integration-with-keap' ),
						'class'       => 'mwb-keap-custom-tags-input',
						'style'       => 'width:100%;min-height:38px;',
						'value'       => get_post_meta( get_the_ID(), 'mwb_keap_custom_tags', true ),
					)
				);
				?>
				<script>
				jQuery(document).ready(function($){
					// When pressing Enter, add a comma if not already present
					$('textarea.mwb-keap-custom-tags-input').on('keydown', function(e){
						if(e.key === 'Enter'){
							e.preventDefault();
							var val = $(this).val();
							if(val && val.slice(-1) !== ','){
								$(this).val(val + ', ');
							}
						}
					});
				});
				</script>
				<style>
				.mwb-keap-custom-tags-input {
					min-height: 38px;
					width: 100%;
				}
				</style>
			</div>
		</div>
				<?php
	}


	public function save_custom_product_data_fields( $post_id ) {
        // Check if the post type is product
        if ( 'product' === get_post_type($post_id) || 'product_variation' === get_post_type($post_id) ) {
            // Save custom tags
            if ( isset( $_POST['meta_box_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['meta_box_nonce'] ) ), 'meta_box_nonce' ) ) {
                if ( isset( $_POST['mwb_keap_custom_tags'] ) ) {
                    $custom_tags = sanitize_textarea_field( $_POST['mwb_keap_custom_tags'] );
                    update_post_meta( $post_id, 'mwb_keap_custom_tags', $custom_tags );
                }
            }
        }
    }

	// END.

}

