<?php

namespace ImageHopper\ImageHopper\ThirdParty\WooCommerce;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @since 2.2.2
 */
class GravityForms {

	/**
	 * @var object $_instance If available, contains an instance of this class.
	 *
	 * @since 2.2.2
	 */
	private static $_instance = null;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return self $_instance An instance of this class.
	 *
	 * @since 2.2.2
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * @since 2.2.2
	 */
	public function init() {
		add_filter( 'woocommerce_add_cart_item_data', [ $this, 'hook_up_edit_entry_support' ], 5, 2 );
		add_filter( 'woocommerce_add_cart_item_data', [ $this, 'post_submission_cleanup' ], 15, 2 );
		add_filter( 'woocommerce_gforms_field_display_text', [ $this, 'cart_image_format' ], 10, 3 );
	}

	/**
	 * On WooCommerce GF processing, ensure IH field is reset
	 *
	 * @param array $cart_item_meta
	 *
	 * @return array
	 *
	 * @since 2.2.2
	 */
	public function hook_up_edit_entry_support( $cart_item_meta ) {
		add_filter( 'image_hopper_pre_get_value_save_entry', [ $this, 'reset_field_value' ], 10, 3 );

		return $cart_item_meta;
	}

	/**
	 * Clean up Gravity Forms file upload data after adding the entry to the cart
	 *
	 * @param array $cart_item_meta
	 *
	 * @return array
	 *
	 * @since 2.2.2
	 */
	public function post_submission_cleanup( $cart_item_meta, $product_id ) {
		/* phpcs:ignore WordPress.Security.NonceVerification.Missing */
		if ( ! isset( $_POST['gform_old_submit'] ) ) {
			return $cart_item_meta;
		}

		if ( ! isset( $cart_item_meta['_gravity_form_data'] ) || ! isset( $cart_item_meta['_gravity_form_lead'] ) ) {
			return $cart_item_meta;
		}

		/* phpcs:ignore WordPress.Security.NonceVerification.Missing */
		$context           = ( isset( $_POST['add-variations-to-cart'] ) && $_POST['add-variations-to-cart'] ) ? 'bulk' : 'single';
		$gravity_form_data = wc_gfpa()->get_gravity_form_data( $product_id, $context );

		if ( empty( $gravity_form_data ) || empty( $gravity_form_data['id'] ) || (int) $gravity_form_data['id'] === 0 ) {
			return $cart_item_meta;
		}

		$form_id = (int) $gravity_form_data['id'];
		$form    = \GFAPI::get_form( $form_id );

		$files = json_decode( rgpost( 'gform_uploaded_files' ), true );

		foreach ( $form['fields'] as $field ) {
			if ( $field->type === 'image_hopper' ) {
				$input = "input_{$field->id}";
				if ( isset( \GFFormsModel::$uploaded_files[ $form_id ][ $input ] ) ) {
					unset( \GFFormsModel::$uploaded_files[ $form_id ][ $input ] );
				}

				if ( isset( $files[ $input ] ) ) {
					unset( $files[ $input ] );
				}
			}
		}

		$_POST['gform_uploaded_files'] = wp_json_encode( $files, JSON_UNESCAPED_UNICODE );

		return $cart_item_meta;
	}

	/**
	 * Reset the Image Hopper field value to the original form submission (Woo GF does two submissions when adding to cart)
	 *
	 * @param string $value
	 * @param array  $form
	 * @param string $input_name
	 *
	 * @return string
	 *
	 * @since 2.2.2
	 */
	public function reset_field_value( $value, $form, $input_name ) {
		global $_gf_uploaded_files;

		return isset( $_gf_uploaded_files[ $input_name ] ) ? $_gf_uploaded_files[ $input_name ] : $value;
	}

	/**
	 * Ensure image is displays at thumbnail size
	 *
	 * @param string    $text
	 * @param mixed     $value
	 * @param \GF_Field $field
	 *
	 * @return string
	 *
	 * @since 2.2.2
	 */
	public function cart_image_format( $text, $value, $field ) {
		if ( $field->type !== 'image_hopper' ) {
			return $text;
		}

		return str_replace( 'max-width: 100%', 'max-width: 100px;', $text );
	}
}
