<?php

namespace ImageHopper\ImageHopper\FirstParty;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @since 2.12
 */
class Webhook {

	/**
	 * @var object $_instance If available, contains an instance of this class.
	 *
	 * @since 2.12
	 */
	private static $_instance = null;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return self $_instance An instance of this class.
	 *
	 * @since 2.12
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * @since 2.12
	 */
	public function init() {
		add_filter( 'gform_webhooks_request_data', [ $this, 'enhance_webhook_data' ], 10, 4 );
	}

	/**
	 * Decode the Image Hopper fields and pass along the content as arrays
	 *
	 * @param array $request_data
	 * @param array $feed
	 * @param array $entry
	 * @param array $form
	 *
	 * @return array
	 *
	 * @since 2.12
	 */
	public function enhance_webhook_data( $request_data, $feed, $entry, $form ) {
		$is_all_fields_request = $feed['meta']['requestBodyType'] === 'all_fields';

		$ih_fields     = \GFAPI::get_fields_by_type( $form, [ 'image_hopper', 'image_hopper_post' ] );
		$ih_fields_ids = array_column( $ih_fields, 'id' );

		/* Detect IH fields in the $request_data array, and map the field ID to the key name */
		$webhook_key_map = array_combine( $ih_fields_ids, $ih_fields_ids );
		if ( ! $is_all_fields_request && ! empty( $feed['meta']['fieldValues'] ) ) {
			foreach ( $feed['meta']['fieldValues'] as $map ) {
				if ( $map['key'] === 'gf_custom' && isset( $webhook_key_map[ $map['value'] ] ) ) {
					$webhook_key_map[ $map['value'] ] = $map['custom_key'];
				}
			}
		}

		/* Transform IH fields into more suitable formats for webhook consumption */
		foreach ( $ih_fields as $field ) {
			if ( empty( $request_data[ $webhook_key_map[ $field->id ] ] ) ) {
				continue;
			}

			$field_value = $field->get_value_export( $entry );

			if ( $field->type === 'image_hopper' ) {
				/* To prevent backwards compatibility issues, create new key */
				$key = $webhook_key_map[ $field->id ] . '_decoded';
				if ( $is_all_fields_request && ! isset( $request_data[ $key ] ) ) {
					$request_data[ $key ] = $field_value;
				}

				$key = $webhook_key_map[ $field->id ] . '_array';
				if ( ! isset( $request_data[ $key ] ) ) {
					$request_data[ $key ] = explode( ' , ', $field_value );
				}
			}

			if ( $field->type === 'image_hopper_post' && $request_data[ $webhook_key_map[ $field->id ] ] ) {
				/* To prevent backwards compatibility issues, create new key */
				$key = $webhook_key_map[ $field->id ] . '_array';
				if ( ! isset( $request_data[ $key ] ) ) {
					$field_value = explode( '|:|', $field_value );

					$request_data[ $key ] = [
						'url'         => isset( $field_value[0] ) ? $field_value[0] : '',
						'title'       => isset( $field_value[1] ) ? $field_value[1] : '',
						'caption'     => isset( $field_value[2] ) ? $field_value[2] : '',
						'description' => isset( $field_value[3] ) ? $field_value[3] : '',
						'alt'         => isset( $field_value[4] ) ? $field_value[4] : '',
					];
				}
			}
		}

		return $request_data;
	}
}
