<?php

namespace ImageHopper\ImageHopper\Fields;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * Class SlimPostField
 *
 * Polyfill this field type so users can continue to view Slim Post fields in their Entry Details page. Disable field for data input.
 *
 * @package ImageHopper\ImageHopper\Fields
 *
 * @since 1.0.0
 */
class SlimPostField extends \GF_Field {

	/**
	 * @var string
	 *
	 * @since 1.0.0
	 */
	public $type = 'slim_post';

	/* Disable button in Form Editor */

	/**
	 * Disable button in Form Editor
	 *
	 * @param array $field_groups
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function add_button( $field_groups ) {
		return $field_groups;
	}

	/**
	 * Make sure nothing shows up on the front end for this field any more
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_field_label_class() {
		return 'gf_hidden';
	}

	/**
	 * Ensure the field entry data can still be viewed when using merge tags
	 *
	 * @param array|string $value
	 * @param string       $input_id
	 * @param array        $entry
	 * @param array        $form
	 * @param string       $modifier
	 * @param array|string $raw_value
	 * @param bool         $url_encode
	 * @param bool         $esc_html
	 * @param string       $format
	 * @param bool         $nl2br
	 *
	 * @return array|mixed|string|string[]
	 *
	 * @since 1.0.0
	 */
	public function get_value_merge_tag( $value, $input_id, $entry, $form, $modifier, $raw_value, $url_encode, $esc_html, $format, $nl2br ) {
		list( $url, $title, $caption, $description ) = array_pad( explode( '|:|', $value ), 4, false );
		switch ( $modifier ) {
			case 'title':
				return $title;

			case 'caption':
				return $caption;

			case 'description':
				return $description;

			default:
				return str_replace( ' ', '%20', $url );
		}
	}

	/**
	 * Ensure the field entry data can still be viewed when viewing Entry List
	 *
	 * @param array|string $value
	 * @param array        $entry
	 * @param string       $field_id
	 * @param array        $columns
	 * @param array        $form
	 *
	 * @return array|string
	 *
	 * @since 1.0.0
	 */
	public function get_value_entry_list( $value, $entry, $field_id, $columns, $form ) {
		list( $url, $title, $caption, $description ) = rgexplode( '|:|', $value, 4 );
		if ( ! empty( $url ) ) {
			$thumb = \GFEntryList::get_icon_url( $url );
			/* phpcs:ignore WordPress.WP.I18n.TextDomainMismatch */
			$value = "<a href='" . esc_attr( $url ) . "' target='_blank' title='" . esc_attr__( 'Click to view', 'slim-gravityforms' ) . "'><img src='" . esc_url( $thumb ) . "'/></a>";
		}
		return $value;
	}

	/**
	 * Ensure the field entry data can still be viewed when viewing Entry Details
	 *
	 * @param array|string $value
	 * @param string       $currency
	 * @param false        $use_text
	 * @param string       $format
	 * @param string       $media
	 *
	 * @return array|string
	 *
	 * @since 1.0.0
	 */
	public function get_value_entry_detail( $value, $currency = '', $use_text = false, $format = 'html', $media = 'screen' ) {
		$ary         = explode( '|:|', $value );
		$url         = count( $ary ) > 0 ? $ary[0] : '';
		$title       = count( $ary ) > 1 ? $ary[1] : '';
		$caption     = count( $ary ) > 2 ? $ary[2] : '';
		$description = count( $ary ) > 3 ? $ary[3] : '';

		if ( ! empty( $url ) ) {
			$url = str_replace( ' ', '%20', $url );

			switch ( $format ) {
				case 'text':
					$value  = $url;
					$value .= ! empty( $title ) ? "\n\n" . $this->label . ' (' . __( 'Title', 'image-hopper' ) . '): ' . $title : '';
					$value .= ! empty( $caption ) ? "\n\n" . $this->label . ' (' . __( 'Caption', 'image-hopper' ) . '): ' . $caption : '';
					$value .= ! empty( $description ) ? "\n\n" . $this->label . ' (' . __( 'Description', 'image-hopper' ) . '): ' . $description : '';
					break;

				default:
					$value  = "<a href='$url' target='_blank' title='" . __( 'Click to view', 'image-hopper' ) . "'><img src='$url' width='100' /></a>";
					$value .= ! empty( $title ) ? "<div>Title: $title</div>" : '';
					$value .= ! empty( $caption ) ? "<div>Caption: $caption</div>" : '';
					$value .= ! empty( $description ) ? "<div>Description: $description</div>" : '';

					break;
			}
		}

		return $value;
	}

	/**
	 * Disable field validation
	 *
	 * @param array|string $value
	 * @param array        $form
	 *
	 * @since 1.0.0
	 */
	public function validate( $value, $form ) {
	}
}
