/*
 * FilePond
 * Licensed under MIT, https://opensource.org/licenses/MIT/
 * Please visit https://pqina.nl/filepond/ for details.
 */

export const isArray = value => Array.isArray(value)

export const isEmpty = value => value == null

export const trim = str => str.trim()

export const toString = value => '' + value

export const toArray = (value, splitter = ',') => {
  if (isEmpty(value)) {
    return []
  }
  if (isArray(value)) {
    return value
  }
  return toString(value)
    .split(splitter)
    .map(trim)
    .filter(str => str.length)
}

export const isBoolean = value => typeof value === 'boolean'

export const toBoolean = value => (isBoolean(value) ? value : value === 'true')

export const isString = value => typeof value === 'string'

export const toNumber = value =>
  isNumber(value) ? value : isString(value) ? toString(value).replace(/[a-z]+/gi, '') : 0

export const isNumber = value => typeof value === 'number'

export const toInt = value => parseInt(toNumber(value), 10)

export const toFloat = value => parseFloat(toNumber(value))

export const isInt = value => isNumber(value) && isFinite(value) && Math.floor(value) === value

export const toBytes = (value) => {
  // is in bytes
  if (isInt(value)) {
    return value
  }

  // is natural file size
  let naturalFileSize = toString(value).trim()

  // if is value in megabytes
  if (/MB$/i.test(naturalFileSize)) {
    naturalFileSize = naturalFileSize.replace(/MB$i/, '').trim()
    return toNumber(naturalFileSize) * 1048576
  }

  // if is value in kilobytes
  if (/KB/i.test(naturalFileSize)) {
    naturalFileSize = naturalFileSize.replace(/KB$i/, '').trim()
    return toNumber(naturalFileSize) * 1024
  }

  return toInt(naturalFileSize)
}
