<?php

namespace ImageHopper\ImageHopper\ThirdParty\Avatars;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */
abstract class Avatars {

	/**
	 * The user meta key for the supporting local avatar class
	 *
	 * @return string
	 * @since 2.2
	 *
	 */
	abstract public function get_user_meta_key();

	/**
	 * Transform the Image Hopper input value to the user meta value required for the avatar plugin
	 *
	 * @param string $value
	 *
	 * @return mixed
	 *
	 * @since 2.2
	 */
	abstract public function encode( $value );

	/**
	 * Transform the avatar user meta value to the Image Hopper input value
	 *
	 * @param mixed $value
	 *
	 * @return string
	 *
	 * @since 2.2
	 */
	abstract public function decode( $value );

	/**
	 * @since 2.2
	 */
	public function init() {
		add_filter( 'gform_user_registration_user_data_pre_populate', [ $this, 'populate' ], 5, 3 );
		add_filter( 'gform_user_registration_meta_value', [ $this, 'process' ], 10, 4 );
	}

	/**
	 * Automatically populate the Image Hopper field for the user registration update form
	 *
	 * @param array $mapped_fields
	 * @param array $form
	 * @param array $feed
	 *
	 * @return array
	 *
	 * @since 2.2
	 */
	public function populate( $mapped_fields, $form, $feed ) {
		foreach ( $feed['meta']['userMeta'] as $meta ) {
			if ( ( $meta['key'] === $this->get_user_meta_key() || $meta['custom_key'] === $this->get_user_meta_key() ) && ! empty( $mapped_fields[ $meta['value'] ] ) ) {
				foreach ( $form['fields'] as $field ) {
					if ( (int) $meta['value'] === $field->id ) {
						$mapped_fields[ $field->id ] = $this->decode( $mapped_fields[ $field->id ] );

						break;
					}
				}
			}
		}

		return $mapped_fields;
	}

	/**
	 * Automatically convert Image Hopper fields into the appropriate format
	 *
	 * @param mixed  $value
	 * @param string $meta_key
	 *
	 * @return mixed
	 *
	 * @since 2.2
	 */
	public function process( $value, $meta_key, $meta, $form ) {
		$user_meta_key = $this->get_user_meta_key();

		if ( $meta_key !== $user_meta_key || ! is_array( $meta ) ) {
			return $value;
		}

		/* User Registration v5.3.1 changes the format of $meta, so we need to account for both formats */
		$field_id = '';
		foreach ( $meta as $key => $meta_value ) {
			/* Check post v5.3.1 format */
			if ( is_array( $meta_value ) &&
				 isset( $meta_value['key'], $meta_value['custom_key'], $meta_value['value'], $meta_value['custom_value'] ) &&
			 (
				( $meta_value['key'] ) === $user_meta_key ||
				( $meta_value['custom_key'] ) === $user_meta_key ) ) {
					$field_id = ! empty( $meta_value['custom_value'] ) ? $meta_value['custom_value'] : ( $meta_value['value'] );
					break;
			}

			/* Check pre-v5.3.1 format */
			if ( $key === $user_meta_key ) {
				$field_id = $meta_value;
				break;
			}
		}

		$field = \GFFormsModel::get_field( $form, $field_id );
		if ( $field === null || $field->type !== 'image_hopper' ) {
			return $value;
		}

		/* Parse the string into an array so we can use the first uploaded image */
		$value = array_map( 'trim', explode( ',', $value ) );

		return ! empty( $value[0] ) ? $this->encode( $value[0] ) : '';
	}
}
