import * as FilePond from 'filepond'
import { getFiles, getUniqueFilename } from './helpers/getFiles'
import { addFile, addOrUpdateFile } from './helpers/addFile'
import { deleteFile } from './helpers/deleteFile'
import { prepareTmpFilesForPond } from './helpers/prepareTmpFilesForPond'
import { prepareSavedFilesForPond } from './helpers/prepareSavedFilesForPond'
import { getEntryId } from './helpers/getEntryId'
import { getImageSize } from './helpers/getImageSize'
import { maybeAddMinimumWarningMessage, removeMinimumWarningMessage } from './helpers/minimumImageSizeWarning'
import { reorderFiles } from './helpers/reorderFiles'
import keyboardNavigation from './helpers/keyboardNavigation'
import addTabIndex from './helpers/addTabIndex'
import { setImageSize } from './helpers/setImageSize'
import '../css/style.scss'
import versionCompare from './helpers/versionCompare'
import { maybeAddServersideMessage, removeServersideMessage } from './helpers/serverSideError'
import { maybeHandleNestedFormsOverflow } from './helpers/nestedForms'
import { addCalculationsSupport, processCountMergetag } from './helpers/calculations'
import { getValidJSON } from './helpers/getValidJSON'
import { addCacheBustingToUrl, removeCacheBustingFromUrl } from './helpers/cacheBusting'
import { toBytes } from './helpers/typeConverter'
import { getFormElement } from './helpers/getFormElement'
import i10nFilepond from './helpers/i10nFilepond'
import { getUniqueId, setUniqueId } from './helpers/uniqueId'
import getUrlParam from './helpers/getUrlParam'
import { getFormId } from './helpers/getFormId'

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

(($, gform) => {
  const pondStore = []
  const adminEntryPage = typeof adminpage !== 'undefined' && adminpage.indexOf('_page_gf_entries') > -1
  const l10n = image_hopper_js_strings.l10n
  const imageEndpoint = image_hopper_js_strings.process_endpoint

  // define a single resize observer for the page
  const resizeObserver = new ResizeObserver(entries => {
    for (const entry of entries) {
      window.requestAnimationFrame(() => setImageSize(entry))
    }
  })

  // setup FilePond plugins
  window.ImageHopperFilePondPlugins.forEach(plugin => FilePond.registerPlugin(plugin))

  // load Image Hopper when the form is rendered
  $(document).on('gform_post_render', (event, formId) => preSetup(formId))

  // Add support for Entry Details edit page in admin
  $(() => {
    if (adminEntryPage) {
      preSetup(getUrlParam('id'))
    }
  })

  // Check we can set up Image Hopper
  function preSetup (formId) {
    // Don't run if currently submitting the form
    if (window['gf_submitting_' + formId] || window['gf_submitting_undefined']) {
      return
    }

    // don't run if displaying current form's confirmation page
    if ($('#gform_confirmation_wrapper_' + formId).length > 0) {
      return
    }

    const form = getFormElement(formId)
    // exit if no form found
    if (!form) return

    const $form = $(form)

    // generate unique ID
    if (getUniqueId($form) === '') {
      setUniqueId($form)
    }

    // field counter merge tag support in calculations
    addCalculationsSupport($form)

    setUp($form)
  }

  // Do everything we need to initialize Image Hopper
  // Only initialize when upload field is in the viewport
  function setUp ($form) {
    const uniqueId = getUniqueId($form)
    const formId = getFormId($form)
    const currentPage = $(`input[name=gform_source_page_number_${formId}]`).val()
    const pendingUploads = {}

    gform.doAction('image_hopper_filepond_plugin_registration', FilePond, $form, currentPage)

    // count the number of IH fields on a single page
    const imageHopperFields = $form.find(`.gfield--type-image_hopper:not(.gfield_visibility_hidden), .gfield--type-image_hopper_post:not(.gfield_visibility_hidden)`)
    const visibleImageHopperFields = imageHopperFields.filter(':visible')
    const imageHopperCounter = visibleImageHopperFields.length > 0 ? visibleImageHopperFields.length : imageHopperFields.length

    console.debug( 'Visible Image Hopper fields: ' + imageHopperCounter )

    // event listeners for form submission / change events
    if (versionCompare(image_hopper_js_strings.gfVersion, '2.9.0', '>=')) {
      const preFormSubmission = (data) => {
        let dataFormId = data.form.dataset.formid
        if (!dataFormId) {
          dataFormId = data.form.querySelector('input[name="gform_submit"]').value
        }

        if (+dataFormId !== +formId || data.abort) {
          return data
        }

        if (Object.keys(pendingUploads).length > 0) {
          data.abort = true
        }

        onSubmitActions({ data: { formId: formId } })

        return data
      }

      // Remove the event listener (if it exists) before adding the new one
      // this ensures the listener doesn't execute multiple times when using Gravity Wiz Nested Forms
      // Gravity Forms JS removeFilter() doesn't work like it's PHP counterpart when trying to remove
      // specific functions. It only seems to reliably removes all functions on a specific priority.
      // Define the "IHR" priority (I = 104, H = 103, R = 113) which should be random enough to prevent
      // conflicts with other plugins
      const ihPriority = 104103113
      window.gform.utils.removeFilter('gform/submission/pre_submission', ihPriority)
      window.gform.utils.addFilter('gform/submission/pre_submission', preFormSubmission, ihPriority)
    } else {
      // remove existing filter if it exists
      $form
        .off('submit', onSubmitActions)
        .on('submit', { formId: formId }, onSubmitActions)
    }

    $form.find('.ginput_container_image_hopper > input[type="file"]').each(function () {
      const elem = this
      const formId = $(elem).data('formId')
      const fieldId = $(elem).data('fieldId')

      if (!Array.isArray(pondStore[formId])) {
        pondStore[formId] = []
      }

      // Maybe we can run it earlier?
      // On the initial form load, prefill the #gform_uploaded_files_{form_id} source of truth
      let files = []
      if ((adminEntryPage || parseInt(currentPage) === 1) && getFiles(fieldId, formId).length === 0) {
        files = [...prepareTmpFilesForPond(getFiles(fieldId, formId)), ...prepareSavedFilesForPond($(elem).data('files'))]
        files = files.map(item => {
          addFile(formId, fieldId, {
            temp_filename: '',
            uploaded_filename: removeCacheBustingFromUrl(item.source)
          })

          item.source = addCacheBustingToUrl(item.source)

          return item
        })
      } else {
        // otherwise, load from the source of truth
        files = prepareTmpFilesForPond(getFiles(fieldId, formId))
      }

      console.debug('Field ID ' + fieldId + ': Number of Image Hopper files preloaded: ' + files.length)

      // masquerade as a multifile upload field, and return a function for any call to the object
      window.gfMultiFileUploader.uploaders[`gform_multifile_upload_${formId}_${fieldId}`] = new Proxy({}, {
        get: () => () => ''
      })

      // initialize field only when in the viewport
      const viewportObserver = new IntersectionObserver(
        async (entries) => {
          for (let entry of entries) {
            if (!entry.isIntersecting) {
              continue
            }

            initialize(elem, files)
            viewportObserver.disconnect()

            break
          }
        },
        { threshold: 0.2 }
      )

      viewportObserver.observe(elem)
    })

    function initialize (elem, files) {
      const formId = $(elem).data('formId')
      const fieldId = $(elem).data('fieldId')
      const entryId = getEntryId(elem)
      const nonce = $(elem).data('nonce')

      // Create the FilePond instance
      gform.doAction('image_hopper_pre_filepond_create', FilePond, $form, currentPage, formId, fieldId, entryId, files, elem)

      let acceptedFileTypes = ['image/png', '.png', 'image/jpeg', '.jpg', '.jpeg', 'image/gif', '.gif', 'image/bmp', '.bmp', 'image/heic', '.heic']
      if (versionCompare(image_hopper_js_strings.wpVersion, '5.8', '>=')) {
        acceptedFileTypes.push('image/webp', '.webp')
      }

      if (versionCompare(image_hopper_js_strings.wpVersion, '6.5-RC1', '>=')) {
        acceptedFileTypes.push('image/avif', '.avif')
      }

      // temporary fix for android/chrome bug
      // https://issuetracker.google.com/issues/317289301
      acceptedFileTypes.push('android/force-camera-workaround')

      const originalMaxFiles = $(elem).data('maxFiles') !== undefined ? Number.parseInt($(elem).data('maxFiles')) : 0
      const maxFiles = originalMaxFiles > 0 && files.length > originalMaxFiles ? files.length : originalMaxFiles
      $(elem).removeAttr('data-max-files')

      const className = maxFiles === 1 ? 'single-image-only' : ''
      $(elem).addClass(className)

      const config = gform.applyFilters('image_hopper_filepond_config', {
        allowMultiple: maxFiles !== 1,
        allowReorder: maxFiles !== 1,
        allowPaste: imageHopperCounter === 1,
        maxFiles: ! adminEntryPage && maxFiles > 0 ? maxFiles : null,
        credits: false,
        files: files,
        itemInsertLocation: 'after',
        styleItemPanelAspectRatio: 0.75,
        imageResizeMode: 'contain',
        imageTransformCanvasMemoryLimit: 4096 * 4096,
        imageTransformOutputStripImageHead: false,
        imagePreviewMaxHeight: 600,
        maxParallelUploads: 2,
        acceptedFileTypes: acceptedFileTypes,
        fileValidateTypeLabelExpectedTypesMap: {
          'image/jpeg': '.jpg',
          'image/gif': '.gif',
          'image/png': '.png',
          'image/bmp': '.bmp',
          'image/webp': '.wepb',
          'image/heic': '.heic',
          'image/avif': '.avif'
        },
        fileValidateTypeDetectType: (source, type) => {
          return new Promise((resolve, reject) => {
            /* .heic images on Windows don't get a correct mime type, so we'll force it based on the extension */
            if (type === '') {
              const getExtensionFromFilename = name => name.split('.').pop()
              const extension = getExtensionFromFilename(source.name)
              if (extension.toLowerCase() === 'heic') {
                resolve('image/heic')
              }
            }
            resolve(type)
          })
        },
        instantUpload: true,
        allowMinimumUploadDuration: false,

        /* Ensure every new image added has a unique name so it doesn't create problems with our editor update feature */
        fileRenameFunction: (file) => {
          return getUniqueFilename(fieldId, formId, file)
        },

        server: {

          // upload the file to the server
          process: (fieldName, file, metadata, load, error, progress, abort, transfer, options) => {
            // fieldName is the name of the input field
            // file is the actual file object to send
            const formData = new FormData()
            formData.append('name', file.name)
            formData.append('form_id', formId)
            formData.append('field_id', fieldId)
            formData.append('gform_unique_id', uniqueId)
            formData.append('original_filename', file.name)
            formData.append('file', file)
            formData.append('_gform_file_upload_nonce_' + formId, nonce)

            const pondRootElement = $('#field_' + formId + '_' + fieldId).find('.filepond--root')[0]
            removeServersideMessage(pondRootElement, file.name)

            try {
              // Do file size validation
              // It is done here so the check is completed on the transformed image file
              const maxSize = $(elem).data('max-file-size')
              if (maxSize && file.size > toBytes(maxSize)) {
                maybeAddServersideMessage(pondRootElement, file.name, l10n.labelMaxTotalFileSizeExceeded + '. ' + l10n.labelMaxFileSize.replace('{filesize}', maxSize))
                throw new Error()
              }

              const request = new XMLHttpRequest()
              request.open('POST', imageEndpoint)

              // update the progress indicator
              request.upload.onprogress = (e) => {
                progress(e.lengthComputable, e.loaded, e.total * 1.01)
              }

              // Should call the load method when done and pass the returned server file id
              // this server file id is then used later on when reverting or restoring a file
              // so your server knows which file to return without exposing that info to the client
              request.onload = function () {
                if (request.status >= 200 && request.status < 300) {
                  // the load method accepts either a string (id) or an object
                  try {
                    const jsonString = getValidJSON(request.responseText)
                    var jsonResponse = JSON.parse(jsonString)
                  } catch (e) {
                    console.error(request.responseText)
                    error('error')
                  }

                  if (jsonResponse.status === 'ok') {
                    addOrUpdateFile(formId, fieldId, file, {
                      temp_filename: jsonResponse.data.temp_filename,
                      uploaded_filename: jsonResponse.data.uploaded_filename
                    })

                    load(jsonResponse.data.temp_filename)
                  } else {
                    // display serverside validation errors
                    switch (jsonResponse.error.code || '') {
                      case 104:
                      case 'invalid_file':
                      case 'illegal_extension':
                      case 'illegal_type':
                        maybeAddServersideMessage(pondRootElement, file.name, jsonResponse.error.message)
                    }

                    error('error')
                  }
                } else {
                  // Can call the error method if something is wrong, should exit after
                  error('error')
                }
              }

              request.send(formData)

              // Should expose an abort method so the request can be cancelled
              return {
                abort: () => {
                  // This function is entered if the user has tapped the cancel button
                  request.abort()

                  // Let FilePond know the request has been cancelled
                  abort()
                }
              }
            } catch (exception) {
              error('error')
            }
          },
          // delete a limbo file (tmp upload)
          revert: (uniqueFileId, load, error) => {
            const formData = new FormData()
            formData.append('form_id', formId)
            formData.append('field_id', fieldId)
            formData.append('file', uniqueFileId)
            formData.append('gform_unique_id', uniqueId)
            formData.append('_gform_file_upload_nonce_' + formId, nonce)

            try {
              const request = new XMLHttpRequest()
              request.open('POST', imageEndpoint + '&ih_page=revert')
              request.responseType = 'blob'

              request.onload = function () {
                if (request.status >= 200 && request.status < 300) {
                  deleteFile(formId, fieldId, uniqueFileId)
                  load()
                } else {
                  error('error')
                }
              }

              request.send(formData)
            } catch (exception) {
              error('error')
            }
          },
          // display a limbo (tmp) file
          restore: (uniqueFileId, load, error, progress, abort, headers) => {
            const formData = new FormData()
            formData.append('form_id', formId)
            formData.append('field_id', fieldId)
            formData.append('file', uniqueFileId)
            formData.append('gform_unique_id', uniqueId)
            formData.append('_gform_file_upload_nonce_' + formId, nonce)

            try {
              const request = new XMLHttpRequest()
              request.open('POST', imageEndpoint + '&ih_page=restore')
              request.responseType = 'blob'

              // update the progress indicator
              request.upload.onprogress = (e) => {
                progress(e.lengthComputable, e.loaded, e.total)
              }

              request.onload = function () {
                if (request.status >= 200 && request.status < 300) {
                  const blob = request.response
                  blob.name = decodeURIComponent(request.getResponseHeader('content-disposition').split('filename*=UTF-8\'\'')[1])
                  load(blob)
                } else {
                  error('error')
                }
              }

              request.send(formData)

              // Should expose an abort method so the request can be cancelled
              return {
                abort: () => {
                  // This function is entered if the user has tapped the cancel button
                  request.abort()

                  // Let FilePond know the request has been cancelled
                  abort()
                }
              }
            } catch (exception) {
              error('error')
            }
          },

          // display a local (already uploaded/saved) file
          load: (uniqueFileId, load, error, progress, abort, headers) => {
            try {

              // CORS workaround for Dropbox files
              if (uniqueFileId.includes('www.dropbox.com')) {
                uniqueFileId = uniqueFileId.replace('www.dropbox.com', 'dl.dropboxusercontent.com')
              }

              const request = new XMLHttpRequest()
              request.open('GET', uniqueFileId)
              request.responseType = 'blob'

              request.onload = function () {
                if (request.status >= 200 && request.status < 300) {
                  load(request.response)
                } else {
                  error('error')
                }
              }

              request.send()
            } catch (exception) {
              error('error')
            }
          },

          // dont fetch from external sources
          fetch: null,
        },
        beforeRemoveFile: (item) => {
          if (item.origin !== FilePond.FileOrigin.LOCAL) {
            deleteFile(formId, fieldId, removeCacheBustingFromUrl(item.serverId || item.source.name)) // handle both limbo and input files
            return true
          }

          if (confirm(l10n.labelDeleteFileConfirm)) {
            deleteFile(formId, fieldId, removeCacheBustingFromUrl(item.serverId))
            return true
          }

          return false
        },
        ...i10nFilepond,
      }, FilePond, $form, currentPage, formId, fieldId, entryId, files, elem)

      const pondObject = FilePond.create(elem, config)
      pondStore[formId].push(pondObject)

      // Add IH Count Merge Tag support for use in calculations
      processCountMergetag(formId, fieldId, pondObject)

      // Enable or disable Filepond when the conditional logic changes for the field
      gform.addAction('gform_post_conditional_logic_field_action', (fid, action, targetId) => {
        if (targetId === `#field_${formId}_${fieldId}`) {
          pondObject.setOptions({ disabled: action === 'hide' })
        }
      })

      const pond = pondObject.element
      if (pond === undefined) {
        return
      }

      const pondParentElement = pond.parentElement

      /*
       * Inject the appropriate mark-up to support
       * Gravity Forms Theme Framework
       */
      if ($(pondParentElement).closest('.gform_wrapper').hasClass('gform-theme--framework')) {
        pond.addEventListener('FilePond:init', e => {
          // Dropzone
          $(pondParentElement)
            .find('.filepond--drop-label')
            .addClass('gform-theme-field-control')

          $(pondParentElement)
            .find('.filepond--label-action')
            .addClass('button gform-theme-button gform-theme-button--control')
            .before('<br>')

          // Filepond image preview
          $(pondParentElement)
            .find('.filepond--list')
            .addClass('gform-theme__no-reset--children')
        })
      }

      // Add in size of images based on container
      setImageSize(pond)
      resizeObserver.observe(pond)

      gform.doAction('image_hopper_post_filepond_create', FilePond, $form, currentPage, formId, fieldId, entryId, files, elem, pondObject)

      // Feedback on action (visible on screen)
      $(pond).after('<div class="ih-feedback-messages" aria-live="polite" />')

      // Screen reader only live feedback on action (not visible on screen)
      $(pond).after('<div class="ih-screenreader-feedback-messages screen-reader-text" role="alert" tabindex="-1" />')

      // Prevent propagation of scroll event
      $(pond).on('change input', e => {
        return e.stopPropagation()
      })

      // Listen to keyboard navigation
      if (config.allowReorder) {
        $(pond).on('keydown', e => keyboardNavigation(e, pond, pondObject, fieldId, formId))

        // Screen reader Instructions for reordering images (not visible on screen)
        $(pond).after('<div id="ih-reorder-description" class="screen-reader-text">' + l10n.imageReorderInstructions + '</div>')
      }

      // Add warning if over max file limit
      if (originalMaxFiles !== maxFiles && $(pond).parents('.gfield').find('.validation_message').length === 0) {
        $(pond).parent().find('.ih-feedback-messages').append(`<div class="gfield_description validation_message gfield_validation_message validation_message maximum_files_reached">${l10n.imageValidationMaxFilesReached.replace('%s', $(elem).data('maxFiles'))}</div>`)
      }

      // Started file load
      pond.addEventListener('FilePond:addfilestart', e => {
        if (e.detail.file.origin === FilePond.FileOrigin.INPUT) {
          $(pond).parents('.gfield').find('.validation_message').remove()
          $(pond).parents('.gfield').removeClass('gfield_error')
        }
      })

      // File loaded without errors
      pond.addEventListener('FilePond:addfile', e => {
        if (config.allowReorder) {
          // Ensure all images are tabbable
          addTabIndex(e)
        }

        if (e.detail.file.origin === FilePond.FileOrigin.INPUT && e.detail.error === null) {
          // Fix image order
          const files = e.detail.pond.getFiles()
          const index = files.findIndex(file => file.id === e.detail.file.id)

          addFile(formId, fieldId, {
            temp_filename: '',
            uploaded_filename: e.detail.file.filename,
            status: 'placeholder'
          }, index)

          pendingUploads[e.detail.file.id] = true
        }

        if (e.detail.file.origin !== FilePond.FileOrigin.LOCAL) {

          if (!$(elem).data('min-image-warning') || e.detail.error !== null) {
            return
          }

          const minWidth = $(elem).data('min-image-warning-width')
          const minHeight = $(elem).data('min-image-warning-height')

          // Display minimum width/height warning, if needed
          getImageSize(e.detail.file.file).then(({ width, height }) => {
            maybeAddMinimumWarningMessage(pond, e.detail.file.id, e.detail.file.filename, width, height, minWidth, minHeight)
            maybeHandleNestedFormsOverflow($form)
          })
        }
      })

      pond.addEventListener('FilePond:reorderfiles', e => {
        const { target } = e.detail

        // get Filepond files, but skip those with processing errors
        const files = e.detail.pond.getFiles().filter(file => [
          FilePond.FileStatus.INIT,
          FilePond.FileStatus.IDLE,
          FilePond.FileStatus.PROCESSING_QUEUED,
          FilePond.FileStatus.PROCESSING,
          FilePond.FileStatus.PROCESSING_COMPLETE,
          FilePond.FileStatus.LOADING,
          FilePond.FileStatus.LOAD_ERROR,
        ].includes(file.status))

        reorderFiles(files, fieldId, formId)

        // Reorder in the DOM so the tab order is consistent
        files.forEach(item => $(e.target).find('.filepond--list').prepend($(e.target).find('#filepond--item-' + item.id)))
        const list = e.target.querySelector('ul.filepond--list')
        Array.from(list.children).reverse().forEach(element => list.appendChild(element))

        $form.trigger('change')

        if (files[target] === undefined) {
          return
        }

        // Screen reader feedback
        $(pond)
          .parent()
          .find('.ih-screenreader-feedback-messages')
          .text(
            files[target].filename + '. ' +
            l10n.imageReorderActiveFeedback
              .replace('%1$s', (target + 1))
              .replace('%2$s', files.length)
          )
      })

      pond.addEventListener('FilePond:preparefile', e => {
        if (!$(elem).data('min-image-warning') || $(elem).data('image-resize-upscale') || e.detail.error) {
          return
        }

        const minWidth = $(elem).data('min-image-warning-width')
        const minHeight = $(elem).data('min-image-warning-height')

        // Display minimum width/height warning, if needed
        getImageSize(e.detail.output).then(({ width, height }) => {
          removeMinimumWarningMessage(pond, e.detail.file.id)
          maybeAddMinimumWarningMessage(pond, e.detail.file.id, e.detail.file.filename, width, height, minWidth, minHeight)
          maybeHandleNestedFormsOverflow($form)
        })

        // Usually the image is already tracked by this point, but not if editing an existing image
        pendingUploads[e.detail.file.id] = true
      })

      // Finished processing a file
      pond.addEventListener('FilePond:processfile', e => {
        delete pendingUploads[e.detail.file.id]

        const files = e.detail.pond.getFiles().filter(file => [
          FilePond.FileStatus.INIT,
          FilePond.FileStatus.IDLE,
          FilePond.FileStatus.PROCESSING_QUEUED,
          FilePond.FileStatus.PROCESSING,
          FilePond.FileStatus.PROCESSING_COMPLETE,
          FilePond.FileStatus.LOADING,
          FilePond.FileStatus.LOAD_ERROR,
        ].includes(file.status))

        reorderFiles(files, fieldId, formId)

        if (e.detail.file.origin !== FilePond.FileOrigin.LOCAL) {
          // Trigger change event for Gravity PDF Previewer
          $form.trigger('change')
        }
      })

      // If error rehydrating the file origin, remove pending upload
      pond.addEventListener('FilePond:error', e => {
        delete pendingUploads[e.detail.file.id]

        if (e.detail.file.origin !== FilePond.FileOrigin.LOCAL) {
          removeMinimumWarningMessage(pond, e.detail.file.id)
        }
      })

      pond.addEventListener('FilePond:processfileabort', e => {
        if (e.detail.file.origin !== FilePond.FileOrigin.LOCAL) {
          removeMinimumWarningMessage(pond, e.detail.file.id)
        }
      })

      pond.addEventListener('FilePond:updatefiles', e => {
        maybeHandleNestedFormsOverflow($form)
      })

      pond.addEventListener('FilePond:removefile', e => {
        delete pendingUploads[e.detail.file.id]

        if (e.detail.file.origin === FilePond.FileOrigin.INPUT) {
          // Trigger change event for Gravity PDF Previewer
          $form.trigger('change')

          removeMinimumWarningMessage(pond, e.detail.file.id)
          removeServersideMessage(pond, e.detail.file.file.name)
        }
      })

      // Check for max files warning
      pond.addEventListener('FilePond:warning', e => {
        if (e.detail.error.body === 'Max files') {

          // remove the file from the datastore (if exists)
          deleteFile(formId, fieldId, removeCacheBustingFromUrl(e.detail.file))

          if(!pondParentElement.querySelector('.maximum_files_reached')) {
            $(pond).parent().find('.ih-feedback-messages').append(`<div class="gfield_description validation_message gfield_validation_message validation_message maximum_files_reached">${l10n.imageValidationMaxFilesReached.replace('%s', $(elem).data('max-files'))}</div>`)
            $(pond).parents('.gfield').addClass('gfield_error')
            maybeHandleNestedFormsOverflow($form)
          }
        }
      })
    }

    function onSubmitActions (event) {
      const formId = event.data.formId

      // Prevent form submission while the upload process is still in progress
      if (Object.keys(pendingUploads).length > 0 && (window['gf_submitting_' + formId] || window['gf_submitting_undefined'])) {
        alert(l10n.labelFormSubmitWaitForUpload)

        window['gf_submitting_' + formId] = false
        // fallback when addon (GravityView) does not correctly define the form ID
        if (window['gf_submitting_undefined']) {
          window['gf_submitting_undefined'] = false
        }

        $('#gform_ajax_spinner_' + formId).remove()

        return false
      }

      // Remove any invalid files in the list
      if (pondStore[formId]) {
        pondStore[formId].forEach(pond => {
          const fieldId = pond.id.replace('input_' + formId + '_', '')
          getFiles(fieldId, formId).forEach(image => {
            if (image.status && image.status === 'placeholder') {
              deleteFile(formId, fieldId, image.uploaded_filename)
            }
          })

          if (pond.element.parentElement.classList.contains('ginput_container_image_hopper_post')) {
            // Remove cache-busting parameter from Post Image URL
            const filepondInputs = pond.element.querySelectorAll('.filepond--data input')
            filepondInputs.forEach((input) => {
              input.value = removeCacheBustingFromUrl(input.value)
            })
          } else {
            // Unset the standard Image Hopper field inputs injected by Filepond which is not in the expected
            // format required for Gravity Forms.
            //
            // Note to future self:
            // Image Hopper always looks at $_POST['gform_uploaded_files'] for the upload info (both new and existing),
            // but Gravity Forms passes existing files in $_POST['input_{id}'] and new files in $_POST['gform_uploaded_files']
            // Compatibility with other add-ons may require Image Hopper to adhere to this defacto standard
            const filepondInputs = pond.element.querySelectorAll('.filepond--data input')
            filepondInputs.forEach((input) => {
              input.value = '[]'
            })
          }
        })
      }
    }
  }
})(jQuery, gform)
