const plugin = ({ addFilter, utils }) => {
  // get quick reference to Type utils
  const { getFileFromBlob, getExtensionFromFilename, getFilenameWithoutExtension } = utils

  // called for each file that is loaded
  // right before it is set to the item state
  // should return a promise
  addFilter(
    'LOAD_FILE',
    (file, { query }) =>
      new Promise((resolve, reject) => {
        if (getExtensionFromFilename(file.name).toLowerCase() !== 'heic') {
          resolve(file)
          return
        }

        import(/* webpackChunkName: "heicto" */ './heicto').then(module => {
          const heicto = module.default

          if (!heicto.isHeic(file)) {
            return getFileFromBlob(file, file.name, null, 'jpg')
          }

          const qualityAsPercentage = query('GET_IMAGE_TRANSFORM_OUTPUT_QUALITY')
          const quality = qualityAsPercentage === null ? null : qualityAsPercentage / 100

          heicto.heicTo({
            blob: file,
            toType: 'image/jpeg',
            quality
          }).then(blob => {
            resolve(getFileFromBlob(blob, getFilenameWithoutExtension(file.name), null, 'jpg'))
          })
            .catch(error => {
              /* handle files masquerading as HEIC images */
              if (error.message.startsWith('ERR_USER Image is already browser readable:')) {
                resolve(getFileFromBlob(file, file.name, null, 'jpg'))
                return
              }

              console.error(error.message)

              reject(new Error({
                status: {
                  main: query('GET_LABEL_FILE_LOAD_ERROR')
                }
              }))
            })
        })
      })
  )

  return {
    options: {}
  }
}

// fire pluginloaded event if running in browser, this allows registering the plugin when using async script tags
const isBrowser = typeof window !== 'undefined' && typeof window.document !== 'undefined'
if (isBrowser) {
  document.dispatchEvent(new CustomEvent('FilePond:pluginloaded', { detail: plugin }))
}

export default plugin
