<?php

namespace ImageHopper\ImageHopper\FirstParty;

use ImageHopper\ImageHopper\Fields\ImageHopperPostField;
use ImageHopper\ImageHopper\Helpers\FileHelper;
use ImageHopper\ImageHopper\ImageHopperAddOn;
use ImageHopper\ImageHopper\ThirdParty\WooCommerce\GravityForms;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @since 2.10
 */
class RenameUploads {

	/**
	 * @var object $_instance If available, contains an instance of this class.
	 *
	 * @since 2.10
	 */
	private static $_instance = null;

	/**
	 * @var array The current entry meta fields (if they exist)
	 */
	protected $current_entry_fields;

	/**
	 * @var int The current entry ID being processed
	 */
	protected $current_entry_id;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return self $_instance An instance of this class.
	 *
	 * @since 2.10
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * @since 2.10
	 */
	public function init() {
		/* Don't load if Gravity Forms does not support file upload path metadata */
		if ( ! $this->is_rename_supported() ) {
			return;
		}

		/* Register new field setting */
		add_action( 'gform_field_advanced_settings', [ $this, 'field_rename_image_settings' ], 20 );
		add_filter( 'gform_tooltips', [ $this, 'add_tooltip' ] );

		/* Handle file renaming */
		add_filter( 'gform_entry_post_save', [ $this, 'rename_all_image_hopper_files' ], 1, 2 );
		add_action( 'gform_after_update_entry', [ $this, 'rename_after_update_entry' ], 1, 2 );

		/* Sync the renamed file(s) to the Woo order */
		if ( method_exists( 'WC_GFPA_Compatibility', 'is_wc_version_gte_2_7' ) && \WC_GFPA_Compatibility::is_wc_version_gte_2_7() ) {
			add_action( 'woocommerce_gravityforms_entry_created', [ $this, 'sync_renamed_files_to_woo_order' ], 10, 3 );
		}
	}

	/**
	 * Output the mark-up for the Rename File setting
	 *
	 * @param int $position The position the settings should be located at.
	 *
	 * @since 2.10
	 */
	public function field_rename_image_settings( $position ) {
		if ( $position === 450 ) {
			?>
			<li class="image_hopper_rename_image_field field_setting">

				<label for="input_rename_imagehopper_files" style="display:block" class="section_label">
					<?= esc_html__( 'Rename Upload', 'image-hopper' ); ?>
					<?php gform_tooltip( 'ih_rename_image_hopper_files' ); ?>
				</label>

				<input id="input_rename_imagehopper_files" type="text" onkeyup="SetFieldProperty('inputRenameImageHopperFiles', this.value);" onchange="SetFieldProperty('inputRenameImageHopperFiles', this.value);" />

				<small><?= esc_html__( 'Use {name} to include the original filename.', 'image-hopper' ); ?></small>
			</li>
			<?php
		}
	}

	/**
	 * @param array $tooltips
	 *
	 * @return array
	 *
	 * @since 2.10
	 */
	public function add_tooltip( $tooltips ) {
		$tooltips['ih_rename_image_hopper_files'] = '<h6>' . esc_html__(
			'Rename Upload',
			'image-hopper'
		) . '</h6>' . esc_html__( 'Set the desired path and filename for each upload. Merge Tags are supported. If filename exists, a unique counter is added.', 'image-hopper' );

		return $tooltips;
	}

	/**
	 * Rename any uploaded images when the entry is edited
	 *
	 * @param form $form
	 * @param int $entry_id
	 *
	 * @return void
	 *
	 * @since 2.10
	 */
	public function rename_after_update_entry( $form, $entry_id ) {
		$entry = \GFAPI::get_entry( $entry_id );
		$this->rename_all_image_hopper_files( $entry, $form );
	}

	/**
	 * If needed, rename any uploaded files on the filesystem and in the DB
	 *
	 * @param array $entry
	 * @param array $form
	 *
	 * @return array
	 *
	 * @since 2.10
	 */
	public function rename_all_image_hopper_files( $entry, $form ) {
		$ih = ImageHopperAddOn::get_instance();

		$entry_id = (int) $entry['id'];
		$fields   = \GFAPI::get_fields_by_type( $form, [ 'image_hopper', 'image_hopper_post' ] );

		$ih->log_debug( sprintf( 'Begin renaming uploads for form #%d / entry #%d', $form['id'], $entry_id ) );
		if ( empty( $fields ) ) {
			$ih->log_debug( sprintf( 'No Image Hopper fields detected in form #%d', $form['id'] ) );
		}

		/* Set up database processors */
		\GFFormsModel::begin_batch_field_operations();
		$processor = \GFFormsModel::get_entry_meta_batch_processor();
		$processor::begin_batch_entry_meta_operations();

		foreach ( $fields as $field ) {
			/* Skip fields that haven't been configured for renaming */
			if ( empty( $this->get_field_rename_template( $field ) ) ) {
				continue;
			}

			$ih->log_debug( sprintf( 'Begin renaming file(s) for form #%d / entry #%d / field #%d', $form['id'], $entry_id, $field->id ) );

			/* Skip fields that don't have a value */
			if ( empty( $entry[ $field->id ] ) ) {
				$ih->log_debug( sprintf( 'Skipping field #%d. No entry value found.', $field->id ) );

				continue;
			}

			$files = $this->get_field_files( $entry, $field );
			if ( empty( $files ) ) {
				$ih->log_error( sprintf( 'Skipping field #%d. No valid value found: %s', $field->id, print_r( $entry[ $field->id ], true ) ) );  // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

				continue;
			}

			$original_files = $files;
			foreach ( $files as $key => $source_url ) {
				$meta = $this->get_file_meta( $source_url, $entry_id );

				/* Skip images already renamed, or missing the required meta data */
				if ( ! empty( $meta['ih_renamed'] ) ) {
					$ih->log_debug( sprintf( 'Skip file; already renamed: %s', $source_url ) );

					continue;
				}

				/* Get file info */
				try {
					/* Use the path/url saved in the image metadata, if available */
					if ( isset( $meta['path'], $meta['url'] ) ) {
						$source_path = str_replace( $meta['url'], $meta['path'], $source_url );
					}

					/* If no meta exists, or the source path isn't found, try another method */
					if ( ! isset( $source_path ) || ! is_file( $source_path ) ) {
						$source_path = FileHelper::url_to_path( $source_url );
					}
				} catch ( \Exception $e ) {
					$ih->log_error( sprintf( 'Could not convert the source url to path: %s', $source_url ) );

					continue;
				}

				try {
					$target_path = $this->get_target_path( $source_path, $entry, $form, $field );

					/* Use the path/url saved in the image metadata, if available */
					if ( isset( $meta['path'], $meta['url'] ) ) {
						$target_url = str_replace( $meta['path'], $meta['url'], $target_path );
					}

					/* If no meta exists, or the string replacement didn't work, try another way */
					if ( ! isset( $target_url ) || $target_url === $target_path ) {
						$target_url = FileHelper::path_to_url( $target_path );
					}
				} catch ( \Exception $e ) {
					$ih->log_error( sprintf( 'Could not convert the target path to a url: %s', $target_path ) );

					continue;
				}

				/* Rename the file */
				if ( ! $target_path ) {
					$ih->log_error( sprintf( 'Could not locate source file on disk for %s.', $source_url ) );

					continue;
				}

				if ( ! rename( $source_path, $target_path ) ) {
					$ih->log_error( sprintf( 'Could not rename/move the file from %s to %s', $source_path, $target_path ) );

					continue;
				} else {
					$ih->log_debug( sprintf( 'Successfully renamed/moved file from %s to %s', $source_path, $target_path ) );
				}

				/* Queue the database updates */
				$meta = [
					'path'       => pathinfo( $target_path, PATHINFO_DIRNAME ),
					'url'        => pathinfo( $target_url, PATHINFO_DIRNAME ),
					'file_name'  => pathinfo( $target_path, PATHINFO_BASENAME ),
					'ih_renamed' => true,
				];

				$files[ $key ] = $target_url;

				$ih->log_debug( sprintf( 'New target file meta data: %s', print_r( $meta, true ) ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

				$processor::queue_batch_entry_meta_operation( $form, $entry, $this->get_file_meta_key( $target_url ), $meta );

				/* Delete the original meta (cannot currently batch delete, and I don't currently want to write raw SQL to do so) */
				gform_delete_meta( $entry_id, $this->get_file_meta_key( $source_url ) );

				/* Cleanup variables so they aren't affected on the next loop */
				unset( $source_url, $source_path, $target_url, $target_path );
			}

			/* Only update the field value if a file was renamed */
			if ( $original_files !== $files ) {
				$this->set_field_files( $entry, $form, $field, $files );
			}

			$ih->log_debug( sprintf( 'End renaming file(s) for form #%d / entry #%d / field #%d', $form['id'], $entry_id, $field->id ) );
		}

		/* Execute database batches and rename files */
		\GFFormsModel::commit_batch_field_operations();
		$processor::commit_batch_entry_meta_operations();

		$ih->log_debug( sprintf( 'End renaming uploads for form #%d / entry #%d', $form['id'], $entry_id ) );

		return $entry;
	}

	/**
	 * Return the current field's rename template, if present
	 *
	 * @param \GF_Field $field
	 *
	 * @return string
	 *
	 * @since 2.10
	 */
	protected function get_field_rename_template( \GF_Field $field ) {
		return isset( $field['inputRenameImageHopperFiles'] ) ? $field->inputRenameImageHopperFiles : '';
	}

	protected function get_current_entry_fields( $entry_id ) {
		global $wpdb;

		if ( $this->current_entry_id === $entry_id && is_array( $this->current_entry_fields ) ) {
			return $this->current_entry_fields;
		}

		$entry_meta_table = \GFFormsModel::get_entry_meta_table_name();
		/* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
		$current_fields = $wpdb->get_results( $wpdb->prepare( "SELECT id, meta_key, item_index FROM $entry_meta_table WHERE entry_id=%d", $entry_id ) );

		$this->current_entry_id     = $entry_id;
		$this->current_entry_fields = $current_fields;

		return $this->current_entry_fields;
	}

	/**
	 * Get the MySQL row ID in the entry details for the entry/field
	 *
	 * @param int $field_id
	 * @param int $entry_id
	 *
	 * @return int
	 *
	 * @since 2.10
	 */
	protected function get_entry_meta_id( $field_id, $entry_id ) {
		$current_fields = $this->get_current_entry_fields( $entry_id );

		return \GFFormsModel::get_lead_detail_id( $current_fields, $field_id );
	}

	/**
	 * Return an array of URLs for the current entry/field
	 *
	 * @param array $entry
	 * @param \GF_Field $field
	 *
	 * @return string[]
	 *
	 * @since 2.10
	 */
	protected function get_field_files( $entry, $field ) {
		switch ( $field->type ) {
			case 'image_hopper':
				return json_decode( $entry[ $field->id ], true );
			case 'image_hopper_post':
				$post_image_data = explode( '|:|', $entry[ $field->id ] );

				return ! empty( $post_image_data[0] ) ? [ $post_image_data[0] ] : [];
		}
	}

	/**
	 * Get the meta data associated with the current file (if any).
	 * Special consideration is made for a site that has changed HTTP protocols
	 *
	 * @param string $file_url
	 * @param int $entry_id
	 *
	 * @return mixed
	 *
	 * @since 2.10
	 */
	protected function get_file_meta( $file_url, $entry_id ) {
		$meta = $this->get_meta( $file_url, $entry_id );

		/* Sites can change protocols, so if no meta exists for the image try the other protocol */
		if ( empty( $meta ) ) {
			if ( strpos( $file_url, 'http://' ) === 0 ) {
				$file_url = 'https://' . substr( $file_url, 7 );
			} elseif ( strpos( $file_url, 'https://' ) === 0 ) {
				$file_url = 'http://' . substr( $file_url, 8 );
			}

			$meta = $this->get_meta( $file_url, $entry_id );
		}

		return $meta;
	}

	/**
	 * Get the meta key hash for the current URL (used to get the meta)
	 *
	 * @param string $file_url
	 *
	 * @return string
	 *
	 * @since 2.10
	 */
	protected function get_file_meta_key( $file_url ) {
		return \GF_Field_FileUpload::get_file_upload_path_meta_key_hash( $file_url );
	}

	/**
	 * Return the new path and name of a file, based on the rename template
	 *
	 * @param string $original_filename
	 * @param array $entry
	 * @param array $form
	 * @param \GF_Field $field
	 *
	 * @return array
	 *
	 * @since 2.10
	 */
	protected function get_filename( $original_filename, $entry, $form, $field ) {

		$ih = ImageHopperAddOn::get_instance();

		$template = $this->get_field_rename_template( $field );

		$ih->log_debug( sprintf( 'Get the target filename for "%s" using the template "%s"', $original_filename, $template ) );

		$filename = str_replace( '{name}', $original_filename, $template );
		$filename = \GFCommon::replace_variables( $filename, $form, $entry, false, true, false, 'text' );

		/*
		 * The filename may contain a path.
		 * Split the value into individual names, sanitize, remove any empty values
		 * and put back together again.
		 */
		$file_structure = array_map(
			static function ( $value ) {
				return trim( sanitize_file_name( $value ) );
			},
			explode( '/', $filename )
		);

		$filename = array_pop( $file_structure );
		$path     = implode( '/', $file_structure );

		$ih->log_debug( sprintf( 'The template "%s" has been processed to the filename "%s"', $template, $filename ) );

		return [ $path, $filename ];
	}

	/**
	 * Retrieve the entry metadata for the current file
	 *
	 * @param string $file_url
	 * @param int $entry_id
	 *
	 * @return mixed
	 *
	 * @since 2.10
	 */
	protected function get_meta( $file_url, $entry_id ) {
		global $_gform_lead_meta;

		$meta_key = $this->get_file_meta_key( $file_url );

		/* Flush cache, as it could be stale */
		$cache_key = get_current_blog_id() . '_' . $entry_id . '_' . $meta_key;
		unset( $_gform_lead_meta[ $cache_key ] );

		return gform_get_meta( $entry_id, $meta_key );
	}

	/**
	 * Given a file path, convert to the source path and create the directory structure (if needed)
	 *
	 * @param string    $source_path
	 * @param array     $entry
	 * @param array     $form
	 * @param \GF_Field $field
	 *
	 * @return string|false
	 *
	 * @since 2.10
	 */
	protected function get_target_path( $source_path, $entry, $form, $field ) {
		/* Skip if we cannot find the source file */
		if ( ! is_file( $source_path ) ) {
			return false;
		}

		/* Get source info */
		$file_details                                = pathinfo( $source_path );
		$target_root                                 = $file_details['dirname'];
		list( $target_base_path_suffix, $file_name ) = $this->get_filename( $file_details['filename'], $entry, $form, $field );

		/* For 3rd party compatibility eg. GravityView, ignore the user-defined path if a Post Image field */
		if ( $field instanceof ImageHopperPostField ) {
			$target_base_path_suffix = '';
		}

		/*
		 * If source path is within the standard GF upload directory and a user-defined path is included in the filename,
		 * store files in GF form upload root, instead of in the month/year directories.
		 */
		$form_root = \GFFormsModel::get_upload_path( $form['id'] );
		if ( ! empty( $target_base_path_suffix ) && strpos( $target_root, $form_root ) === 0 ) {
			$target_root = trailingslashit( $form_root );
		}

		$target_base_path = trailingslashit( $target_root . $target_base_path_suffix );
		$extension        = '.' . $file_details['extension'];

		/* Create the directory (if needed) */
		wp_mkdir_p( $target_base_path );
		if ( ! is_file( $target_base_path . 'index.html' ) ) {
			\GFCommon::recursive_add_index_file( $target_base_path );
		}

		$target_base_path = trailingslashit( realpath( $target_base_path ) );

		/* Check for directory traversal outside the target root */
		if ( strpos( $target_base_path, trailingslashit( realpath( $target_root ) ) ) === false ) {
			/* Outside target root; ignore user-defined path */
			$target_base_path = trailingslashit( $file_details['dirname'] );
		}

		/* Get unique target info */
		$counter     = 1;
		$target_path = $target_base_path . $file_name . $extension;
		while ( is_file( $target_path ) ) {
			$target_path = $target_base_path . $file_name . $counter . $extension;
			++$counter;
		}

		/* Remove '.' from the end if file does not have a file extension */
		$target_path = trim( $target_path, '.' );

		return $target_path;
	}

	/**
	 * Check if supported Gravity Forms version in use
	 *
	 * @return bool
	 *
	 * @since 2.10
	 */
	protected function is_rename_supported() {
		return version_compare( \GFForms::$version, '2.5.16', '>=' );
	}

	/**
	 *
	 *
	 * @param array $entry
	 * @param array $form
	 * @param \GF_Field $field
	 * @param array $files
	 *
	 * @return void
	 *
	 * @since 2.10
	 */
	protected function set_field_files( &$entry, $form, $field, $files ) {
		global $_gf_uploaded_files;

		$ih = ImageHopperAddOn::get_instance();

		/* Update the processed file cache */
		$input_name                        = 'input_' . $field->id;
		$_gf_uploaded_files[ $input_name ] = wp_json_encode( $files, JSON_UNESCAPED_UNICODE );

		$new_value = '';
		switch ( $field->type ) {
			case 'image_hopper':
				$new_value = $_gf_uploaded_files[ $input_name ];
				break;

			case 'image_hopper_post':
				$post_image_data    = explode( '|:|', $entry[ $field->id ] );
				$post_image_data[0] = $files[0];
				$new_value          = implode( '|:|', $post_image_data );
				break;
		}

		$ih->log_debug( sprintf( 'Queue field value update for form #%d / entry #%d / field #%d. Value: %s', $form['id'], $entry['id'], $field->id, print_r( $new_value, true ) ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

		\GFFormsModel::queue_batch_field_operation( $form, $entry, $field, $this->get_entry_meta_id( $field->id, $entry['id'] ), $field->id, $new_value );
	}

	/**
	 * If WooCommerce GF Product add-on in use, sync any renamed files back to the order
	 * so the images will display correctly.
	 *
	 * @param int $entry_id
	 * @param int $order_id
	 * @param \WC_Order_Item $order_item
	 *
	 * @return void
	 *
	 * @since 2.10
	 */
	public function sync_renamed_files_to_woo_order( $entry_id, $order_id, $order_item ) {
		if ( ! $this->is_rename_supported() ) {
			return;
		}

		$ih = ImageHopperAddOn::get_instance();

		$ih->log_debug( sprintf( 'Begin syncing renamed files to WooCommerce order for order ID #%d, line item #%d', $order_id, $order_item->get_id() ) );

		$entry = \GFAPI::get_entry( $entry_id );
		if ( is_wp_error( $entry ) ) {
			$ih->log_error( sprintf( 'Could not locate entry ID #%d', $entry_id ) );

			return;
		}

		$form = \GFAPI::get_form( $entry['form_id'] );
		if ( is_wp_error( $form ) ) {
			$ih->log_error( sprintf( 'Could not locate form ID #%d', $entry['form_id'] ) );

			return;
		}

		remove_filter( 'woocommerce_gforms_field_display_text', [ GravityForms::get_instance(), 'cart_image_format' ] );

		try {
			$order_history = $order_item->get_meta( '_gravity_forms_history' );
			if ( ! isset( $order_history['_gravity_form_lead'] ) ) {
				$ih->log_error( sprintf( 'Could not locate order item meta _gravity_forms_history / _gravity_form_lead: %s', print_r( $order_history, true ) ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

				return;
			}

			/* Update the entry meta for relevant fields */
			foreach ( \GFAPI::get_fields_by_type( $form, [ 'image_hopper', 'image_hopper_post' ] ) as $field ) {
				if ( ! $this->get_field_rename_template( $field ) ) {
					continue;
				}

				$display_text = apply_filters( 'woocommerce_gforms_order_meta_title', \GFCommon::get_label( $field ), $field, $entry, $form, $order_item->get_id(), $order_item );
				if ( empty( $display_text ) ) {
					$display_text = $field['id'] . ' -';
				}

				$display_value = $field->get_value_entry_detail( \GFFormsModel::get_lead_field_value( $entry, $field ) );
				$display_value = apply_filters( 'woocommerce_gforms_field_display_text', $display_value, $display_text, $field, $entry, $form );

				$order_item->update_meta_data( $display_text, $display_value );

				if ( isset( $entry[ $field->id ] ) ) {
					$order_history['_gravity_form_lead'][ $field->id ] = $entry[ $field->id ];
				}
			}

			$order_item->update_meta_data( '_gravity_forms_history', $order_history );
			$order_item->save_meta_data();
		} catch ( \Exception $e ) {
			$ih->log_error( sprintf( 'Woocommerce Order Item meta Exception: %s', print_r( $e->getMessage(), true ) ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
		}

		$ih->log_debug( sprintf( 'End syncing renamed files to WooCommerce order for order ID #%d, line item #%d', $order_id, $order_item->get_id() ) );
	}
}
