import fileMimeType from '../helpers/fileMimeType'

const plugin = ({ addFilter, utils }) => {
  // get quick reference to Type utils
  const { getFileFromBlob, getExtensionFromFilename, getFilenameWithoutExtension } = utils

  const fileToBlob = async (file, mimeType) => new Blob([new Uint8Array(await file.arrayBuffer())], { type: mimeType })

  // called for each file that is loaded
  // right before it is set to the item state
  // should return a promise
  addFilter(
    'LOAD_FILE',
    (file, { query }) =>
      new Promise((resolve, reject) => {
        if (getExtensionFromFilename(file.name).toLowerCase() !== 'heic') {
          resolve(file)
          return
        }

        import(/* webpackChunkName: "heicto" */ './heicto').then(module => {
          const heicto = module.default
          const qualityAsPercentage = query('GET_IMAGE_TRANSFORM_OUTPUT_QUALITY')
          const quality = qualityAsPercentage === null ? null : qualityAsPercentage / 100

          heicto.heicTo({
            blob: file,
            type: 'image/jpeg',
            quality
          }).then(blob => {
            resolve(getFileFromBlob(blob, getFilenameWithoutExtension(file.name), null, fileMimeType[blob.type] || 'png'))
          })
            .catch(async error => {
              /* handle files masquerading as HEIC images */
              if (error === 'Error: HEIF image not found') {
                // the mime type is likely image/heic and we don't know what the specific image type is
                // the most common format to convert these images to is JPG, so lets try that
                // if there's a file extension mismatch Gravity Forms will throw an server-side error
                resolve(getFileFromBlob(await fileToBlob(file, 'image/jpeg'), getFilenameWithoutExtension(file.name), null, 'jpg'))
                return
              }

              console.error(error)

              reject(new Error({
                status: {
                  main: query('GET_LABEL_FILE_LOAD_ERROR')
                }
              }))
            })
        })
      })
  )

  return {
    options: {}
  }
}

// fire pluginloaded event if running in browser, this allows registering the plugin when using async script tags
const isBrowser = typeof window !== 'undefined' && typeof window.document !== 'undefined'
if (isBrowser) {
  document.dispatchEvent(new CustomEvent('FilePond:pluginloaded', { detail: plugin }))
}

export default plugin
